/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns;

import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.utils.FStack;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Element;
import org.eclipse.stp.b2j.core.xml.internal.w3c.NamedNodeMap;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Node;

/**
 * 
 * @author amiguel
 *
 * A class used to keep track of namespaces in XML files and resolve both
 * normally namespaced and target namespaced elements and attributes into 
 * fully qualified names (strings)
 */
public class NamespaceTranslator {

private static final boolean USE_PREFIX = true;
	
public static final String NAMESPACE_XSD = "http://www.w3.org/2001/XMLSchema";
public static final String NAMESPACE_XSI = "http://www.w3.org/2001/XMLSchema-instance";
public static final String NAMESPACE_WSDL = "http://schemas.xmlsoap.org/wsdl/";
public static final String NAMESPACE_BPEL = "http://schemas.xmlsoap.org/ws/2004/03/business-process/";
//has this been removed from the spec? is it in the BPEL namespace now?
//public static final String NAMESPACE_PLNK = "http://schemas.xmlsoap.org/ws/2004/03/partner-link/";
public static final String NAMESPACE_WSA = "http://schemas.xmlsoap.org/ws/2004/03/addressing";
public static final String NAMESPACE_ENGINE = "http://www.eclipse.org/stp/b2j/2006/02";

//public final String PACKAGE_XSD = namespaceToPackage(NAMESPACE_XSD);
private String PACKAGE_XSD = "error.xsd.package.not.set";

private static final String DEFAULT_PREFIX = "XSD_DEFAULT_";

private static final String TAGNAME_PREFIX = "XSD_TAG_";
private static final String ATTRIBUTE_PREFIX = "XSD_ATTR_";
private static final String ELEMENT_PREFIX = "XSD_ELEM_";

private static final String CLASS_PREFIX = "_";

private static NamespaceMapping NSMAPPING_DEFAULT = new NamespaceMapping("","xmlns.defaultns");
private static NamespaceMapping NSMAPPING_XSD = new NamespaceMapping(NAMESPACE_XSD,namespaceToPackage(NAMESPACE_XSD));


public String PROGRAM_PREFIX = "";


/**
 * 
 * @author amiguel
 *
 */
class NamespaceStack extends FStack {
}
static class NamespaceMapping {
	public NamespaceMapping() {}
	public NamespaceMapping(String xmlns, String javans) {
		xml_namespace = xmlns;
		java_namespace = javans;
	}
	String xml_namespace;
	String java_namespace;	
}

NamespaceStack targetNamespaces = new NamespaceStack();
NamespaceStack defaultNamespaces = new NamespaceStack();

HashMap mappings = new HashMap();
//HashMap simples = new HashMap();

Util util;


	private void pushNamespace(String key, NamespaceMapping ns) {
		NamespaceStack stack = (NamespaceStack)mappings.get(key);
		if (stack == null) {
			stack = new NamespaceStack();
			mappings.put(key,stack);
		}
		stack.push(ns);
	}
	private NamespaceMapping fetchNamespace(String key) throws NamespaceException {
		NamespaceStack stack = (NamespaceStack)mappings.get(key);
		if (stack == null) {
			throw new NamespaceException("Asked to fetch namespace "+key+" but namespace not found");
		}
		if (stack.size() == 0) {
			throw new NamespaceException("Asked to fetch namespace "+key+" but namespace not found (declared out of scope?)");
		}
		return (NamespaceMapping)stack.peek();
	}
	private void popNamespace(String key) {
		NamespaceStack stack = (NamespaceStack)mappings.get(key);
		stack.pop();
	}

	public NamespaceTranslator(String program_prefix) {
		this.PROGRAM_PREFIX = program_prefix;
		
		targetNamespaces.push(NSMAPPING_DEFAULT);
		defaultNamespaces.push(NSMAPPING_DEFAULT);

		NamespaceStack stack = new NamespaceStack();
		mappings.put("xsd",stack);
		stack.push(NSMAPPING_XSD);
//		pushNamespace("xsd",NSMAPPING_XSD);
		
		try {
			PACKAGE_XSD = getPackage("xsd:faketagname",false);
		} catch (NamespaceException e) {
			throw new Error("Incurred namespace exception while generating XSD package");
		}
	}
	public NamespaceTranslator(String program_prefix, Util compiler_util) {
		this(program_prefix);
		this.util = compiler_util;
	}
	
	public static String getAttribute(String attr_name) {
		return ATTRIBUTE_PREFIX+attr_name;
	}
	
	public static String getDefaultElement(String elem_name) {
		return DEFAULT_PREFIX+elem_name;
	}

	public static String getElement(String elem_name) {
		return ELEMENT_PREFIX+elem_name;
	}
	
	public static String getElementTagName() {
		return TAGNAME_PREFIX+"name";
	}

	public static String getRealName(String name) {
		if (name.startsWith(ELEMENT_PREFIX)) {
			return name.substring(ELEMENT_PREFIX.length());
		} else if (name.startsWith(ATTRIBUTE_PREFIX)) {
			return name.substring(ATTRIBUTE_PREFIX.length());
		} else if (name.startsWith(DEFAULT_PREFIX)) {
			return name.substring(DEFAULT_PREFIX.length());
		}
		return name;
	}
	
	public String getXSDPackage() {
		return PACKAGE_XSD;	
	}

	/**
	 * Add a namespace to the stack (will automatically check if it is a namespace attribute or not)
	 * @param attname the name of the attribute (may be namespace or not)
	 * @param value the value of the attribute
	 * @return whether this attribute was a namespace
	 */
	public boolean addNamespace(String attname, String value) {
		return doNamespace(attname, value, true);	
	}
	/**
	 * Remove a namespace from the stack
	 * @param attname the name of the attribute (may be namespace or not)
	 * @param value the value of the attribute
	 * @return whether this attribute was a namespace and therefore popped from the stack
	 */
	public boolean removeNamespace(String attname, String value) {
		return doNamespace(attname, value, false);	
	}
	public void addNamespaces(Element elem) {
		doNamespaces(elem.getAttributes(), true);	
	}
	public void removeNamespaces(Element elem) {
		doNamespaces(elem.getAttributes(),false);
	}
	
	private boolean doNamespace(String name,String value, boolean add) {
		if (name.equals("xmlns")) {
			if (add) {
				NamespaceMapping ns = new NamespaceMapping(value,namespaceToPackage(value));

//				System.out.println("DEFAULT NAMESPACE:"+name+" -> "+value+" ("+ns.java_namespace+")");
				
				defaultNamespaces.push(ns);
				
			} else {
				defaultNamespaces.pop();
				
			}
			
			return true;
			
		} else if (name.startsWith("xmlns:")) {
			name = name.substring(6);
			
			if (add) {
				NamespaceMapping ns = new NamespaceMapping(value,namespaceToPackage(value));

//				System.out.println("NAMESPACE:"+name+" -> "+value+" ("+ns.java_namespace+")");

				pushNamespace(name,ns);
				
			} else {
				popNamespace(name);
				
			}
			
			return true;
			
		} else if (name.equals("targetNamespace")) {
			if (add) {
				NamespaceMapping ns = new NamespaceMapping(value,namespaceToPackage(value));

//				System.out.println("TARGET NAMESPACE:"+name+" -> "+value+" ("+ns.java_namespace+")");
				
				targetNamespaces.push(ns);
				
			} else {
				targetNamespaces.pop();
				
			}
			
			return true;
		}
		return false;
	}
	
	private void doNamespaces(NamedNodeMap map, boolean add) {
		for (int i = 0; i < map.getLength(); i++) {
			Node attribute = map.item(i);
			if (attribute.getNodeType() == Node.ATTRIBUTE_NODE) {
				String name = attribute.getNodeName();
				String value = attribute.getNodeValue();
				doNamespace(name, value, add);
			}				
		}
	}

	private static String namespaceToPackage(String name) {
		StringBuffer ns = new StringBuffer("xmlns.");
		
		name = name.toLowerCase();
		
		for (int i = 0; i < name.length(); i++) {
			char c = name.charAt(i);
			if (c == '/') {
				ns.append("_");	
			} else if (c == '.') {
				ns.append("_");	
			} else if (c == ':') {
				ns.append("_");	
			} else if (c == '-') {
				ns.append("_");	
			} else if (c == '@') {
				ns.append("_");	
			} else if (c == '#') {
				ns.append("_");	
			} else {
				ns.append(c);
			}
		}
		
		return ns.toString();	
	}

///////////////////////////////////////////////////
// METHODS UNAFFECTED BY TARGET NAMESPACING VS DEFAULT
///////////////////////////////////////////////////

	/**
	 * Get the java class name for an XSD type name
	 */
	public static String getJavaClassName(String s) {
		return CLASS_PREFIX+getName(s);
	}
	
	/**
	 * takes an element name and returns the unqualified name part
	 * @param name
	 * @return the unqualified name of this tag name
	 */
	public static String getName(String name) {
		int index = name.indexOf(":");
		if (index == -1) {
			return name;
		} else {
			return name.substring(index+1);	
		}
	}
	
	/**
	 * takes an element and returns its unqualified name
	 * @param qname
	 * @return the unqualified name of this element
	 */
	public static String getName(Element e) {
		return getName(e.getTagName());	
	}
	
	/**
	 * Check that a qualified name from the current XML document (e.g. 'xsd:blah') is
	 * equal to a particular namespace (e.g. 'http://...xsd.../') and name (e.g. 'blah')
	 * NOTE - this checks namespaces including the default but NOT including the target namespace
	 * @param Element the element to check
	 * @param xmlnamespace the expected namespace
	 * @param ncname the expected name
	 * @return whether the qualified name and the namespace+ncname match
	 */
	public boolean checkQName(Element elem, String xmlnamespace, String ncname) throws NamespaceException {
		String name = getName(elem);
		if (name.equals(ncname)) {
			String namespace = getNamespace(elem,false);
			if (namespace.equals(xmlnamespace)) return true;
		}
		return false;
	}	
	/**
	 * Check that a qualified name from the current XML document (e.g. 'xsd:blah') is
	 * equal to a particular namespace (e.g. 'http://...xsd.../') and name (e.g. 'blah')
	 * NOTE - this checks namespaces including the default but NOT including the target namespace
	 * @param qname the qualified name to check
	 * @param xmlnamespace the expected namespace
	 * @param ncname the expected name
	 * @return whether the qualified name and the namespace+ncname match
	 */
	public boolean checkQName(String qname, String xmlnamespace, String ncname) throws NamespaceException {
		String name = getName(qname);
		if (name.equals(ncname)) {
			String namespace = getNamespace(qname,false);
			if (namespace.equals(xmlnamespace)) return true;
		}
		return false;
	}
	
	
///////////////////////////////////////////////////
// METHODS AFFECTED BY TARGET NAMESPACING VS DEFAULT
///////////////////////////////////////////////////

	/**
	 * takes an element name and gets the XML namespace associated with it
	 * @param name
	 * @return
	 */
	public String getNamespace(String name, boolean target) throws NamespaceException {
		int index = name.indexOf(":");
		if (index == -1) {
			//no namespace specified - using default namespace
			if (target) {
				NamespaceMapping ns = (NamespaceMapping)targetNamespaces.peek();
				return ns.xml_namespace;
			} else {
				NamespaceMapping ns = (NamespaceMapping)defaultNamespaces.peek();
				return ns.xml_namespace;
			}
//			return "";
		} else {
			//namespace specified - converting to java package
			String key = name.substring(0,index);
			NamespaceMapping ns = (NamespaceMapping)fetchNamespace(key);
			
			if (ns == null) throw new NamespaceException("Namespace not found \""+key+"\"");
			
			return ns.xml_namespace;
		} 	
	}

	public String getNamespace(Element e, boolean target) throws NamespaceException {
		return getNamespace(e.getTagName(),target);	
	}

	/**
	 * takes an element name and gets the java package associated with it
	 * @param name
	 * @return
	 */
	public String getPackage(String name, boolean target) throws NamespaceException {
		int index = name.indexOf(":");
		if (index == -1) {
			//no namespace specified - using default namespace
			if (target) {
				NamespaceMapping ns = (NamespaceMapping)targetNamespaces.peek();
				if (USE_PREFIX) {
					return PROGRAM_PREFIX + ns.java_namespace;
				} else {
					return ns.java_namespace;
				}
			} else {
				NamespaceMapping ns = (NamespaceMapping)defaultNamespaces.peek();
				if (USE_PREFIX) {
					return PROGRAM_PREFIX + ns.java_namespace;
				} else {
					return ns.java_namespace;
				}
			}
		} else {
			//namespace specified - converting to java package
			String key = name.substring(0,index);
			NamespaceMapping ns = (NamespaceMapping)fetchNamespace(key);
			if (USE_PREFIX) {
				return PROGRAM_PREFIX + ns.java_namespace;
			} else {
				return ns.java_namespace;
			}
		} 	
	}
	
	public String getPackage(Element e, boolean target) throws NamespaceException {
		return getPackage(e.getTagName(),target);	
	}
	
	public String qualify(String name, boolean target) throws NamespaceException {
		int index = name.indexOf(":");
		
		String pkg = getPackage(name,target);
		String nm = getName(name);
		
/*		
		//this is under the default XSD package
		if (pkg.equals(xsd_package)) {
			
			for (int i = 0; i < xsd_mappings.length; i++) {
				if (nm.equals(xsd_mappings[i][0])) {
					return xsd_mappings[i][1];	
				}
			}
		}
*/

		String qname = pkg+"."+CLASS_PREFIX+nm;
		
		if (util != null) {
			util.addJavaQNameMapping(qname,getNamespace(name,target),nm);
		}
		
		//see if this is a simpleType that has been mapped into an XSD base type
//		String qbase = (String)simples.get(qname);
		String qbase = null;
		if (util != null) {
			qbase = util.getXsdTypeDirectAlias(qname);
		}
		
		if (qbase != null) {
			return qbase;
		} else {
			return qname;
		}
	}
// namespace translators aren't kept from one instance to the next
//	public void putSimple(String qsimple, String qbase) {
//		simples.put(qsimple,qbase);
//	}

///////////////////////////////////////////////////
// XSD to Java type mappings
///////////////////////////////////////////////////
	/*
	public static String getInternalXsdMapping(String ncname) {
		
		String type = Util.getInternalBaseMapping(ncname);
		
		for (int i = 0; i < xsd_mappings.length; i++) {
			if (xsd_mappings[i][0].equals(ncname)) {
				return xsd_mappings[i][1];
			}
		}
		return null;
	}

	public static String[][] xsd_mappings = new String[][]
	{
			{,},
			
	};
	
			state.appendLine("public java.lang.String INTERNAL_VALUE = new String(\"\");\n");
		} else if (mapping.equals(nt.qualify(XSDCodecTranslator.XSD_TYPE_INT,false))) {
			state.appendLine("public Long INTERNAL_VALUE = new Integer(0);\n");
		} else if (mapping.equals(nt.qualify(XSDCodecTranslator.XSD_TYPE_LONG,false))) {
			state.appendLine("public Long INTERNAL_VALUE = new Long(0);\n");
		} else if (mapping.equals(nt.qualify(XSDCodecTranslator.XSD_TYPE_DOUBLE,false))) {
			state.appendLine("public Double INTERNAL_VALUE = new Double(0);\n");
		} else if (mapping.equals(nt.qualify(XSDCodecTranslator.XSD_TYPE_BOOLEAN,false))) {
			state.appendLine("public Boolean INTERNAL_VALUE = new Boolean(false);\n");
		} else if (mapping.equals(nt.qualify(XSDCodecTranslator.XSD_TYPE_DATE,false))) {
			state.appendLine("public java.util.GregorianCalendar INTERNAL_VALUE = new java.util.GregorianCalendar();\n");
		} else if (mapping.equals(nt.qualify(XSDCodecTranslator.XSD_TYPE_LIST,false))) {
			state.appendLine("public java.util.List INTERNAL_VALUE = new java.util.ArrayList();\n");
	
	public static String[][] xsd_mappings = new String[][]
	{
		{"string","java.lang.String"},
		{"normalizedString","java.lang.String"},
		{"token","java.lang.String"},

		{"byte","java.lang.Integer"},
		{"unsignedByte","java.lang.Integer"},

		{"base64Binary","java.lang.String"},
		{"hexBinary","java.lang.String"},

		{"integer","java.lang.Integer"},
		{"positiveInteger","java.lang.Integer"},
		{"negativeInteger","java.lang.Integer"},
		{"nonPositiveInteger","java.lang.Integer"},
		{"nonNegativeInteger","java.lang.Integer"},

		{"int","java.lang.Long"},
		{"unsignedInt","java.lang.Long"},
		{"long","java.lang.Long"},
		{"unsignedLong","java.lang.Long"},

		{"short","java.lang.Integer"},
		{"unsignedShort","java.lang.Integer"},

		{"decimal","java.lang.Double"},
		{"float","java.lang.Double"},
		{"double","java.lang.Double"},

		{"boolean","java.lang.Boolean"},

		{"time","java.util.GregorianCalendar"},
		{"dateTime","java.util.GregorianCalendar"},
		{"duration","java.lang.Long"},
		{"date","java.util.GregorianCalendar"},

		{"gMonth","java.util.GregorianCalendar"},
		{"gYear","java.util.GregorianCalendar"},
		{"gYearMonth","java.util.GregorianCalendar"},
		{"gDay","java.util.GregorianCalendar"},
		{"gMonthDay","java.util.GregorianCalendar"},

		{"Name","java.lang.String"},
		{"QName","java.lang.String"},
		{"NCName","java.lang.String"},
		{"anyURI","java.lang.String"},
		{"language","java.lang.String"},
		{"ID","java.lang.String"},
		{"IDREF","java.lang.String"},
		{"IDREFS","java.lang.String"},
		{"ENTITY","java.lang.String"},
		{"ENTITIES","java.lang.String"},
		{"NOTATION","java.lang.String"},
		{"NMTOKEN","java.lang.String"},
		{"NMTOKENS","java.lang.String"},
	};
*/
}