/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler.xpath;

import org.eclipse.stp.b2j.core.jengine.internal.utils.XPU;

/**
 * 
 * @author amiguel
 * 
 * Reduces precomputable expressions in trees of XPathNodes that represent
 * XPATH expressions.
 * 
 * Any expression that doesn't involve variables is precomputable
 * Its probably not necessary for us to catch every possible precomputable expression in here but
 * A) it is far better to precompute simple expressions than non at all
 * B) if we ever get an XPATH expression interpreter then we can just use it to precomputer everything that
 *    we know to be precomputable
 */
public class XPATHTreePrecomputer {

	public void precompute(XPathNode node) {
		transformNonVariable(node);
	}
		
	private boolean transformNonVariable(XPathNode node) {
		boolean variable = false;
		
		//transform depth first
		for (int i = 0; i < node.getChildCount(); i++) {
			if (transformNonVariable(node.getChild(i))) {
				variable = true;
			}
		}
		
		if (node.getType() == XPathNode.TYPE_VARIABLEREF) {
			variable = true;
		}
		
		//if our children are variable, then we are variable, and we can't be precomputed
		if (!variable) {
			//we should be precomputable
			transformNode(node);
		}
		
		//whether we are an expression that is variable at runtime
		return variable;
	}
	
	private void transformNode(XPathNode node) {
		if (node.type == XPathNode.TYPE_MULTIPLICATIVE
					|| node.type == XPathNode.TYPE_ADDITIVE) {
			if (node.getChildCount() == 2) {

				XPathNode childA = node.getChild(0);
				XPathNode childB = node.getChild(1);
				
				if (childA.getType() == XPathNode.TYPE_NUMBER
					&& childB.getType() == XPathNode.TYPE_NUMBER) {
					double dA = XPU.toNumber(childA.getArg(0)).doubleValue();
					double dB = XPU.toNumber(childB.getArg(0)).doubleValue();
					
					if (node.getArg(0).equals("*")) {
						dA = dA * dB;
					} else if (node.getArg(0).equals("div")) {
						dA = dA / dB;
					} else if (node.getArg(0).equals("mod")) {
						dA = dA % dB;
					} else if (node.getArg(0).equals("-")) {
						dA = dA - dB;
					} else if (node.getArg(0).equals("+")) {
						dA = dA + dB;
					} else {
						System.out.println("UNRECOGNISED OPERATOR");
					}
					childA.setArg(0,XPU.toString(new Double(dA)));
					node.getParent().replaceChild(node,childA);
					
				}
			}
		} else if (node.type == XPathNode.TYPE_UNARY) {
			if (node.getArg(0).equals("-")) {
				if (node.getChildCount() == 1) {
					XPathNode child = node.getChild(0);
					if (child.getType() == XPathNode.TYPE_NUMBER) {
						String number = child.getArg(0);
						double dble = XPU.toNumber(number).doubleValue();
						dble = -dble;
						child.setArg(0,XPU.toString(new Double(dble)));
						node.getParent().replaceChild(node,child);
					}
				}
			}
		} else if (node.type == XPathNode.TYPE_RELATIONAL) {
			if (node.getChildCount() == 2) {

				XPathNode childA = node.getChild(0);
				XPathNode childB = node.getChild(1);
				
				if (childA.getType() == XPathNode.TYPE_NUMBER
					&& childB.getType() == XPathNode.TYPE_NUMBER) {
					
					double dA = XPU.toNumber(childA.getArg(0)).doubleValue();
					double dB = XPU.toNumber(childB.getArg(0)).doubleValue();
					
					boolean ret = false;
					
					if (node.getArg(0).equals(">")) {
						ret = dA > dB;
					} else if (node.getArg(0).equals("<")) {
						ret = dA < dB;
					} else if (node.getArg(0).equals(">=")) {
						ret = dA >= dB;
					} else if (node.getArg(0).equals("<=")) {
						ret = dA <= dB;
					} else {
						System.out.println("UNRECOGNISED OPERATOR");
					}
					
					XPathNode newChild = new XPathNode(XPathNode.TYPE_BOOLEAN);
					newChild.addArg(String.valueOf(ret));
					
					node.getParent().replaceChild(node,newChild);
				}
			}
		} else if (node.type == XPathNode.TYPE_FUNCTION_CALL) {
			
		} else {
		}
	}
}