/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler.xsdmap;


import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.TranslatorLog;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceException;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceTranslator;
import org.eclipse.stp.b2j.core.misc.internal.XMLConfigUtil;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Document;
import org.eclipse.stp.b2j.core.xml.internal.w3c.DocumentBuilder;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Element;

/**
 * 
 * @author amiguel
 *
 * A class to represent a complete XSD file, used in the translation 
 * of BPEL source files into engine Programs
 */
public class XSDMap {

	String LOG_SOURCE = "XSD Translator";
	TranslatorLog log;
	Util compiler_util;
	
	NamespaceTranslator nt;
	
	HashMap topLevel_map = new HashMap();
	ArrayList topLevel_list = new ArrayList();
	HashMap types = new HashMap();
	
	public XSDRootElement getRootElementTypeByNCName(String ncname) {
		return (XSDRootElement)topLevel_map.get(ncname);
	}

	public XSDRootElement getRootElementTypeByQName(String qname) {
		return (XSDRootElement)topLevel_map.get(qname);
	}
	
	public int getRootElementCount() {
		return topLevel_list.size();
	}
	
	public XSDRootElement getRootElement(int index) {
		return (XSDRootElement)topLevel_list.get(index);
	}
	
	public XSDType getType(String qtype) {
		return(XSDType)types.get(qtype);
	}
	
	private void readElement(Element element, XSDComplexType complextype) throws NamespaceException {
		nt.addNamespaces(element);
		
		//element = field inside a class
		String name = element.getAttribute("name");
		String type = element.getAttribute("type");
		
		String minOccurs = element.getAttribute("minOccurs");
		String maxOccurs = element.getAttribute("maxOccurs");
		String def = element.getAttribute("default");
		
		//this element doesn't have a name - it doesn't map to a class field
		if (name.equals("")) return;
		
		if (type.equals("")) {
			//no type, (complex or simple) type is specified for this element
			//this means we have an anonymous type
			
			ArrayList elems = Util.getAllElements(element);
			
			boolean typeOK = false;
			
			for (int i = 0; i < elems.size(); i++) {
				Element complex = (Element)elems.get(i);
				String complexname = NamespaceTranslator.getName(complex);
				if (complexname.equals("complexType")) {
					if (typeOK) {
						log.logWarning(LOG_SOURCE,"two anonymous type definitions for one element");
					}
					typeOK = true;
					
					type = name;
					
					complex.setAttribute("name",type);
					readComplexType(complex);

				} else if (complexname.equals("simpleType")) {
					if (typeOK) {
						log.logWarning(LOG_SOURCE,"two anonymous type definitions for one element");
					}
					typeOK = true;
					
					type = name;
					
					complex.setAttribute("name",type);
					readSimpleType(complex);
					
				} else {
					log.logWarning(LOG_SOURCE,"Unknown element tag name underneath <element> \""+complexname+"\"");
				}
			}

			//turn the type into a fully qualified java reference
			//need to use the target namespace here because thats what we used when we translated it
			type = nt.qualify(type,true);
			
		} else {
			//a type has been specified for this element
			
			//turn the type into a fully qualified java reference
			type = nt.qualify(type,false);
			
		}
		
		XSDComplexTypeElement cselem = new XSDComplexTypeElement();
		cselem.name = name;
		cselem.qtype = type;
		
		complextype.addElement(cselem);
		
		nt.removeNamespaces(element);
	}

	private void readSequence(Element sequence, XSDComplexType complextype) throws NamespaceException {
		nt.addNamespaces(sequence);
		
		//sequence = a bunch of fields in a class
		ArrayList elements = Util.getAllElements(sequence);
		for (int i = 0; i < elements.size(); i++) {
			//translate all the inner bits of this complex type
			readType((Element)elements.get(i),complextype);	
		}
		
		nt.removeNamespaces(sequence);
	}
	
	private void readComplexContent(Element content, XSDComplexType complextype) throws NamespaceException {
		nt.addNamespaces(content);
		
		//sequence = a bunch of fields in a class
		ArrayList elements = Util.getAllElements(content);
		for (int i = 0; i < elements.size(); i++) {
			//translate all the inner bits of this complex type
			readType((Element)elements.get(i),complextype);	
		}
		
		nt.removeNamespaces(content);
	}

	private void readRestriction(Element restriction, XSDComplexType complextype) throws NamespaceException {
		
		nt.addNamespaces(restriction);
		
		XSDType ptmp = (XSDType)types.get(nt.qualify(restriction.getAttribute("base"),false));
		if (ptmp instanceof XSDComplexType) {
			complextype.parent = (XSDComplexType)ptmp;
		}
		
		//sequence = a bunch of fields in a class
		ArrayList elements = Util.getAllElements(restriction);
		for (int i = 0; i < elements.size(); i++) {
			//translate all the inner bits of this complex type
			readType((Element)elements.get(i),complextype);	
		}
		
		nt.removeNamespaces(restriction);
	}
	private void readExtension(Element extension, XSDComplexType complextype) throws NamespaceException {
		
		nt.addNamespaces(extension);
		
		XSDType ptmp = (XSDType)types.get(nt.qualify(extension.getAttribute("base"),false));
		if (ptmp instanceof XSDComplexType) {
			complextype.parent = (XSDComplexType)ptmp;
		}
		
		//sequence = a bunch of fields in a class
		ArrayList elements = Util.getAllElements(extension);
		for (int i = 0; i < elements.size(); i++) {
			//translate all the inner bits of this complex type
			readType((Element)elements.get(i),complextype);	
		}
		
		nt.removeNamespaces(extension);
	}
	
	private void readAttribute(Element attribute, XSDComplexType complextype) throws NamespaceException {
		
		nt.addNamespaces(attribute);
		
		String name = attribute.getAttribute("name");
		String type = attribute.getAttribute("type");
		String use = attribute.getAttribute("use");
		String fixed = attribute.getAttribute("fixed");
		String def = attribute.getAttribute("default");
		
		String qtype = nt.qualify(type,false);

		XSDComplexTypeAttribute attr = new XSDComplexTypeAttribute();
		attr.name = name;
		attr.qtype = qtype;
		attr.fixed = fixed;
		attr.def = def;
		
		complextype.addAttribute(attr);
		
		nt.removeNamespaces(attribute);
	}

	private void readSimpleType(Element simple) throws NamespaceException {
		nt.addNamespaces(simple);

		String name = simple.getAttribute("name");
		String qtype = nt.qualify(name,true);

		String base = compiler_util.getSimpleTypeBase(nt,simple);
		String qbase = nt.qualify(base,false);

		log.logInfo(LOG_SOURCE,"simple type '"+name+"' ("+compiler_util.getQNameFromQName(qtype)+") -> ("+compiler_util.getQNameFromQName(qbase)+")");
//		DBG.info("simple type "+name+" / "+qtype+" ("+qbase+")");
		
		XSDSimpleType type = new XSDSimpleType();
		type.name = name;
		type.qtype = qtype;
		type.qbase = qbase;
		
		types.put(name,type);
		types.put(qtype,type);
		
		nt.removeNamespaces(simple);
	}

	
	private void readComplexType(Element complex) throws NamespaceException {
		nt.addNamespaces(complex);
		
		String name = complex.getAttribute("name");
		String qtype = nt.qualify(name,true);

		log.logInfo(LOG_SOURCE,"complex type '"+name+"' ("+compiler_util.getQNameFromQName(qtype)+")");
//		DBG.info("complex type "+name+" / "+qtype);
		
		XSDComplexType type = new XSDComplexType();
		type.name = name;
		type.qtype = qtype;
				
		ArrayList elements = Util.getAllElements(complex);
		for (int i = 0; i < elements.size(); i++) {
			//recursively translate all the inner bits of this complex type
			readType((Element)elements.get(i),type);	
		}
		
		types.put(name,type);
		types.put(qtype,type);
	
		nt.removeNamespaces(complex);
	}
	
	private void readTopLevelElement(Element elem) throws NamespaceException {
		nt.addNamespaces(elem);
		
		String name = elem.getAttribute("name");
		String type = elem.getAttribute("type");
		
		String qname = nt.qualify(name,true);
		String qtype = nt.qualify(type,false);

		if (type != null && type.length() > 0) {
			compiler_util.setXsdTypeDirectAlias(qname,qtype);
		}
		
		log.logInfo(LOG_SOURCE,"root element '"+name+"' ("+compiler_util.getQNameFromQName(qtype)+")");
//		DBG.info("root element "+qname+" / "+qtype);
		
		String ns = nt.getNamespace(name,true);
		
		XSDRootElement root = new XSDRootElement();
		root.name = name;
		root.namespace = ns;
		root.qname = qname;
		root.qtype = qtype;
		
		topLevel_map.put(name,root);
		topLevel_map.put(qname,root);
		
		topLevel_list.add(root);
		
		nt.removeNamespaces(elem);
	}

	private void readType(Element type, XSDComplexType complextype) throws NamespaceException {
		String tag = NamespaceTranslator.getName(type);

		if (complextype != null) {
			//we are already inside a class (complex Type)
			if (tag.equals("element")) {
				readElement(type,complextype);
			} else if (tag.equals("sequence")) {
				readSequence(type,complextype);
			} else if (tag.equals("attribute")) {
				readAttribute(type,complextype);
			} else if (tag.equals("complexContent")) {
				readComplexContent(type,complextype);
			} else if (tag.equals("restriction")) {
				readRestriction(type,complextype);
			} else if (tag.equals("extension")) {
				readExtension(type,complextype);
			} else if (tag.equals("annotation")) {
				//ignored annotation
			} else if (tag.equals("import")) {
				//ignored annotation
			} else {
				log.logWarning(LOG_SOURCE,"ignored element ["+tag+"] under complex type");
			}
		} else {
			if (tag.equals("element")) {
				readTopLevelElement(type);
			} else if (tag.equals("annotation")) {
				//ignored annotation
			} else {
				log.logWarning(LOG_SOURCE,"ignored element ["+tag+"]");
			}
		}
		//we are not inside a class (complex Type)
		//but we dont need to be as new complex types create
		//new classes anyway
		if (tag.equals("simpleType")) {
			readSimpleType(type);
		} else if (tag.equals("complexType")) {
			readComplexType(type);
		}
	}
	
	private void readSchema(Element schema) throws NamespaceException {

		nt.addNamespaces(schema);

		ArrayList elements = Util.getAllElements(schema);
		for (int i = 0; i < elements.size(); i++) {
			//translate all the complex types in this schema
			readType((Element)elements.get(i),null);	
		}
		
		nt.removeNamespaces(schema);
	}

	public XSDMap(Util compiler_util, TranslatorLog log, String[] xsdxml) throws Exception {
		this(compiler_util,log, xsdxml,new String[0]);
	}

	public XSDMap(Util compiler_util, TranslatorLog log, String[] xsdxml, String[] wsdlxml) throws Exception {
		
		this.log = log;
		this.compiler_util = compiler_util;
//		ArrayList docs = new ArrayList();
		
		nt = compiler_util.createNamespaceTranslator();
		
		//open up all the XSD documents and add them
		for (int i = 0; i < xsdxml.length; i++) {
			DocumentBuilder builder = XMLConfigUtil.getDocumentBuilder();
			Document doc = builder.parse(new ByteArrayInputStream(xsdxml[i].getBytes()));
	
//			docs.add(doc.getDocumentElement());
			readSchema(doc.getDocumentElement());
		}
		
		//add the base types
		NamespaceTranslator tmpNT = compiler_util.createNamespaceTranslator();
		String[] baseTypes = Util.getXSDBaseTypes();
		for (int i = 0; i < baseTypes.length; i++) {
			XSDBaseType type = new XSDBaseType();
			type.qtype = tmpNT.qualify(baseTypes[i],false);
			types.put(type.qtype,type);
		}
		
		//find any XSD type definitions in the WSDL files and add them
		for (int i = 0; i < wsdlxml.length; i++) {
			DocumentBuilder builder = XMLConfigUtil.getDocumentBuilder();
			Document doc = builder.parse(new ByteArrayInputStream(wsdlxml[i].getBytes()));
			
			Element docElem = doc.getDocumentElement();
			nt.addNamespaces(docElem);
			
			ArrayList elems = Util.getAllElements(docElem);
			for (int k = 0; k < elems.size(); k++) {
				Element types = (Element)elems.get(k);
				nt.addNamespaces(types);

				String tag = NamespaceTranslator.getName(types);
				if (tag.equals("types")) {
					ArrayList schemas = Util.getAllElements(types);
					for (int z = 0; z < schemas.size(); z++) {
						Element schema = (Element)schemas.get(z);
						String schematag = NamespaceTranslator.getName(schema);
						if (schematag.equals("schema")) {
							log.logInfo(LOG_SOURCE,"found XSD schema in WSDL document");
//							DBG.info("found XSD schema in WSDL document");
							readSchema(schema);
//							docs.add(schema);
						}
					}
				}
				nt.removeNamespaces(types);
			}
			nt.removeNamespaces(docElem);
		}
//		for (int i = 0; i < docs.size(); i++) {
//			Element schema = (Element)docs.get(i);
//			readSchema(schema);			
//		}
		
	}
	
}