/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.message;


/**
 * @author amiguel
 * 
 * A message - the object which carries the data and is passed around between 
 * MessageReaders and MessageWriters
 */ 
public class Message implements MessageConstants {

int MINLEN = 10;
	
//private ArrayList data;
private Object[] data;
private int type = 0;
private int length = 0;

StringBuffer callstack = null;

public void appendToCallStack(String s) {
	if (callstack == null) callstack = new StringBuffer();
	callstack.append(s);
}
public void appendToCallStack(Throwable t) {
	java.io.ByteArrayOutputStream bout = new java.io.ByteArrayOutputStream();
	t.printStackTrace(new java.io.PrintStream(bout));
	appendToCallStack(new String(bout.toByteArray()));
}

/** 
 * Construct a new message with type=0
 */
public Message() {
	data = new Object[MINLEN];
}//end method

/**
 * Construct a new message with a specific type
 * @param type the type of the message (user defined)
 */
public Message(int type) {
	this.type = type;
	data = new Object[MINLEN];
}//end method

/**
 * Construct a clone of a message
 * @param m the message to clone
 */
public Message(Message m) {

	type = m.type;
	length = m.length;
	
	data = new Object[m.length];
	System.arraycopy(m.data,0,data,0,data.length);
	
	//check for deep clone
	for (int i = 0; i < data.length; i++) {
		if (data[i] instanceof Message) {
			data[i] = new Message((Message)data[i]);
		} else if (data[i] instanceof byte[]) {
			data[i] = ((byte[])data[i]).clone();
		}
	}
	
	if (m.callstack != null) {
		callstack = new StringBuffer(m.callstack.toString());
	}
	
	/*
	this.type = m.getType();
	data = new ArrayList();
	for (int i = 0; i < m.length(); i++) {
		int typ = m.getType(i);
		switch (typ) {
		
		case TYPE_STRING:
			append((String)m.get(i));
			
			break;

		case TYPE_INTEGER:
			append(new Integer( ((Integer)m.get(i)).intValue() ));
			
			break;

		case TYPE_LONG:
			append(new Long( ((Long)m.get(i)).longValue() ));
			
			break;

		case TYPE_DOUBLE:
			append(new Double( ((Double)m.get(i)).doubleValue() ));
		
			break;

		case TYPE_SUBMESSAGE:
			append(new Message((Message)m.get(i)));
			
			break;

		case TYPE_BYTE_ARRAY:
			byte[] b = (byte[])m.get(i);
			byte[] nb = new byte[b.length];
			System.arraycopy(b,0,nb,0,b.length);
			append(nb);
			
			break;
																
		}//end switch
	}//end for
	callstack = m.callstack;
	*/
}//end method

/////////////////////////////////////////////////////////////
// Message Data methods
/////////////////////////////////////////////////////////////

private void add(Object o) {
	if (length == data.length) {
		Object[] ndata = new Object[(int)(data.length * 1.5) + 1];
		System.arraycopy(data,0,ndata,0,length);
		data = ndata;
	}
	data[length] = o;
}

private Object internal_remove(int index) {
	Object o = data[index];
	for (int i = index; i < length-1; i++) {
		data[i] = data[i+1];
	}
	return o;
}

/**
 * Append all the contents of a Message onto the end of this Message
 * @param message the message contents to append
 */
public void appendAll(Message message) {
	for (int i = 0; i < message.length; i++) {
		add(message.data[i]);
		length++;
	}
}

/**
 * Append a sub-Message to the message data
 * @param message the Message to append
 */
public void append(Message message) {
	add(message);
	length++;
}//end method

/**
 * Append a string to the message data
 * @param string the string to append
 */
public void append(String string) {
	add(string);
	length++;
}//end method

/**
 * Append an integer to the message data
 * @param integer the integer to append
 */
public void append(int integer) {
	add(new Integer(integer));
	length++;
}//end method

/**
 * Append an integer to the message data
 * @param integer the integer to append
 */
public void append(Integer integer) {
	add(integer);
	length++;
}//end method

/**
 * Append a long to the message data
 * @param longint the long to append
 */
public void append(long longint) {
	add(new Long(longint));
	length++;
}//end method

/**
 * Append a long to the message data
 * @param longint the long to append
 */
public void append(Long longint) {
	add(longint);
	length++;
}//end method

/**
 * Append a double to the message data
 * @param longfloat the double to append
 */
public void append(double longfloat) {
	add(new Double(longfloat));
	length++;
}//end method

/**
 * Append a double to the message data
 * @param longfloat the double to append
 */
public void append(Double longfloat) {
	add(longfloat);
	length++;
}//end method

/**
 * Append a byte array to the message data
 * @param bytes the byte array to append
 */
public void append(byte[] bytes) {
	add(bytes);
	length++;
}//end method

/**
 * Append a section of a byte array to the message data
 * @param bytes the byte array to append
 * @param off the offset into the byte array to append
 * @param len the number of bytes after the offset to append
 */
public void append(byte[] bytes, int off, int len) {
	byte[] b = new byte[len];
	System.arraycopy(bytes,off,b,0,len);
	add(b);
	length++;
}//end method

/**
 * Get the type of an object in the messages data
 * @param i the index to the object 
 */
public int getType(int i) {
	Object o = get(i);
	if (o == null) {
		//we allow the storing of null strings
		return TYPE_STRING;	
	}
	if (o instanceof String) {
		return TYPE_STRING;
	} else if (o instanceof Integer) {
		return TYPE_INTEGER;
	} else if (o instanceof Long) {
		return TYPE_LONG;
	} else if (o instanceof Double) {
		return TYPE_DOUBLE;
	} else if (o instanceof Message) {
		return TYPE_SUBMESSAGE;
	} else {
		return TYPE_BYTE_ARRAY;	
	}//end if
}//end method

/**
 * Get an object stored in the message
 * @param i the index of the object to get
 * @return the retrieved object
 */
public Object get(int i) {
	return data[i];
}//end method

/**
 * Remove an object stored in this message
 * @param index the index of the object to remove
 * @return the removed object
 */
public Object remove(int index) {
	Object o = internal_remove(index);
	length--;
	return o;	
}//end method

/**
 * Remove the object at the end of this message
 * @return the removed object
 */
public Object pop() {
	Object o = internal_remove(length-1);
	length--;
	return o;	
}



/////////////////////////////////////////////////////////////
// Message Header methods
/////////////////////////////////////////////////////////////

/**
 * Set the type of this message
 * @param i the type to set the message to
 */
public void setType(int i) {
	type = i;
}//end method

/**
 * Get the type of this message
 * @return the type of this message
 */
public int getType() {
	return type;
}//end method


/////////////////////////////////////////////////////////////
// Misc methods
/////////////////////////////////////////////////////////////

/**
 * Get the length of this message
 * @return the number of data objects contained in this message
 */
public int length() {
//	return data.size();
	return length;
}//end method

public boolean equals(Object o) {
	if (this == o) return true;
	if (o instanceof Message) {
		return toString().equals(o.toString());
	} else {
		return super.equals(o);
	}
}

/**
 * Print a string representation of this message
 * @param the string representation of this message
 */
public String toString() {
	StringBuffer sb = new StringBuffer();
	sb.append("Message(");
	sb.append(type);
	sb.append(")");
	
	for (int i = 0; i < length; i++) {
		Object o = data[i];
		sb.append("[");
		
		if (o == null) {
				sb.append("STR (null)");
		} else {
			if (o instanceof String) {
				sb.append("STR,");
				sb.append(o);
			} else if (o instanceof Integer) {
				sb.append("INT,");
				sb.append(o);
			} else if (o instanceof Long) {
				sb.append("LONG,");
				sb.append(o);
			} else if (o instanceof Double) {
				sb.append("DOUBLE,");
				sb.append(o);
			} else if (o instanceof Message) {
				sb.append("SUBMSG,");
				sb.append(o);
			} else {
				sb.append("DATA,");
				sb.append( ((byte[])o).length );
				sb.append(" bytes");
			}//end if
		}
			
		sb.append("]");
		
	}//end for

	if (callstack != null) {
		sb.append("CALLSTACK,"+callstack.toString());
	}
	
	return sb.toString();
}//end method

/**
 * Clone this message
 * @return a copy of this message
 */
public Object clone() {
	return new Message(this);
}//end method

}//end class
