/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.publicapi;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import org.eclipse.stp.b2j.core.jengine.internal.message.Message;
import org.eclipse.stp.b2j.core.jengine.internal.message.MessageReader;
import org.eclipse.stp.b2j.core.jengine.internal.message.MessageWriter;
import org.eclipse.stp.b2j.core.misc.internal.HexData;

/**
 * 
 * @author amiguel
 *
 * Represents a single JAR dependency which the engine must include in the program
 * or in remote engines' classloaders
 */
public class JARDependency {
	private byte[] jar_dat;
	private String path;
	private String name;
	private boolean enabled = true;
	
	//transient - does not need to be saved or loaded
	private String[] namespaces = null;
	private String plugin_name = null;
	
	private JARDependency() {
	}
	/**
	 * Create a new JAR dependency
	 * @param dat the JAR data as a byte array
	 * @param path the java File path to the source JAR file
	 */
	public JARDependency(byte[] dat, String path) {
		this.jar_dat = dat;
		this.path = path;
		this.name = path.substring( Math.max(path.lastIndexOf('/'),path.lastIndexOf('\\')) + 1 );
	}
	
	/**
	 * Set the file path of this JAR dependency
	 * @param path the java File path to the source JAR file
	 */
	public void setFilePath(String path) {
		this.path = path;
		this.name = path.substring( Math.max(path.lastIndexOf('/'),path.lastIndexOf('\\')) + 1 );
	}
	
	/**
	 * Get the source java File path for this JAR dependency
	 * @return the java File path to the source JAR file
	 */
	public String getFilePath() {
		return path;
	}
	
	/**
	 * Get the name part of the source java File path for this JAR dependency
	 * @return the name of the source JAR file
	 */
	public String getFileName() {
		return name;
	}
	
	/**
	 * Get the JAR data 
	 * @return the source JAR file as a byte array
	 */
	public byte[] getJarData() {
		return jar_dat;
	}
	
	/**
	 * Set the plugin providing this JAR dependency
	 * @param name the name of the plugin providing this JAR dependency
	 */
	public void setPluginName(String name) {
		plugin_name = name;
	}
	
	/**
	 * Get the name of the plugin providing this JAR dependency
	 * @return the name of the plugin providing this JAR dependency
	 */
	public String getPluginName() {
		return plugin_name;
	}
	
	/**
	 * Set any namespaces associated with this JAR dependency
	 * @param namespaces any namespaces for bindings etc which require this JAR
	 */
	public void setAssociatedNamespaces(String[] namespaces) {
		this.namespaces = namespaces;
	}
	
	/**
	 * Get any namespaces associated with this JAR dependency (used to filter out
	 * unused JAR dependencies based on the used namespaces in the BPEL)
	 * @return any associated namespaces which require this JAR
	 */
	public String[] getAssociatedNamespaces() {
		return namespaces;
	}
	
	/**
	 * Set whether this JAR dependency is enabled (not ignored)
	 * @param enabled whether this JAR dependency is enabled (should be included in the dependencies)
	 */
	public void setEnabled(boolean enabled) {
		this.enabled = enabled;
	}
	
	/**
	 * Get whether this JAR dependency is enabled (not ignored)
	 * @return whether this JAR dependency is enabled (should be included in the dependencies)
	 */
	public boolean getEnabled() {
		return enabled;
	}
	
	//
	// Static utility methods
	//
	
	
	/**
	 * Serialise this JAR dependency as a hexadecimal string
	 * @param j the JARDependency to serialise as a string
	 * @return a String hexadecimal representation of this JAR dependency
	 * @throws IOException if an error occurs during the serialisation
	 */
	public static String toHex(JARDependency j) throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		MessageWriter mwrite = new MessageWriter(bout);
		mwrite.write(toMessage(j));
		return HexData.byteArrayToHexString(bout.toByteArray());
	}
	
	/**
	 * Deserialise this JAR dependency from a hexadecimal string
	 * @param s the hexadecimal string to deserialise into a JAR dependency
	 * @return the JARDependency represented by this hexadecimal string
	 * @throws IOException if an error occurs during the deserialisation
	 */
	public static JARDependency fromHex(String s) throws IOException {
		ByteArrayInputStream bin = new ByteArrayInputStream(HexData.hexStringToByteArray(s));
		MessageReader mread = new MessageReader(bin);
		Message m = mread.read();
		return fromMessage(m);
	}
	
	public static Message toMessage(JARDependency j) {
		Message m = new Message();
		m.append(j.jar_dat);
		m.append(j.path);
		m.append(j.name);
		if (j.enabled) {
			m.append(1);
		} else {
			m.append(0);
		}
		return m;
	}
	
	public static JARDependency fromMessage(Message m) {
		JARDependency j = new JARDependency();
		j.jar_dat = (byte[])m.get(0);
		j.path = (String)m.get(1);
		j.name = (String)m.get(2);
		if (m.length() > 3) {
			j.enabled = ((Integer)m.get(3)).intValue() == 1;
		}
		return j;
	}
	
	public static JARDependency writeTemporaryDependency(JARDependency dep) throws IOException {
		File tmpjar = File.createTempFile("B2jEngineDependency_",".jar");
		FileOutputStream fout = new FileOutputStream(tmpjar);
		
		fout.write(dep.jar_dat);
		fout.flush();
		fout.close();
		
		tmpjar.deleteOnExit();
		
		dep.path = tmpjar.getCanonicalPath();
		
		return dep;
	}
}