/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.publicapi.engine;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Properties;

import org.eclipse.stp.b2j.core.jengine.internal.api.BpelProgramFactory;
import org.eclipse.stp.b2j.core.jengine.internal.api.EngineFactory;
import org.eclipse.stp.b2j.core.jengine.internal.api.Program;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Switches;
import org.eclipse.stp.b2j.core.jengine.internal.core.api.ControllerInterface;
import org.eclipse.stp.b2j.core.jengine.internal.core.api.DaemonInterface;
import org.eclipse.stp.b2j.core.jengine.internal.core.api.TraceListener;
import org.eclipse.stp.b2j.core.jengine.internal.mainengine.api.SoapDaemonConnector;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;
import org.eclipse.stp.b2j.core.jengine.internal.utils.EngineListenerTraceListener;
import org.eclipse.stp.b2j.core.jengine.internal.utils.StreamUtils;
import org.eclipse.stp.b2j.core.jengine.internal.utils.TraceListenerTranslatorLog;
import org.eclipse.stp.b2j.core.publicapi.B2jPlatform;
import org.eclipse.stp.b2j.core.publicapi.DependencyInfo;
import org.eclipse.stp.b2j.core.publicapi.JARDependency;
import org.eclipse.stp.b2j.core.publicapi.importresolver.StandardWsdlImportResolver;
import org.eclipse.stp.b2j.core.publicapi.jcompiler.IndependantExternalJavaCompiler;
import org.eclipse.stp.b2j.core.publicapi.program.BPELProgram;
import org.eclipse.stp.b2j.core.publicapi.transport.session.SessionAddress;

/**
 * A public API which represents an engine capable of running a BPELProgram program
 * 
 * @author aem
 *
 */
public class IndependantBPELEngine {
	
	BPELProgram program;
	
	boolean compiled = false;
	
	Object running_LOCK = new Object();
	boolean running = false;
	
	ControllerInterface controller = null;
	Message root_runner_ids = null;

	/**
	 * Create a new BPEL engine to run a BPEL program
	 * @param program the BPEL program to run
	 */
	public IndependantBPELEngine(BPELProgram program) {
		this.program = program;
	}

	/**
	 * Run a BPEL program (will use the JVM local engine)
	 * @throws Exception if an error occurs while running the BPEL program
	 */
	public void runProgram() throws Exception {
		runProgram(null,false,false,true,true);
	}

	/**
	 * Run a BPEL program
	 * @param useMiniEngine whether to use a local BPEL engine running in the current JVM
	 * @param terminateEngineOnDisconnect whether the engine should terminate if the client jvm disconnects
	 * @throws Exception if an error occurs while running the BPEL program
	 */
	public void runProgram(boolean useMiniEngine, boolean terminateEngineOnDisconnect) throws Exception {
		runProgram(null,false,false,useMiniEngine,terminateEngineOnDisconnect);
	}
	
	/**
	 * Run a BPEL program
	 * @param debug whether to run the engine in debug mode (slower)
	 * @param verbose whether to produce verbose output when processing the input BPEL file etc.
	 * @param useMiniEngine whether to use a local BPEL engine running in the current JVM
	 * @param terminateEngineOnDisconnect whether the engine should terminate if the client jvm disconnects
	 * @throws Exception if an error occurs while running the BPEL program
	 */
	public void runProgram(boolean debug, boolean verbose, boolean useMiniEngine, boolean terminateEngineOnDisconnect) throws Exception {
		runProgram(null,debug,verbose,useMiniEngine,terminateEngineOnDisconnect);
	}

	/**
	 * Run a BPEL program
	 * @param listener the trace listener to attach to the engine
	 * @param debug whether to run the engine in debug mode (slower)
	 * @param verbose whether to produce verbose output when processing the input BPEL file etc.
	 * @param useMiniEngine whether to use a local BPEL engine running in the current JVM
	 * @param terminateEngineOnDisconnect whether the engine should terminate if the client jvm disconnects
	 * @throws Exception if an error occurs while running the BPEL program
	 */
	public void runProgram(BPELEngineListener listener, boolean debug, boolean verbose, boolean useMiniEngine, boolean terminateEngineOnDisconnect) throws Exception {
		synchronized(running_LOCK) {
			runInternal(program,listener,debug,verbose,useMiniEngine,terminateEngineOnDisconnect,false);
		}
	}
	
	/**
	 * Forcibly terminate this BPEL program
	 * @throws Exception if an error occurs while waiting for this BPEL program to terminate
	 */
	public void terminate() throws Exception {
		controller.terminate();
	}
	
	/**
	 * Wait for the BPEL program to terminate
	 * @throws Exception if an error occurs while waiting for this BPEL program to terminate
	 */
	public void waitForProgramCompletion() throws Exception {
		
		synchronized(running_LOCK) {
			if (!running) {
				return;
			}
		
			for (int i = 0; i < root_runner_ids.length(); i++) {
				Long id = (Long)root_runner_ids.get(i);	
	//			try {
					controller.joinRunner(id);
	//			} catch (Exception e) {
	//				console.debug("Failed to wait on program finish");
	//				console.debug(getStacktrace(e));
	//				B2jPlugin.DBG.logVisibleError(e,"Failed to wait on program finish",false);
	//				return;
	//			}
			}
	
			//give the data reading and printing thread a chance to finish reading any stuff
			try {
				Thread.sleep(1000);
			} catch (InterruptedException e) {}
	
			//close the connection to the engine
			controller.closeConnection();
			
			running = false;
		}
	}

	/**
	 * Compile and validate this BPEL program
	 * @return
	 * @throws Exception
	 */
	public BPELProgram compileProgram(BPELEngineListener console) throws Exception {
		compileProgram(null,program,false,false,false);
		return program;
	}
	
	public void markProgramCompiled() throws Exception {
		compiled = true;
	}
	
	/**
	 * Validate this BPEL program, throws an exception if the validation fails for any reason
	 * @throws Exception
	 */
	public void validateProgram() throws Exception {
		compileProgram(null,program,false,false,true);
	}

	private void compileProgram(BPELEngineListener console, BPELProgram bpel_program, boolean debug, boolean verbose, boolean validateOnly) throws Exception {
		if (console == null) {
			console = new IgnoredTraceListener();
		}

		TraceListener console_tracel = new EngineListenerTraceListener(console);

		ArrayList program_deps = fetchAllProgramDependencies(console);
		
		//
		//Create the engine program
		//NOTE: we do this BEFORE we remove Program dependancies that are also Engine dependancies,
		//otherwise we miss out on 
		//
		Program prog = null;
		try {
			JARDependency[] tmp_program_deps = dependencyListToArray(program_deps);
			console.printInfo("Generating engine program");

			Switches switches = new Switches();
			
			if (debug) {
				//make everything global so we can access it
				switches.FLOW_GRAPH_ASSET_OPTIMISATION = false;
				switches.SINGLE_HOST_ASSET_OPTIMISATION = false;
				//leave all the proper names etc in
				switches.MAP_XSDTYPE_TO_INDEXMAP = false;
				switches.RICH_ENGINE_SERIALISATION = true;
			}
			
			//
			// Set debugging compilation switches
			//
/*			if (mode.equalsIgnoreCase("debug")) {
				Switches.APPEND_TRANSACTIONS_TO_CALLSTACK = true;
			} else {
				Switches.APPEND_TRANSACTIONS_TO_CALLSTACK = false;
			}
*/			
			console.printInfo("");
			TraceListenerTranslatorLog mclog = new TraceListenerTranslatorLog(console_tracel,verbose);
			
//			prog = BpelProgramFactory.createEngineProgramFromBpelSource(debug,switches,new IndependantJaninoJavaCompiler(),new StandardWsdImportResolver(),bpel_program.getBaseURI(),bpel_program.getBpelSource(),tmp_program_deps,mclog,validateOnly);
			prog = BpelProgramFactory.createEngineProgramFromBpelSource(debug,switches,new IndependantExternalJavaCompiler(),new StandardWsdlImportResolver(),bpel_program.getBaseURI(),bpel_program.getBpelSource(),tmp_program_deps,mclog,validateOnly);

			bpel_program.setCompiledData("Program",prog);
			bpel_program.setCompiledData("Program Dependencies",program_deps);
			
			compiled = true;

		} catch (Exception e) {
			printFatalProblem(console,"Failed to build engine program from BPEL source",e);
			
			throw e;
			
//			printFatalProblem(console,"Failed to build engine program from BPEL source",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to build engine program from BPEL source",false);
		}
	}
	
	private Program runInternal(BPELProgram bpel_program, BPELEngineListener console, boolean debug, boolean verbose, boolean useMiniEngine, boolean exitOnDisconnect, boolean validateOnly) throws Exception {

		if (console == null) {
			console = new IgnoredTraceListener();
		}
		
		TraceListener console_tracel = new EngineListenerTraceListener(console);
		
//		System.out.println(bpel_program.source);
		
		if (!compiled) {
			compileProgram(console,bpel_program,debug,verbose,false);
		}
		//compile the program for execution
//		compileProgram(console,console_tracel,program,debug,verbose,false);
		
		Program prog = (Program)bpel_program.getCompiledData("Program");
		ArrayList program_deps = (ArrayList)bpel_program.getCompiledData("Program Dependencies");

		SessionAddress[] workers = bpel_program.getWorkerHosts();
		SessionAddress[] worker_daemons = bpel_program.getWorkerDaemons();
		
		for (int i = 0; i < workers.length; i++) {
			prog.addHostAddress(workers[i],worker_daemons[i]);
		}

		ArrayList engine_deps = fetchAllEngineDependencies(console);
		
		//
		//remove any multiples from Program deps that are included in Engine deps
		//
		for (int i = 0; i < program_deps.size(); i++) {
			JARDependency dep = (JARDependency)program_deps.get(i);
			for (int k = 0; k < engine_deps.size(); k++) {
				JARDependency tmp = (JARDependency)engine_deps.get(k);
				if (dep.getFilePath().equals(tmp.getFilePath())) {
					program_deps.remove(i--);
					break;
				}
			}
		}
		
		JARDependency[] engine_deps_array = dependencyListToArray(engine_deps);
		JARDependency[] program_deps_array = dependencyListToArray(program_deps);
		
		if (engine_deps_array.length > 0 || program_deps_array.length > 0) {
			console.printInfo("");
			console.printInfo("Dependencies:");
		}

		for (int i = 0; i < engine_deps_array.length; i++) {
			console.printInfo("  (Engine Dependency)  "+engine_deps_array[i].getFilePath());
		}
		for (int i = 0; i < program_deps_array.length; i++) {
			console.printInfo("  (Program Dependency)  "+program_deps_array[i].getFilePath());
		}

		if (engine_deps_array.length > 0 || program_deps_array.length > 0) {
			console.printInfo("");
		}
		
		String coordinator_hostname = null;
		
		DaemonInterface daemon = null;
		try {
			if (!useMiniEngine) {
				File plugin_file = new File("./");
				File jar_file = new File("./b2j.jar");
				
//				URL plugin_dir = Platform.asLocalURL(Platform.find(B2jPlugin.getDefault().getBundle(),new Path("/")));
//				File plugin_file = new File(plugin_dir.getFile());
//				File jar_file = new File(plugin_dir.getFile()+"/b2j.jar");
				
				//start an engine daemon 
				console.printInfo("Starting main engine daemon in "+plugin_file);
				EngineFactory.createMainEngineDaemon(plugin_file,jar_file,11000);
	
				SessionAddress daemon_address = bpel_program.getCoordinatorDaemon();
//				SessionAddress daemon_address = (SessionAddress)daemon_list.get(0);
				coordinator_hostname = daemon_address.getListenerHost();

				String protocol;
				if (daemon_address.getRequiresEncryption()) {
					protocol = "(HTTPS)";
				} else {
					protocol = "(HTTP)";
				}
				
				String password;
				if (daemon_address.getRequiresPassword()) {
					password = "(requires password)";
				} else {
					password = "(no password)";
				}
				
				//connect to the engine daemon 
				console.printInfo("Connecting to main engine daemon on "+coordinator_hostname+" "+protocol+" "+password);
				daemon = new SoapDaemonConnector(daemon_address);
//				daemon = EngineFactory.connectToMainEngineDaemon(daemon_address);
				
			} else {
				//connect to the engine daemon 
				console.printInfo("Connecting to mini engine daemon");
				daemon = EngineFactory.connectToMiniEngineDaemon();
				
			}
		} catch (Exception e) {
			printFatalProblem(console,"Failed to connect to engine daemon",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to connect to engine daemon",false);
			return null;
		}
		
		//start up the engine
		controller = null;
		try {
			console.printInfo("Creating engine instance");
			controller = daemon.newEngine(bpel_program.getName(),console_tracel,engine_deps_array,bpel_program.getCoordinatorHost());
			
			//set JAR dependencies
			
			daemon.close();
		} catch (Exception e) {
			printFatalProblem(console,"Failed to create new engine instance",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to create new engine instance",false);
			return null;
		}
		
		long offset = 0;
		try {
			console.printInfo("Setting engine program");

			if (!useMiniEngine) {
				console.printInfo("");
				console.printInfo("Distribution:");
				console.printInfo("  (Coordinator host)   "+coordinator_hostname);
				
				for (int i = 0; i < prog.getAddressCount(); i++) {
					console.printInfo("  (Worker host)        "+((SessionAddress)prog.getAddresses().get(i)).getListenerHost());
				}
				console.printInfo("");
			}
			
			offset = controller.setProgram(prog);
		} catch (Exception e) {
			printFatalProblem(console,"Failed to set engine program in engine instance",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to set engine program in engine instance",false);
			return null;
		}

		try {
			if (!exitOnDisconnect) {
				controller.setHeadless(true);
				console.printInfo("Headless mode, engine will continue to run after workbench disconnects");
			}
		} catch (Exception e) {
			printFatalProblem(console,"Failed to set headless in engine instance",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to set headless in engine instance",false);
			return null;
		}
		
		try {
			if (debug) {
				console.printInfo("Debug mode, turning on all engine logging");
				controller.setLogLevel(true,true,false);
//				controller.setLogLevel(true,true,true);
			} else {
				console.printInfo("Turning on only warning and error logging");
				controller.setLogLevel(true,true,false);
			}
		} catch (Exception e) {
			printFatalProblem(console,"Failed to set log level in engine instance",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to set log level in engine instance",false);
			return null;
		}

		try {
			console.printInfo("Launching engine program");
			console.printInfo("");
			root_runner_ids = controller.launchRunner(1,"engine_main",0,new String[0]);
		} catch (Exception e) {
			printFatalProblem(console,"Failed to start engine program",e);
//			B2jPlugin.DBG.logVisibleError(e,"Failed to start engine program",false);
			return null;
		}
		
		running = true;
		
		return null;
	}
	
	//
	// Utility methods
	//
	
	/**
	 * Convert a list of JARDependency objects to an array
	 * @param list
	 * @return
	 */
	private JARDependency[] dependencyListToArray(ArrayList list) {
		JARDependency[] array = new JARDependency[list.size()];
		list.toArray(array);
		return array;
	}
		
	private void printFatalProblem(BPELEngineListener console, String description, Exception e) {
		console.printDebug("\n\nERROR: "+description);
		console.printInfo("\nProblem Description:");
		if (e.getMessage() != null) {
			console.printDebug(e.getMessage());
		} else {
			console.printDebug("");
		}
		console.printInfo("\nProblem Details:");
		console.printDebug(getStacktrace(e));
	}
	
	private static String getStacktrace(Throwable t) {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		PrintStream print = new PrintStream(bout);
		t.printStackTrace(print);
		return new String(bout.toByteArray()).replace('\t',' ');
	}


	/**
	 * Fetch all the JAR dependencies which the engine program has (e.g. user specified JARs or 
	 * WSDL binding JARs etc)
	 * @param console
	 * @return
	 */
	private ArrayList fetchAllProgramDependencies(BPELEngineListener console) throws IOException {
		DependencyInfo[] infos = B2jPlatform.getPortDependencyInfo();
		return fetchAllDependencies(console,infos);
	}
	
	/**
	 * Fetch all the JAR dependencies which the underlying java engine has (e.g. transport
	 * extension point JARs etc)
	 * @param console
	 * @return
	 */
	private ArrayList fetchAllEngineDependencies(BPELEngineListener console) throws IOException {
		DependencyInfo[] infos = B2jPlatform.getEngineDependencyInfo();
		return fetchAllDependencies(console,infos);
	}
	
	private ArrayList fetchAllDependencies(BPELEngineListener console, DependencyInfo[] infos) throws IOException {
		ArrayList jardeps = new ArrayList();
		
		for (int i = 0; i < infos.length; i++) {
			Properties[] props = infos[i].getResources();
			for (int k = 0; k < props.length; k++) {
				String path = props[k].getProperty("JAR");
				
				path = infos[i].getRelativePath(path);

				InputStream in = new FileInputStream(path);
				in = new BufferedInputStream(in);
				
				byte[] dat = StreamUtils.readAll(in);
				
				in.close();
				
				if (path != null) {
					JARDependency dep = new JARDependency(dat,path);
					jardeps.add(dep);
				}
			}
		}
		return jardeps;
	}
	
	private class IgnoredTraceListener implements BPELEngineListener {

		public void printInfo(String s) {
		}

		public void printDebug(String s) {
		}

		public void printEngineInfo(String s) {
		}

		public void printEngineDebug(String s) {
		}
	}
}