/*******************************************************************************

* Copyright (c) 2007 IONA Technologies PLC

* All rights reserved. This program and the accompanying materials

* are made available under the terms of the Eclipse Public License v1.0

* which accompanies this distribution, and is available at

* http://www.eclipse.org/legal/epl-v10.html

* 

* Contributors:

*     IONA Technologies PLC - initial API and implementation

*******************************************************************************/
package org.eclipse.stp.sc.sca.java.builders;

import java.io.IOException;
import java.util.Map;
import java.util.jar.JarOutputStream;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.utils.JDTUtils;
import org.eclipse.stp.sc.common.utils.JarUtils;
import org.eclipse.stp.sc.sca.java.workspace.ScaWorkspaceManager;

/**
 * @author jma
 * package the sca java project output to jar file format for launch.
 * <Notice> this builder should be added to the last in the builder list
 * Jar file layout
 * <li>/classes
 * <li>/META-INF
 * <li>/META-INF/MANIFEST
 * <li>/META-INF/sca/default.scdl
 * <li>/wsdl
 * <li> set main-class in MANIFEST file
 */
public class JarBuilder extends IncrementalProjectBuilder{

	public static final String BUILDER_ID = "org.eclipse.stp.sc.sca.java.jarbuilder";
	private static final LoggingProxy LOG = LoggingProxy.getlogger(JarBuilder.class);
	
	/**
	 * build the sca jar file
	 */
	protected IProject[] build(int kind, Map args, IProgressMonitor monitor) throws CoreException {
		if (kind == IncrementalProjectBuilder.FULL_BUILD) {
			LOG.debug("build jar in full build");
			buildJar(monitor);
		} else {
			IResourceDelta delta = getDelta(getProject());

            if (delta == null) {
            	buildJar(monitor);
            } else {
                incrementalBuild(delta, monitor);
            }
		}
		return null;
	}
	
	
	protected void incrementalBuild(IResourceDelta delta,
	        IProgressMonitor monitor) throws CoreException {
	        // the visitor does the work.
	        LOG.debug("incrementalBuild for project:" + getProject().getName());
	        delta.accept(new JarBuildDeltaVisitor());
	        
	}
	
	private void buildJar(IProgressMonitor monitor) throws CoreException {
		try {
			// delete the previous generated jar file
			deleteJar(monitor);
			JarOutputStream jos = null;
			IFile jarFile = ScaWorkspaceManager.getJarOutputFile(getProject());
			IFile manifestFile = ScaWorkspaceManager.getManiInfFile(
					getProject());
			if (manifestFile.exists()) {
				String manifestPath = manifestFile.getRawLocation().toOSString();
				// create jar
				LOG.debug("sca jar file:" + jarFile.getRawLocation().toOSString());
				jos = JarUtils.createJarOutputStream(jarFile
						.getLocation().toOSString(), manifestPath);
			} else {
				jos = JarUtils.createJarOutputStream(jarFile
						.getLocation().toOSString());
			}
			
			addClassFiles(jos);
			addScdlFile(jos);
			addWsdlFile(jos);
			addMavenFiles(jos);
			jos.close();
			ScaWorkspaceManager.getBinFolder(getProject()).refreshLocal(
					IResource.DEPTH_INFINITE, monitor);
		} catch (IOException ioe) {
			LOG.error("error during create jar file", ioe);
		}

	}
	
	private void addMavenFiles(JarOutputStream output) throws CoreException, IOException {
		IFolder metaFolder = getProject().getFolder("META-INF");
		if (!metaFolder.exists()) {
			return;
		}
		IFolder mvnFolder = metaFolder.getFolder("maven");
		if (!mvnFolder.exists()) {
			return;
		}
		
		JarUtils.addOneTargetDir(output, mvnFolder, "META-INF/maven");
	}
	
	private void addClassFiles(JarOutputStream output) throws CoreException, IOException {
    	IJavaProject javaProject = JDTUtils.findJavaProject(getProject().getName());
    	IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
    	IFolder clsFolder = root.getFolder(javaProject.getOutputLocation());
    	LOG.debug("add clsFolder:" + clsFolder.getRawLocation().toOSString());
    	JarUtils.addOneTargetDir(output, clsFolder, "");
    }
	
	private void addScdlFile(JarOutputStream output) throws CoreException, IOException {
		IFolder scdlFolder = ScaWorkspaceManager.getScdlFileFolder(getProject());
		LOG.debug("add scdlFolder:" + scdlFolder.getRawLocation().toOSString());
		JarUtils.addOneTargetDir(output, scdlFolder, "META-INF/sca");
	}
	
	private void addWsdlFile(JarOutputStream output) throws CoreException, IOException {
		IFolder wsdlFolder = ScaWorkspaceManager.getWSDLFolder(getProject());
		LOG.debug("add wsdlFolder:" + wsdlFolder.getRawLocation().toOSString());
		JarUtils.addOneTargetDir(output, wsdlFolder, ScaWorkspaceManager.WSDL_FOLDER);
	}
	
	/**
	 * clean the sca jar file
	 * 
	 * @param monitor,
	 *            the progress monitor
	 * @throws CoreException
	 */
	protected void clean(IProgressMonitor monitor) throws CoreException {
		LOG.debug("sca jar builder clean for the project:"
				+ getProject().getName());
		deleteJar(monitor);
		ScaWorkspaceManager.getBinFolder(getProject()).refreshLocal(
				IResource.DEPTH_INFINITE, monitor);
	}
	
	
	private void deleteJar(IProgressMonitor monitor) throws CoreException {
		IFile jarFile = ScaWorkspaceManager.getJarOutputFile(getProject());
		if (jarFile.exists()) {
			jarFile.delete(true, monitor);
		}
		
	}
	

    class JarBuildDeltaVisitor implements IResourceDeltaVisitor {
        public boolean visit(IResourceDelta delta) throws CoreException {
            IResource res = delta.getResource();

            switch (delta.getKind()) {
            case IResourceDelta.ADDED:
            case IResourceDelta.CHANGED:
            case IResourceDelta.REMOVED:
                checkResourceRebuild(res);
                break;
            default:
                break;
            }

            return true;
        }
    }
    
    private void checkResourceRebuild(IResource res) throws CoreException {
        if (!(res instanceof IFile)) {
            return;
        }

        IFile file = (IFile)res;
        String fileExt = file.getFileExtension();
        if (fileExt.equals("java") 
            || fileExt.equals("wsdl")
            || fileExt.equals("scdl")) {
        	buildJar(null);
        }
    }

}
