/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler.xsd;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.TranslatorLog;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLMap;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceException;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceTranslator;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xsdmap.XSDMap;
import org.eclipse.stp.b2j.core.misc.internal.XMLConfigUtil;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.XSDTypeTranslator;
import org.eclipse.stp.b2j.core.publicapi.jcompiler.CompilerUnit;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Document;
import org.eclipse.stp.b2j.core.xml.internal.w3c.DocumentBuilder;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Element;


/**
 * 
 * @author amiguel
 *
 * Translates XML Schema (XSD) source files into any necessary engine Program
 * classes (e.g. compex types -> classes)
 */
public class XSDTranslator {

Util compiler_util;
NamespaceTranslator nt;

XSDMap xsdmap;
WSDLMap wsdlmap;

XSDTypeTranslator[] codecs;

TranslatorLog log;
String LOG_SOURCE = "XSDTranslator";

class XSDState implements CompilerUnit {
	int SCOPE = 1;
	StringBuffer decl = new StringBuffer();
	StringBuffer prog = new StringBuffer();

	String pkg_name;
	String class_name;
	
	String superclass;
	ArrayList imports = new ArrayList();
	ArrayList ifaces = new ArrayList();
	
	ArrayList field_types = new ArrayList();
	ArrayList field_names = new ArrayList();
	ArrayList field_elems = new ArrayList();
	
	ArrayList attr_types = new ArrayList();
	ArrayList attr_names = new ArrayList();
	
	public XSDState(String pkg_name, String class_name) {
		this.pkg_name = pkg_name;
		this.class_name = class_name;
		
		addInterface(Util.jengine_package+".utils.XPathAccessible");
	}

	public String getPackageName() {
		return pkg_name;	
	}

	public String getClassName() {
		return NamespaceTranslator.getJavaClassName(class_name);
	}

	public String[] getFieldTypes() {
		String[] types = new String[field_types.size()];
		for (int i = 0; i < types.length; i++) {
			types[i] = (String)field_types.get(i);
		}
		return types;
	}
	public String[] getFieldNames() {
		String[] names = new String[field_names.size()];
		for (int i = 0; i < names.length; i++) {
			names[i] = (String)field_names.get(i);
		}
		return names;
	}
	public String[] getFieldElements() {
		String[] elems = new String[field_elems.size()];
		for (int i = 0; i < elems.length; i++) {
			elems[i] = (String)field_elems.get(i);
		}
		return elems;
	}

	public String[] getAttributeTypes() {
		String[] types = new String[attr_types.size()];
		for (int i = 0; i < types.length; i++) {
			types[i] = (String)attr_types.get(i);
		}
		return types;
	}
	public String[] getAttributeNames() {
		String[] names = new String[attr_names.size()];
		for (int i = 0; i < names.length; i++) {
			names[i] = (String)attr_names.get(i);
		}
		return names;
	}

	private void setSuperClass(String sname) {
		superclass = sname;
	}
	
	private void addImport(String iport) {
		imports.add(iport);
	}
	
	private void addImports(String[] imports) {
		for (int i = 0; i < imports.length; i++) {
			addImport(imports[i]);
		}
	}
	
	private void addInterface(String iface) {
		ifaces.add(iface);
	}

	private void addInterfaces(String[] ifaces) {
		for (int i = 0; i < ifaces.length; i++) {
			addInterface(ifaces[i]);
		}
	}

	private void addField(String type, String name, String element) {
		field_types.add(type);
		field_names.add(name);
		field_elems.add(element);
	}

	private void addAttribute(String type, String name) {
		attr_types.add(type);
		attr_names.add(name);
	}
	
	private void appendLine(String s) {
		for (int i = 0; i < SCOPE; i++) {
			prog.append("  ");	
		}
		prog.append(s).append("\n");
	}
	
	private void incrementScope() {
		SCOPE++;
	}
	private void decrementScope() {
		if (SCOPE >= 1) SCOPE--;	
	}

	public void declToString(StringBuffer buf) {

		//append my declarations
		buf.append(decl.toString());
	}
	
	public void progToString(StringBuffer buf) {

		buf.append("//@@START_STRIP\n");

		//append my logic
		buf.append("package "+pkg_name+";\n");
		buf.append("import "+Util.jengine_package+".utils.*;\n");
		buf.append("import "+Util.jengine_package+".core.*;\n");
		buf.append("import "+Util.jengine_package+".message.Message;\n");
		
		for (int i = 0; i < imports.size(); i++) {
			buf.append("import "+imports.get(i)+";\n");
		}
		
		
		if (superclass == null) {
			//doesn't extend anything specific == extends xsd:anyType
			String myname = getPackageName() + "." + getClassName();
			try {
				if (myname.equals(nt.qualify("xsd:anyType",false))) {
					//we are xsd:anyType and therefore abstract
					buf.append("public abstract class "+NamespaceTranslator.getJavaClassName(class_name)+" \n");
				} else {
					//we extend xsd:anyType
					buf.append("public class "+NamespaceTranslator.getJavaClassName(class_name)+" extends "+nt.qualify("xsd:anyType",false)+" \n");
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else {
			buf.append("public class "+NamespaceTranslator.getJavaClassName(class_name)+" extends "+superclass+" \n");
		}
		if (ifaces.size() > 0) {
			buf.append("  implements\n");	
		}
		for (int i = 0; i < ifaces.size(); i++) {
			buf.append("  "+ifaces.get(i));	
			if (i < ifaces.size()-1) {
				buf.append(",");	
			}
			buf.append("\n");
		}
		buf.append("{\n");	
		buf.append("//@@END_STRIP\n");

		buf.append("\n");	
		buf.append("  public String "+NamespaceTranslator.getElementTagName()+" = null;\n");
		buf.append("\n");	
		
		buf.append(prog.toString());

		xpathToString(buf);

		buf.append("//@@START_STRIP\n");
		buf.append("}\n");
		buf.append("//@@END_STRIP\n\n");

	}

	public void xpathToString(StringBuffer buf) {
		buf.append("public XNodeSet getFieldSet() {\n");
		buf.append("  XNodeSet fields = new XNodeSet("+field_names.size()+");\n");
		for (int i = 0; i < field_names.size(); i++) {
			String realname = String.valueOf(field_names.get(i));
			String fieldname = NamespaceTranslator.getElement(realname);
//			String fieldname = String.valueOf(field_names.get(i));
//			String realname = NamespaceTranslator.getRealName(fieldname);
			
			buf.append("  if ("+fieldname+" != null)\n");
			buf.append("    for (int i = 0; i < "+fieldname+".length; i++)\n");
			buf.append("      fields.add(new XNode(\""+realname+"\","+fieldname+"[i],"+fieldname+",i,null));\n");
		}
		buf.append("  return fields;\n");
		buf.append("}\n");
		buf.append("public XNodeSet getFieldSet(XNode parent) {\n");
		buf.append("  XNodeSet fields = new XNodeSet("+field_names.size()+");\n");
		for (int i = 0; i < field_names.size(); i++) {
			String realname = String.valueOf(field_names.get(i));
			String fieldname = NamespaceTranslator.getElement(realname);
//			String fieldname = String.valueOf(field_names.get(i));
//			String realname = NamespaceTranslator.getRealName(fieldname);
			
			buf.append("  if ("+fieldname+" != null)\n");
			buf.append("    for (int i = 0; i < "+fieldname+".length; i++)\n");
			buf.append("      fields.add(new XNode(\""+realname+"\","+fieldname+"[i],"+fieldname+",i,parent));\n");
		}
		buf.append("  return fields;\n");
		buf.append("}\n");
	}
	
	public String toString() {
		StringBuffer buf = new StringBuffer();
		
		buf.append("//XSD CLASS\n");
		declToString(buf);
		progToString(buf);
		
		return buf.toString();	
	}
	
//	public String toCompilerString() {
//		StringBuffer buf = new StringBuffer();
//		
//		buf.append("//XSD CLASS\n");
//		buf.append(decl);
//		buf.append(prog);
//		
//		return buf.toString();	
//	}

}//end class

int COMPLEX = 1;
int SIMPLE = 1;

	private String getNextComplex() {
		return "UnknownComplexType"+COMPLEX++;
	}

	private String getNextSimple() {
		return "UnknownSimpleType"+SIMPLE++;
	}

	public XSDTranslator(Util compiler_util, TranslatorLog log, XSDMap xsdmap, WSDLMap wsdlmap) {
		this.xsdmap = xsdmap;
		this.wsdlmap = wsdlmap;
		this.compiler_util = compiler_util;
		this.log = log;
		nt = compiler_util.createNamespaceTranslator();
	}

	public void translateElement(ArrayList states, XSDState state, Element element) throws XSDTranslatorException, NamespaceException {
		//element = field inside a class
		String name = element.getAttribute("name");
		String type = element.getAttribute("type");
		String ref = element.getAttribute("ref");

		String minOccurs = element.getAttribute("minOccurs");
		String maxOccurs = element.getAttribute("maxOccurs");
		String def = element.getAttribute("default");
		
		String xmldef = XMLConfigUtil.toRawXML(def);

		//TODO this element is a reference
		if (element.hasAttribute("ref")) {
			return;
		}
		
		//this element doesn't have a name - it doesn't map to a class field
		if (name.equals("")) {
			return;
		}
		
		if (type.equals("")) {

/*			//this is a reference to another element
			if (element.hasAttribute("ref")) {
				XSDRootElement refElem = xsdmap.getRootElementTypeByQName(nt.qualify(ref,false));
				
				if (refElem != null) {
					name = refElem.getNCName();
					type = refElem.getQualifiedType();
				} else {
					
				}
			}*/
			
			//no type (complex or simple); so type is defined within this element
			
			ArrayList elems = Util.getAllElements(element);
			
			for (int i = 0; i < elems.size(); i++) {
				Element complex = (Element)elems.get(i);
				String complexname = NamespaceTranslator.getName(complex);
				if (complexname.equals("complexType")) {
					
//					type = getNextComplex();
					type = name;
					
					complex.setAttribute("name",type);
					translateComplexType(states,complex);

				} else if (complexname.equals("simpleType")) {
					
//					type = getNextSimple();
					type = name;
					
					complex.setAttribute("name",type);
					translateSimpleType(states,complex);
					
				} else {
					log.logWarning(LOG_SOURCE,"Unknown element tag name underneath <element> \""+complexname+"\"");	
				}
			}

			//turn the type into a fully qualified java reference
			//need to use the target namespace here because thats what we used when we translated it
			type = nt.qualify(type,true);
			
		} else {
			//turn the type into a fully qualified java reference
			type = nt.qualify(type,false);
		}
		
		
		String mname = NamespaceTranslator.getElement(name);		
		
		if (state != null) {
			//TODO if we're a top level element then we have nowhere to make this mapping...
			state.addField(type,name,name);
			
			if (xmldef.length() > 0) {		
				state.appendLine("public "+type+" "+NamespaceTranslator.getDefaultElement(mname)+" = new "+type+"(\""+xmldef+"\");");
			}
			state.appendLine("public "+type+"[] "+mname+" = new "+type+"[0];");
		} else {
			log.logWarning(LOG_SOURCE, "UNIMPLEMENTED: Couldn't make top level mapping from "+name+" to "+type);
		}
	}

	public void translateSequence(ArrayList states, XSDState state, Element sequence) throws XSDTranslatorException, NamespaceException {
		//sequence = a bunch of fields in a class
		ArrayList elements = Util.getAllElements(sequence);
		for (int i = 0; i < elements.size(); i++) {
			//translate all the inner bits of this complex type
			translateType(states, state, (Element)elements.get(i));	
		}
	}
	public void translateComplexContent(ArrayList states, XSDState state, Element sequence) throws XSDTranslatorException, NamespaceException {
		//sequence = a bunch of fields in a class
		ArrayList elements = Util.getAllElements(sequence);
		for (int i = 0; i < elements.size(); i++) {
			//translate all the inner bits of this complex type
			translateType(states, state, (Element)elements.get(i));	
		}
	}
	public void translateRestriction(ArrayList states, XSDState state, Element sequence) throws XSDTranslatorException, NamespaceException {
		
		//TODO translate all the parent elements etc
		
		//sequence = a bunch of fields in a class
		ArrayList elements = Util.getAllElements(sequence);
		for (int i = 0; i < elements.size(); i++) {
			//translate all the inner bits of this complex type
			translateType(states, state, (Element)elements.get(i));	
		}
	}
	public void translateExtension(ArrayList states, XSDState state, Element sequence) throws XSDTranslatorException, NamespaceException {
		
		//TODO translate all the parent elements etc
		
		//sequence = a bunch of fields in a class
		ArrayList elements = Util.getAllElements(sequence);
		for (int i = 0; i < elements.size(); i++) {
			//translate all the inner bits of this complex type
			translateType(states, state, (Element)elements.get(i));	
		}
	}

	public void translateAttribute(ArrayList states, XSDState state, Element attribute) throws XSDTranslatorException, NamespaceException {
		
		String name = attribute.getAttribute("name");
		String ref = attribute.getAttribute("ref");
		String type = attribute.getAttribute("type");
		String use = attribute.getAttribute("use");
		String fixed = attribute.getAttribute("fixed");
		String def = attribute.getAttribute("default");
		
		//TODO this attribute is a reference
		if (attribute.hasAttribute("ref")) return;
			
		String qtype = nt.qualify(type,false);
		String xmldef = XMLConfigUtil.toRawXML(def);
		
		String mname = NamespaceTranslator.getAttribute(name);
		
		if (xmldef.length() > 0) {		
			//we have a default value, so compile that in
			state.appendLine("public "+qtype+" "+mname+" = new "+qtype+"(\""+xmldef+"\");");
		} else {
			//we have no default value, so compile in a null
			state.appendLine("public "+qtype+" "+mname+" = null;");
		}
		
		state.addAttribute(qtype,name);
	}

	public void translateSimpleType(ArrayList states, Element simple) throws XSDTranslatorException, NamespaceException {

		String name = simple.getAttribute("name");

		String pkg = nt.getPackage(name,true);
		String cname = NamespaceTranslator.getName(name);

		XSDState state = new XSDState(pkg,cname);

		String base = compiler_util.getSimpleTypeBase(nt,simple);
		
		state.setSuperClass(nt.qualify(base,false));

		log.logInfo(LOG_SOURCE,"simple type - "+nt.qualify(name,true)+" ("+base+")");
		
		for (int i = 0; i < codecs.length; i++) {
			//get the codecs to build implementation of any interfaces for this complex type
			try {
				codecs[i].translateSimple(nt,name,base,simple);
				
				state.appendLine(codecs[i].getMethods());
				state.addInterfaces(codecs[i].getImplementations());
				state.addImports(codecs[i].getImports());
			} catch (Exception e) {
				log.logWarning(LOG_SOURCE,"Problem when codec "+codecs[i].getClass()+" compiling complex XSD type "+name+": "+e);
//				DBG.logVisibleError(e,"Problem when codec "+codecs[i].getClass()+" compiling complex XSD type "+name,false);
			}
		}		

		states.add(state);

	}

	
	public void translateComplexType(ArrayList states, Element complex) throws XSDTranslatorException, NamespaceException {

		String name = complex.getAttribute("name");
		
		String pkg = nt.getPackage(name,true);
		String cname = NamespaceTranslator.getName(name);

		log.logInfo(LOG_SOURCE,"complex type - "+nt.qualify(name,true));
		
		XSDState state = new XSDState(pkg,cname);
		
		ArrayList elements = Util.getAllElements(complex);
		for (int i = 0; i < elements.size(); i++) {
			//translate all the inner bits of this complex type
			translateType(states, state, (Element)elements.get(i));	
		}
	
		for (int i = 0; i < codecs.length; i++) {
			//get the codecs to build implementation of any interfaces for this complex type
			try {
				codecs[i].translateComplex(nt,name,state.getAttributeTypes(),state.getAttributeNames(),state.getFieldTypes(),state.getFieldNames(),state.getFieldElements(),complex);
				
				state.appendLine(codecs[i].getMethods());
				state.addInterfaces(codecs[i].getImplementations());
				state.addImports(codecs[i].getImports());
			} catch (Exception e) {
				log.logWarning(LOG_SOURCE,"Problem when codec "+codecs[i].getClass()+" compiling complex XSD type "+name+": "+e);
			}
		}		
		
		states.add(state);
	}
	
	public void translateTopLevelElement(ArrayList states, XSDState state, Element elem) throws XSDTranslatorException, NamespaceException {
		String name = elem.getAttribute("name");
		String type = elem.getAttribute("type");
		
		if (type.equals("")) {
			
			if (name.equals("")) {
				throw new XSDTranslatorException("top level XSD element has neither name nor type");
			}
			
			translateElement(states,state,elem);
		} else {
			log.logWarning(LOG_SOURCE, "UNIMPLEMENTED: unable to map top level elements to types ("+state+")");
			
			log.logWarning(LOG_SOURCE, "making direct alias from "+compiler_util.getQNameFromQName(nt.qualify(name,true))+" to "+compiler_util.getQNameFromQName(nt.qualify(type,false)));
			//TODO just stick a straight alias from the element to the type in here, might break if the name != type?
			compiler_util.setXsdTypeDirectAlias(nt.qualify(name,true), nt.qualify(type,false));
			
			/*
			String pkg = nt.getPackage(name,true);
			String cname = NamespaceTranslator.getName(name);

			XSDState nstate = new XSDState(pkg,cname);

			String base = type;

			nstate.setSuperClass(nt.qualify(base,false));

			log.logInfo(LOG_SOURCE,"top level element mapping - "+nt.qualify(name,true)+" ("+base+")");
			
			for (int i = 0; i < codecs.length; i++) {
				//get the codecs to build implementation of any interfaces for this complex type
				try {
					codecs[i].translateSimple(nt,name,base,elem);
					
					nstate.appendLine(codecs[i].getMethods());
					nstate.addInterfaces(codecs[i].getImplementations());
					nstate.addImports(codecs[i].getImports());
				} catch (Exception e) {
					log.logWarning(LOG_SOURCE,"Problem when codec "+codecs[i].getClass()+" compiling complex XSD type "+name+": "+e);
				}
			}		

			states.add(nstate);*/
		}
		
		/*
		 * We use the XSDMap instead now
		for (int i = 0; i < codecs.length; i++) {
			try {
				codecs[i].translateTopLevelElement(nt,name,nt.qualify(type,false));
			} catch (Exception e) {
				DBG.logVisibleError(e,"Problem when codec "+codecs[i].getClass()+" compiling top level XSD element "+name,false);
			}
		}
		*/		
	}
	
	public void translateType(ArrayList states, XSDState state, Element type) throws XSDTranslatorException, NamespaceException {
		String tag = NamespaceTranslator.getName(type);

		if (state != null) {
			//we are already inside a class (complex Type)
			if (tag.equals("element")) {
				translateElement(states,state,type);
			} else if (tag.equals("sequence")) {
				translateSequence(states,state,type);
			} else if (tag.equals("attribute")) {
				translateAttribute(states,state,type);
			} else if (tag.equals("complexContent")) {
				translateComplexContent(states,state,type);
			} else if (tag.equals("restriction")) {
				translateRestriction(states,state,type);
			} else if (tag.equals("extension")) {
				translateExtension(states,state,type);
			}
		} else {
			if (tag.equals("element")) {
				translateTopLevelElement(states,state,type);
			}
		}
		//we are not inside a class (complex Type)
		//but we dont need to be as new complex types create
		//new classes anyway
		if (tag.equals("simpleType")) {
			translateSimpleType(states,type);
		} else if (tag.equals("complexType")) {
			translateComplexType(states,type);
		}
	}
	
	public void translateSchema(ArrayList states, Element schema) throws XSDTranslatorException, NamespaceException {

		nt.addNamespaces(schema);

		ArrayList elements = Util.getAllElements(schema);
		for (int i = 0; i < elements.size(); i++) {
			//translate all the complex types in this schema
			translateType(states, null, (Element)elements.get(i));	
		}
		
		nt.removeNamespaces(schema);
	}

	public void translateBaseType(ArrayList states, String name) throws XSDTranslatorException, NamespaceException {
		
		String pkg = nt.getPackage(name,false);
		String cname = NamespaceTranslator.getName(name);

		XSDState state = new XSDState(pkg,cname);

		String mapping = compiler_util.getInternalBaseMapping(nt,nt.qualify(name,false));
		
		boolean hasParent = !mapping.equals(nt.qualify(name,false));
		
		if (mapping.equals(nt.qualify(XSDTypeTranslator.XSD_TYPE_STRING,false))) {
			state.appendLine("public java.lang.String INTERNAL_VALUE = new String(\"\");\n");
		} else if (mapping.equals(nt.qualify(XSDTypeTranslator.XSD_TYPE_INT,false))) {
			state.appendLine("public Integer INTERNAL_VALUE = new Integer(0);\n");
		} else if (mapping.equals(nt.qualify(XSDTypeTranslator.XSD_TYPE_LONG,false))) {
			state.appendLine("public Long INTERNAL_VALUE = new Long(0);\n");
		} else if (mapping.equals(nt.qualify(XSDTypeTranslator.XSD_TYPE_DOUBLE,false))) {
			state.appendLine("public Double INTERNAL_VALUE = new Double(0);\n");
		} else if (mapping.equals(nt.qualify(XSDTypeTranslator.XSD_TYPE_BOOLEAN,false))) {
			state.appendLine("public Boolean INTERNAL_VALUE = new Boolean(false);\n");
		} else if (mapping.equals(nt.qualify(XSDTypeTranslator.XSD_TYPE_DATE,false))) {
			state.appendLine("public java.util.GregorianCalendar INTERNAL_VALUE = new java.util.GregorianCalendar();\n");
		} else if (mapping.equals(nt.qualify(XSDTypeTranslator.XSD_TYPE_LIST,false))) {
			state.appendLine("public java.util.List INTERNAL_VALUE = new java.util.ArrayList();\n");
		} else if (mapping.equals(nt.qualify(XSDTypeTranslator.XSD_TYPE_ANYTYPE,false))) {
			//nothing
		}

		if (hasParent && !mapping.equals(nt.qualify(XSDTypeTranslator.XSD_TYPE_ANYTYPE,false))) {
			//our superclass is not anytype
			state.setSuperClass(mapping);
		}
		
//		if (mapping.equals(nt.qualify(XSDCodec.XSD_TYPE_ANYTYPE))) {
//			state.setSuperClass(nt.qualify("xsd:anyType"));
//		}

		for (int i = 0; i < codecs.length; i++) {
			
			//get the codecs to build implementation of any interfaces for this base type
			try {
				codecs[i].translateBase(nt,name);
			 
				state.appendLine(codecs[i].getMethods());
				state.addInterfaces(codecs[i].getImplementations());
				state.addImports(codecs[i].getImports());
			} catch (XSDTranslatorException e) {
				throw e;
			} catch (Exception e) {
				log.logWarning(LOG_SOURCE,"Problem when codec "+codecs[i].getClass()+" compiling XSD base type "+name+": "+e);
				throw new XSDTranslatorException("Problem when codec "+codecs[i].getClass().getName()+" compiling XSD base type "+name+" (see log for more info)");
			}
		}		

		states.add(state);		
	}
	
	public void translateXsdAnyType(ArrayList states) throws XSDTranslatorException, NamespaceException {
		translateBaseType(states,"xsd:anyType");	
	}

	public void translateBaseTypes(ArrayList states) throws XSDTranslatorException, NamespaceException {

		//NamespaceTranslator understands "xsd:" by default
		
		String[] bases = Util.getXSDBaseTypes();
		for (int i = 0; i < bases.length; i++) {
			if (bases[i].equalsIgnoreCase("xsd:anyType")) {
				//SKIP ANYTYPE - its a special
			} else {
				translateBaseType(states,bases[i]);	
			}
		}
	}

	public XSDState[] getJava(String[] xsd, String[] wsdl, String[] xsd_files, String[] wsdl_files) throws Exception {
		
		ArrayList states = new ArrayList();

		for (int i = 0; i < xsd.length; i++) {
			DocumentBuilder builder = XMLConfigUtil.getDocumentBuilder();
			Document doc = builder.parse(new ByteArrayInputStream(xsd[i].getBytes()));
			
			Element doc_element = doc.getDocumentElement();
			
			//translate all types defined in the schema
			translateSchema(states,doc_element);
		}

		for (int i = 0; i < wsdl.length; i++) {
			//translate any WSDL type definitions
			DocumentBuilder builder = XMLConfigUtil.getDocumentBuilder();
			Document doc = builder.parse(new ByteArrayInputStream(wsdl[i].getBytes()));
			
			Element definitions = doc.getDocumentElement();
			
			nt.addNamespaces(definitions);
			
			ArrayList typesList = Util.getAllElements(definitions);
			for (int k = 0; k < typesList.size(); k++) {
				Element types = (Element)typesList.get(k);
				nt.addNamespaces(types);
				
				if (nt.checkQName(types,NamespaceTranslator.NAMESPACE_WSDL,"types")) {
					//found a wsdl types element
					ArrayList schemas = Util.getAllElements(types);
					for (int z = 0; z < schemas.size(); z++) {
						Element schema = (Element)schemas.get(z);
						nt.addNamespaces(schema);
						if (nt.checkQName(schema,NamespaceTranslator.NAMESPACE_XSD,"schema")) {
							translateSchema(states,schema);
						}
						nt.removeNamespaces(schema);
					}
				}
				nt.removeNamespaces(types);
			}
			
			nt.removeNamespaces(definitions);

			//translate all types defined in the schema
			translateSchema(states,definitions);
		}
		
		//translate base XSD types
		translateBaseTypes(states);
		
		//translate XSD:anyType last so that codecs can benefit from list of all mappings
		translateXsdAnyType(states);

		HashMap test = new HashMap();
		for (int i = 0; i < states.size(); i++) {
			CompilerUnit state = (XSDState)states.get(i);
			if (test.get(state.getClassName()) != null) {
				states.remove(i--);
			} else {
				test.put(state.getClassName(),state);
			}
		}
		
		XSDState[] ret = new XSDState[states.size()];
		for (int i = 0; i < ret.length; i++) {
			ret[i] = (XSDState)states.get(i);	
		}

		return ret;
	}

	public CompilerUnit[] getDefinitionSources(String[] xsd, String[] wsdl, XSDTypeTranslator[] codecs, String[] xsd_files, String[] wsdl_files) throws Exception {
		this.codecs = codecs;

		XSDState[] java = getJava(xsd,wsdl,xsd_files,wsdl_files);
		
		return java;
	}
	
//	class MappingClassComparator {
//	}

	
}