/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.internal.saf.impl;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.ecore.util.ExtendedMetaData;
import org.eclipse.osgi.util.NLS;
import org.eclipse.stp.core.saf.Context;
import org.eclipse.stp.core.saf.ISAFManager;
import org.eclipse.stp.core.saf.handler.AbstractComponentHandler;
import org.eclipse.stp.core.saf.handler.AbstractEntryPointHandler;
import org.eclipse.stp.core.saf.handler.AbstractExternalServiceHandler;
import org.eclipse.stp.core.saf.handler.AbstractHandler;
import org.eclipse.stp.core.saf.handler.AbstractInterfaceHandler;
import org.eclipse.stp.core.saf.handler.IComponentHandler;
import org.eclipse.stp.core.saf.handler.IEntryPointHandler;
import org.eclipse.stp.core.saf.handler.IExternalServiceHandler;
import org.eclipse.stp.core.saf.handler.IHandler;
import org.eclipse.stp.core.saf.handler.IInterfaceHandler;
import org.eclipse.stp.core.sca.Binding;
import org.eclipse.stp.core.sca.Component;
import org.eclipse.stp.core.sca.EntryPoint;
import org.eclipse.stp.core.sca.ExternalService;
import org.eclipse.stp.core.sca.Interface;
import org.eclipse.stp.core.sca.SCAPackage;
import org.osgi.framework.Bundle;

/**
 * This is the singleton class that user programs use to interact with the SOA assembly framework. 
 */
public class SAFManager implements ISAFManager {

    private static SAFManager instance = null;
	
	/**
	 * private constructor to enforce singleton
	 */
	private SAFManager()
	{
		super();
		initialize();
	}

	/**
	 * Returns the SAFManager singleton
	 * @return ISAFManager
	 */
	public static ISAFManager getInstance()
	{
		if (instance == null) {
			instance = new SAFManager();
		}
		return instance;
	}
	
//	/**
//	 * This class holds the information describing a kind of a component, import, or export
//	 * and its handler.
//	 */
//	public class KindDescriptor {
//
//		private String kind = null;
//		private String name = null;
//		private String description = null;
//		private URL smallIcon = null;
//		private URL largeIcon = null;
//		private IConfigurationElement configElement = null;
//		private Class clazz = null;
//		
//		/**
//		 * @param kind
//		 * @param name
//		 * @param description
//		 * @param smallIcon
//		 * @param largeIcon
//		 * @param handler
//		 * @param configElement
//		 */
//		public KindDescriptor(String kind, String name, String description, URL smallIcon, URL largeIcon, IConfigurationElement configElement,Class clazz) {
//			this.kind = kind;
//			this.name = name;
//			this.description = description;
//			this.smallIcon = smallIcon;
//			this.largeIcon = largeIcon;
//			this.configElement = configElement;
//			this.clazz = clazz;
//		}
//
//		/**
//		 * @param kind
//		 */
//		public KindDescriptor(String kind) {
//			this(kind, null, null, null, null, null, null);
//		}
//
//		public String getKind() {
//			return kind;
//		}
//
//		public String getName() {
//			return name;
//		}
//
//		public String getDescription() {
//			return description;
//		}
//
//		public URL getSmallIcon() {
//			return smallIcon;
//		}
//
//		public URL getLargeIcon() {
//			return largeIcon;
//		}
//		
//		/**
//		 * @return the handler
//		 */
//		public IHandler getHandler() {
//			
//			IHandler handler = null;
//			try {
//				handler = (IHandler) configElement.createExecutableExtension(ISAFConstants.CLASS_ATTR);
//			} catch (Exception e) {
//				// Consume the exception
//			}
//			if (handler == null) {
//				SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
//						NLS.bind(Messages.stp_core_saf_cannotLoadHandlerClass, configElement.getName(), configElement.getDeclaringExtension().getNamespace()));
//			}
//			
//			if (!clazz.isInstance(handler)) {
//				
//				String msg = null;
//				if(clazz.equals(AbstractComponentHandler.class))
//						msg = Messages.stp_core_saf_classDoesNotExtendIComponentHandler;
//				else if(clazz.equals(AbstractEntryPointHandler.class))
//					msg = Messages.stp_core_saf_classDoesNotExtendIEntryPointHandler;	
//				else if(clazz.equals(AbstractExternalServiceHandler.class))
//					msg = Messages.stp_core_saf_classDoesNotExtendIExternalServiceHandler;	
//				else if(clazz.equals(AbstractInterfaceHandler.class))
//					msg = Messages.stp_core_saf_classDoesNotExtendIInterfaceHandler;	
//			
//				SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
//				    NLS.bind(msg,configElement.getDeclaringExtension().getNamespace(), 
//				    		configElement.getAttribute(ISAFConstants.CLASS_ATTR)));
//				handler = null;
//			}
//			
//			if(handler != null)
//				((AbstractHandler)handler).initialize(kind, name, description, smallIcon, largeIcon);
//
//			return handler;
//		}
//
//		/**
//		 * @return the configuration element
//		 */
//		public IConfigurationElement getConfigElement() {
//			return configElement;
//		}
//
//
//	} // class KindDescriptor
	
	private class KindConstraint {
	
		private boolean includes = true;
		private boolean extensible = true;
		private List kinds = null;
		
		/**
		 * @param include
		 * @param extensible
		 */
		public KindConstraint(boolean include, boolean extensible){
			this.includes = include;
			this.extensible = extensible;
			this.kinds = new ArrayList();
		}
		
		/**
		 * @return <code>true</code> if the context is extensible
		 */
		public boolean isExtensible(){
			return extensible;
		}
		
		/**
		 * @return <code>true</code> if the context includes the specified
		 *         kinds; <code>false</code> otherwise
		 */
		public boolean includes(){
			return includes;
		}
		
		/**
		 * @param kind the kind to add to the list
		 */
		public void addKind(String kind){
			kinds.add(kind);
		}
		
		/**
		 * @return list of strings representing the kinds defined by this constraint 
		 */
		public List getKinds(){
			return kinds;
		}

	}
	
	private class ContextParserOutput{
		private List contextKeys;
		private KindConstraint compKindConstraint;
		private KindConstraint epKindConstraint;
		private KindConstraint esKindConstraint;
		
		/**
		 * @param contextKeys
		 * @param compKindConstraint
		 * @param epKindConstraint
		 * @param esKindConstraint
		 */
		public ContextParserOutput(List contextKeys, KindConstraint compKindConstraint, KindConstraint epKindConstraint, KindConstraint esKindConstraint){
			this.contextKeys = contextKeys;
			this.compKindConstraint = compKindConstraint;
			this.epKindConstraint = epKindConstraint;
			this.esKindConstraint = esKindConstraint;			
		}
		
		/**
		 * @return the componentKindConstraint defined by the context
		 */
		public KindConstraint getCompKindConstraint() {
			return compKindConstraint;
		}
		/**
		 * @return List of context keys derived from processing CONTEXTKEY elements.
		 */
		public List getContextKeys() {
			return contextKeys;
		}
		/**
		 * @return the entryPointKindConstraint defined by the context
		 */
		public KindConstraint getEpKindConstraint() {
			return epKindConstraint;
		}
		/**
		 * @return the externalServiceKindConstraint defined by the context
		 */
		public KindConstraint getEsKindConstraint() {
			return esKindConstraint;
		}
		
		
	}
	
	// =================================================================================
	
	// Handler cache
	private Hashtable componentHandlers = null;
	private Hashtable externalServiceHandlers = null;
	private Hashtable entryPointHandlers = null;
	private Hashtable interfaceHandlers = null;
	// Context cache
	private Hashtable componentKindFilters = null;
	private Hashtable externalServiceKindFilters = null;
	private Hashtable entryPointKindFilters = null;
		
	private void initialize(){ 
	
		componentHandlers = new Hashtable();
		externalServiceHandlers = new Hashtable();
		entryPointHandlers = new Hashtable();
		interfaceHandlers = new Hashtable();

		IExtension[] extensions = SAFActivator.getDefault().getHandlerContributions();
		int extensionCount = extensions.length;
		for (int i = 0; i < extensionCount; i++) {
			IExtension extension = extensions[i];
			IConfigurationElement[] elements = extension.getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				IConfigurationElement element = elements[j];


				String kind = element.getAttribute(ISAFConstants.KIND_ATTR);
				if ((kind == null) || kind.equals("")) { //$NON-NLS-1$
					SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
							NLS.bind(Messages.stp_core_saf_attributeKindMissing, element.getName(), extension.getNamespace()));
					continue;
				}

				String descKindName = element.getAttribute(ISAFConstants.DESC_KIND_NAME_ATTR);
				if ((descKindName == null) || descKindName.equals("")) { //$NON-NLS-1$
					SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
							NLS.bind(Messages.stp_core_saf_attributeDescriptiveKindNameMissing, element.getName(), extension.getNamespace()));
					continue;
				}
				
				String name = element.getAttribute(ISAFConstants.NAME_ATTR);
				if ((name == null) || name.equals("")) { //$NON-NLS-1$
					SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
							NLS.bind(Messages.stp_core_saf_attributeNameMissing, element.getName(), extension.getNamespace()));
					continue;
				}

				String description = element.getAttribute(ISAFConstants.DESCRIPTION_ATTR);

				String smallIconURI = element.getAttribute(ISAFConstants.SMALL_ICON_ATTR);
				URL smallIconURL = null;
				if (smallIconURI != null) {
					try {
						smallIconURL = getImageURL(smallIconURI, extension);
					} catch (MalformedURLException e) {
						SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
								NLS.bind(Messages.stp_core_saf_attributeIconInvalid, new Object[] { smallIconURI,
										ISAFConstants.SMALL_ICON_ATTR, element.getName(), extension.getNamespace() }));
					}
				}

				String largeIconURI = element.getAttribute(ISAFConstants.LARGE_ICON_ATTR);
				URL largeIconURL = null;
				if (largeIconURI != null) {
					try {
						largeIconURL = getImageURL(largeIconURI, extension);
					} catch (MalformedURLException e) {
						SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
								NLS.bind(Messages.stp_core_saf_attributeIconInvalid, new Object[] { largeIconURI,
										ISAFConstants.LARGE_ICON_ATTR, element.getName(), extension.getNamespace() }));
					}
				}

				IHandler handler = null;
				try {
					handler = (IHandler) element.createExecutableExtension(ISAFConstants.CLASS_ATTR);
				} catch (Exception e) {
					// Consume the exception
				}
				if (handler == null) {
					SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
							NLS.bind(Messages.stp_core_saf_cannotLoadHandlerClass, element.getName(), extension.getNamespace()));
					continue;
				}
					
				((AbstractHandler)handler).initialize(kind, descKindName, name, description, smallIconURL, largeIconURL);
				
				if (element.getName().equals(ISAFConstants.COMPONENT_HANDLER_TAG)) {
					checkAndAddToHandlerMap(handler, element, componentHandlers, AbstractComponentHandler.class, extension,
							Messages.stp_core_saf_componentHandlerAlreadyExists);
				} else if (element.getName().equals(ISAFConstants.EXTERNAL_SERVICE_HANDLER_TAG)) {
					checkAndAddToHandlerMap(handler, element, externalServiceHandlers, AbstractExternalServiceHandler.class, extension,
							Messages.stp_core_saf_externalServiceHandlerAlreadyExists);
				} else if (element.getName().equals(ISAFConstants.ENTRY_POINT_HANDLER_TAG)) {
					checkAndAddToHandlerMap(handler, element, entryPointHandlers, AbstractEntryPointHandler.class, extension,
							Messages.stp_core_saf_entryPointHandlerAlreadyExists);
				} else if (element.getName().equals(ISAFConstants.INTERFACE_HANDLER_TAG)) {
					checkAndAddToHandlerMap(handler, element, interfaceHandlers, AbstractInterfaceHandler.class, extension,
							Messages.stp_core_saf_interfaceHandlerAlreadyExists);
				}

			}
		}

		// Now process CONTEXT and CONTEXT EXTENSIONS.
		// 
		// His is done in 3 steps.
		//
		// Step 1:
		// First go through each context extension point and process each of its
		// child elements: contextKey(s), componentKindConstraint, entryPointKindConstraint,
		// and externalServiceKindConstraint.
		//
		// Each child element is processed independently as follows:
		// contextKey: 
		//		- generate a context key as follows: {namespace}rootType#subType
		//		- store each key in a context key list
		//
		// componentKindConstraint / entryPointKindConstraint / externalServiceKindConstraint:
		//		- cycle through each kind and expand the list by removing wildcards
		//
		// Once the elements are processed, for each key in the context key list generate a new
		// map entry into a staging hashmap that will hold the map of keys to kind constraint list. 
		// There is one staging map per constraint type (component, entry point, external service).
		// This concludes processing of the CONTEXT extension points.
		//
		// Step 2:
		// Process the CONTEXT EXTENSIONS.  Each of these extensions has the same structure as a CONTEXT,
		// so the child elements are processed the same way.
		//
		// Once the list of constraint keys is generated, do the following for each constraint type:
		//  1 - Check to see if the staging map defines the key.  If it doesn't, it is an error, because an extension was defined
		//      for a context that doesn't exist.
		//  2 - If the staging map defines the key, see if the corresponding context is extensible.
		//		If it's not, it is an error, because an extension was defined for a context that cannot be extended.
		//  3 - Merge the current constraint kind list in the staging map with the constraint kind list from the extension.
		//
		// Step 3:
		// All constraint kind lists have been merged and are ready to be processed.  They are organized by context key and
		// are stored in the staging maps.
		//
		// For each entry in the list, cycle through the kinds and build a new map (kind -> KindDescriptor).  This map is registered
		// in the global map keyed off by the corresponding context key.  The result is that for each context key a new handler's map is defined,
		// a shrunken version of the full handler contribution map.  
		//
		// If a context is provided by the user, the corresponding handler map is used.  If none is provided, then the global map is
		// used instead.
		//
		
		
		// STEP 1: process CONTEXT
		
		Hashtable stagingComponentConstraints = new Hashtable();
		Hashtable stagingEntryPointConstraints = new Hashtable();
		Hashtable stagingExternalServiceConstraints = new Hashtable();

		extensions = SAFActivator.getDefault().getContextContibutions();
		extensionCount = extensions.length;
		for (int i = 0; i < extensionCount; i++) {
			IExtension extension = extensions[i];
			IConfigurationElement[] elements = extension.getConfigurationElements();

			// Cycle through context elements
			for (int j = 0; j < elements.length; j++) {
				IConfigurationElement element = elements[j];

				String extensible = element.getAttribute(ISAFConstants.EXTENSIBLE_ATTR);
				if ((extensible == null) || extensible.equals("")) { //$NON-NLS-1$
					SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
							NLS.bind(Messages.stp_core_saf_attributeMissing, new Object[] { ISAFConstants.EXTENSIBLE_ATTR,
									element.getName(), extension.getNamespace() }));
					continue;
				}

				boolean isExtensible = Boolean.valueOf(extensible).booleanValue();

				// Cycle through context's children (contextKey,
				// componentKindConstrain, externalServiceKindConstrain,
				// entryPointKindConstrain)
				ContextParserOutput parserOutput = parseContextChildren(false, element, extension, isExtensible);

				List contextKeys = parserOutput.getContextKeys();
				KindConstraint componentConstraint = parserOutput.getCompKindConstraint();
				KindConstraint entryPointConstraint = parserOutput.getEpKindConstraint();
				KindConstraint externalServiceConstraint = parserOutput.getEsKindConstraint();

				// generate context entries
				if (!contextKeys.isEmpty()) {
					for (Iterator iter = contextKeys.iterator(); iter.hasNext();) {
						String key = (String) iter.next();
						if (componentConstraint != null) {
							if (stagingComponentConstraints.containsKey(key)) {
								SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
										NLS.bind(Messages.stp_core_saf_contextAlreadyDefinedForComponent, new Object[] { key,
												extension.getNamespace() }));
							} else
								stagingComponentConstraints.put(key, componentConstraint);
						}
						if (entryPointConstraint != null) {
							if (stagingEntryPointConstraints.containsKey(key)) {
								SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
										NLS.bind(Messages.stp_core_saf_contextAlreadyDefinedForEntryPoint, new Object[] { key,
												extension.getNamespace() }));
							} else
								stagingEntryPointConstraints.put(key, entryPointConstraint);
						}
						if (externalServiceConstraint != null) {
							if (stagingExternalServiceConstraints.containsKey(key)) {
								SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
										NLS.bind(Messages.stp_core_saf_contextAlreadyDefinedForExternalService, new Object[] { key,
												extension.getNamespace() }));
							} else
								stagingExternalServiceConstraints.put(key, externalServiceConstraint);
						}
					}
				}
			}
		}

		// STEP 2: process CONTEXT EXTENSION

		extensions = SAFActivator.getDefault().getContextExtensionContibutions();
		extensionCount = extensions.length;
		for (int i = 0; i < extensionCount; i++) {
			IExtension extension = extensions[i];
			IConfigurationElement[] elements = extension.getConfigurationElements();

			// Cycle through context elements
			for (int j = 0; j < elements.length; j++) {
				IConfigurationElement element = elements[j];

				// Cycle through contextExtension's children (contextKey,
				// componentKindConstrain, externalServiceKindConstrain,
				// entryPointKindConstrain)
				ContextParserOutput parserOutput = parseContextChildren(true, element, extension, false);

				List contextKeys = parserOutput.getContextKeys();
				KindConstraint componentConstraint = parserOutput.getCompKindConstraint();
				KindConstraint entryPointConstraint = parserOutput.getEpKindConstraint();
				KindConstraint externalServiceConstraint = parserOutput.getEsKindConstraint();

				// generate filter entries
				if (!contextKeys.isEmpty()) {
					for (Iterator iter = contextKeys.iterator(); iter.hasNext();) {
						String key = (String) iter.next();
						if (componentConstraint != null) {
							mergeContextExtensions(key, componentConstraint, stagingComponentConstraints, extension,
									Messages.stp_core_saf_contextComponentConstraintInvalid,
									Messages.stp_core_saf_contextExtensionConflictForComponent);
						}
						if (entryPointConstraint != null) {
							mergeContextExtensions(key, entryPointConstraint, stagingEntryPointConstraints, extension,
									Messages.stp_core_saf_contextEntryPointConstraintInvalid,
									Messages.stp_core_saf_contextExtensionConflictForEntryPoint);
						}
						if (externalServiceConstraint != null) {
							mergeContextExtensions(key, externalServiceConstraint, stagingExternalServiceConstraints, extension,
									Messages.stp_core_saf_contextExternalServiceConstraintInvalid,
									Messages.stp_core_saf_contextExtensionConflictForExternalService);
						}
					}
				}
			}
		}

		// STEP 3: Generate the global context maps from the staging maps.
		
		// At this point, the staging list has all the types defined for each
		// context key. We now build the global hash map with the corresponding
		// list of handlers for each list.

		componentKindFilters = generateKindsList(stagingComponentConstraints, componentHandlers);
		entryPointKindFilters = generateKindsList(stagingEntryPointConstraints, entryPointHandlers);
		externalServiceKindFilters = generateKindsList(stagingExternalServiceConstraints, externalServiceHandlers);

	}
		
	private void  mergeContextExtensions(String contextExtensionKey, KindConstraint extConstraint, Hashtable stagingConstraintMap, IExtension extension, String contextConstraintInvalid_msg, String contextExtensionConflict_msg) {
	
		// No context is defined with the key; therefore, it cannot be extended. Ignore.
		if(!stagingConstraintMap.containsKey(contextExtensionKey)){
			// extension defined is invalid....no context to extend...ignore
			SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
				    NLS.bind(contextConstraintInvalid_msg,
				    		new Object[] {contextExtensionKey, extension.getNamespace()}));

		}else{
			KindConstraint contextConstraint = (KindConstraint) stagingConstraintMap.get(contextExtensionKey);
			// Key is defined, but context is not extensible.
			if(contextConstraint != null && !contextConstraint.isExtensible()){
				SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
					    NLS.bind(Messages.stp_core_saf_contextNotExtensible,
					    		new Object[] {contextExtensionKey, extension.getNamespace()}));
			}else{
				boolean success = false;
				// If either they both exclude or include then merge the list of
				// kinds.
				// If one excludes and the other one includes, there is no need
				// to merge. The
				// extension's constraints are ignored, because the context
				// contrain overrides it.
				if(!(extConstraint.includes() ^ contextConstraint.includes())){
					for (Iterator iter = extConstraint.getKinds().iterator(); iter.hasNext();) {
						String constraint = (String) iter.next();
						if(!contextConstraint.getKinds().contains(constraint))
							contextConstraint.getKinds().add(constraint);
						
					}
					success = true;
				}
								
				if(!success){
					SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
						    NLS.bind(contextExtensionConflict_msg,
						    		new Object[] {contextExtensionKey, extension.getNamespace()}));
				}									
			}								
		}

	}

	private ContextParserOutput parseContextChildren(boolean processingExtension, IConfigurationElement context, IExtension extension, boolean isExtensible){
		
		List contextKeys = new ArrayList();
		KindConstraint componentConstraint = null;
		KindConstraint entryPointConstraint = null;
		KindConstraint externalServiceConstraint = null;

		IConfigurationElement[] children = context.getChildren();
		
		for (int k = 0; k < children.length; k++) {  
			IConfigurationElement child = children[k];
			
			if(child.getName().equals(ISAFConstants.CONTEXT_KEY_TAG)){
				
				String namespace = child.getAttribute(ISAFConstants.NAMESPACE_ATTR);
				if ((namespace == null) || namespace.equals("")) { //$NON-NLS-1$
					SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
					    NLS.bind(Messages.stp_core_saf_attributeMissing,
					    		new Object[] {ISAFConstants.NAMESPACE_ATTR,
					    		child.getName(), extension.getNamespace()}));
					continue;
				}
				
				String rootType = child.getAttribute(ISAFConstants.ROOT_TYPE_ATTR);
				if ((rootType == null) || rootType.equals("")) { //$NON-NLS-1$
					SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
					    NLS.bind(Messages.stp_core_saf_attributeMissing,
					    		new Object[] {ISAFConstants.ROOT_TYPE_ATTR,
					    		child.getName(), extension.getNamespace()}));
					continue;
				}
				
				String subType = child.getAttribute(ISAFConstants.SUB_TYPE_ATTR);
				contextKeys.add(createFilterKey(namespace, rootType, subType));
				
			}else{ 

				if((child.getName().equals(ISAFConstants.COMPONENT_KIND_CONSTRAINT_TAG) && componentConstraint != null) ||
						(child.getName().equals(ISAFConstants.ENTRY_POINT_KIND_CONSTRAINT_TAG) && entryPointConstraint != null) ||
						(child.getName().equals(ISAFConstants.EXTERNAL_SERVICE_KIND_CONSTRAINT_TAG) && externalServiceConstraint != null)){
					
					SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
						    NLS.bind(Messages.stp_core_saf_duplicateElement,
						    		new Object[] {child.getName(),
						    		context.getName(), extension.getNamespace()}));
					continue;
				}
				
			    String kindsIncluded = child.getAttribute(ISAFConstants.INCLUDE_KINDS_ATTR);
				if ((kindsIncluded == null) || kindsIncluded.equals("")) { //$NON-NLS-1$
					SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
					    NLS.bind(Messages.stp_core_saf_attributeMissing,
					    		new Object[] {ISAFConstants.INCLUDE_KINDS_ATTR,
					    		child.getName(), extension.getNamespace()}));
					continue;
				}
				
				boolean kindsInc = Boolean.valueOf(kindsIncluded).booleanValue();

				String kinds = child.getAttribute(ISAFConstants.KINDS_ATTR);
				if ((kinds == null) || kinds.equals("")) { //$NON-NLS-1$
					SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
					    NLS.bind(Messages.stp_core_saf_attributeMissing,
					    		new Object[] {ISAFConstants.KINDS_ATTR,
					    		child.getName(), extension.getNamespace()}));
					continue;
				}
				
				// Parse the kinds list.
				List kindsList = new ArrayList();
				kinds = kinds.replaceAll("[*]", ".*"); //$NON-NLS-1$ //$NON-NLS-2$
				StringTokenizer tokenizer = new StringTokenizer(kinds,","); //$NON-NLS-1$
				while(tokenizer.hasMoreTokens()){
					kindsList.add(tokenizer.nextToken());
				}
			       
				if(child.getName().equals(ISAFConstants.COMPONENT_KIND_CONSTRAINT_TAG)){
					componentConstraint =  expandKindsList(isExtensible, kindsInc, kindsList,componentHandlers); 
				}else if(child.getName().equals(ISAFConstants.ENTRY_POINT_KIND_CONSTRAINT_TAG)){
					entryPointConstraint = expandKindsList(isExtensible,kindsInc,kindsList,entryPointHandlers);
				}else if(child.getName().equals(ISAFConstants.EXTERNAL_SERVICE_KIND_CONSTRAINT_TAG)){
					externalServiceConstraint = expandKindsList(isExtensible,kindsInc,kindsList,externalServiceHandlers);
				}
			}
		}
		
		if(!processingExtension){
			
			if(componentConstraint == null){
				componentConstraint = new SAFManager.KindConstraint(true, isExtensible);
				componentConstraint.getKinds().addAll(componentHandlers.keySet());
			}
			
			if(entryPointConstraint == null){
				entryPointConstraint = new SAFManager.KindConstraint(true, isExtensible);
				entryPointConstraint.getKinds().addAll(entryPointHandlers.keySet());
			}
			
			if(externalServiceConstraint == null){
				externalServiceConstraint = new SAFManager.KindConstraint(true, isExtensible);
				externalServiceConstraint.getKinds().addAll(externalServiceHandlers.keySet());
			}
		}
		
		return new ContextParserOutput(contextKeys, componentConstraint, entryPointConstraint, externalServiceConstraint);		
	}
	
	private KindConstraint expandKindsList(boolean isExtensible, boolean kindsInc, List kindsList,	Hashtable handlersMap) {

		KindConstraint result = new SAFManager.KindConstraint(kindsInc,isExtensible);

		// If there is one wildcard entry, 
		// ignore the list and include all handlers.
		if (kindsList.contains(".*")){ //$NON-NLS-1$
			result.getKinds().addAll(Arrays.asList(handlersMap.keySet().toArray()));
			return result;
		}
		// Otherwise, cycle through the list of handler keys and find matching ones.
		for (Iterator iterator = handlersMap.keySet().iterator(); iterator
				.hasNext();) {
			String key = (String) iterator.next();
			for (Iterator iter = kindsList.iterator(); iter.hasNext();) {
				String kind = (String) iter.next();
				if (key.matches(kind)){
					result.addKind(key);
					break;
				}
			}
		}
		
		return result;
	}
	
	private Hashtable generateKindsList(Hashtable stagedMap, Hashtable handlersMap) {

		Hashtable result = new Hashtable();
		
		for (Iterator iter = stagedMap.entrySet().iterator(); iter.hasNext();) {
			Map.Entry stagedEntry = (Map.Entry) iter.next();
			
			Hashtable handlers = new Hashtable();
		
			KindConstraint constraint = (KindConstraint)stagedEntry.getValue();
			boolean include = constraint.includes();

			for (Iterator iterator = handlersMap.entrySet().iterator(); iterator.hasNext();) {
				Map.Entry handlersEntry = (Map.Entry) iterator.next();
					
				// If we're including and they match, add it
				if((include && constraint.getKinds().contains(handlersEntry.getKey())) ||
						// or if we're excluding and they don't match, add it
						(!include && !constraint.getKinds().contains(handlersEntry.getKey())))
					handlers.put(handlersEntry.getKey(), handlersEntry.getValue());

			}
			result.put(stagedEntry.getKey(), handlers);
		}
		return result;
	}
	
	private Collection getFilteredKinds(Context context, Hashtable handlers, Hashtable filters){
		
		Hashtable map = null;
		if(context != null){
			String key = createFilterKey(context);
			map = (Hashtable)filters.get(key);
		}
		
		if(map == null)
			map = handlers;
		
		return map.values();
	}
	
	private String createFilterKey(Context context){
		return createFilterKey(context.getNamespace(), context.getRootType(), context.getSubType());
	}
	
	private String createFilterKey(String namespace, String rootType, String subType){
		if(namespace == null && rootType == null) return null;
		String key = namespace+":"+rootType; //$NON-NLS-1$
		if(subType != null)
			key = key + "#" + subType; //$NON-NLS-1$
		return key;		
	}
	
	private URL getImageURL(String uri, IExtension extension) throws MalformedURLException{
		String pluginId = extension.getNamespace();

		Bundle bundle = Platform.getBundle(pluginId);

		URL baseURL = bundle.getEntry("/"); //$NON-NLS-1$
		return new URL(baseURL, uri);
	}

	private void checkAndAddToHandlerMap(IHandler handler, 
						IConfigurationElement element, Hashtable handlerMap, 
						Class handlerClass, IExtension extension,
						String dupHandlerKey)
	{
		if (!handlerClass.isInstance(handler)) {
			SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
			    NLS.bind(Messages.stp_core_saf_classDoesNotExtendAbstractHandler,
			    		new Object[]{element.getAttribute(ISAFConstants.CLASS_ATTR),
			    		extension.getNamespace(),handlerClass.getName()}));
			return;
		}
		String kind = handler.getKind();
		Object oldHandler = handlerMap.get(kind);
		if ((oldHandler != null) && !oldHandler.equals(handler)){
			SAFActivator.log(this, "initialize", IStatus.WARNING, //$NON-NLS-1$
				NLS.bind(dupHandlerKey, new Object[]{ kind,
				    	extension.getNamespace(), 
				    	element.getAttribute(ISAFConstants.CLASS_ATTR)}));
		}
		
		handlerMap.put(kind, handler);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getAllComponentKinds(org.eclipse.stp.core.saf.Context)
	 */
	public IComponentHandler[] getAllComponentHandlers(Context context)
	{
		Collection handlers = getFilteredKinds(context, componentHandlers, componentKindFilters);
		return (IComponentHandler[])handlers.toArray(new IComponentHandler[handlers.size()]);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getAllExternalServiceKinds(org.eclipse.stp.core.saf.Context)
	 */
	public IExternalServiceHandler[] getAllExternalServiceHandlers(Context context)
	{
		Collection handlers = getFilteredKinds(context, externalServiceHandlers, externalServiceKindFilters);
		return (IExternalServiceHandler[])handlers.toArray(new IExternalServiceHandler[handlers.size()]);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getAllEntryPointKinds(org.eclipse.stp.core.saf.Context)
	 */
	public IEntryPointHandler[] getAllEntryPointHandlers(Context context)
	{
		Collection handlers = getFilteredKinds(context, entryPointHandlers, entryPointKindFilters);
		return (IEntryPointHandler[])handlers.toArray(new IEntryPointHandler[handlers.size()]);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getAllInterfaceKinds()
	 */
	public IInterfaceHandler[] getAllInterfaceHandlers()
	{
		Collection handlers = getFilteredKinds(null, interfaceHandlers, null);
		return (IInterfaceHandler[])handlers.toArray(new IInterfaceHandler[handlers.size()]);
	}

	// ================
	// Create Component
	// ================

	
	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getComponentKindsFor(org.eclipse.stp.core.saf.Context, java.lang.Object)
	 */
	public IComponentHandler[] getComponentHandlersFor(Context context, Object implementationObj)
	{
		ArrayList result = new ArrayList();
		Collection handlers = getFilteredKinds(context, componentHandlers, componentKindFilters);
		for (Iterator iter = handlers.iterator(); iter.hasNext();) {
			IComponentHandler handler = (IComponentHandler)iter.next();
			try {
				if (handler.canCreateComponentFor(implementationObj)) {
					result.add(handler);
				}
			}
			catch(Exception exc) {
				SAFActivator.log(this, "getComponentKindsFor", IStatus.WARNING, exc); //$NON-NLS-1$
			}
		}
		return (IComponentHandler[])result.toArray(new IComponentHandler[result.size()]); 
	}

	
	static String defaultComponentImplementation = null;
	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getDefaultComponentKind()
	 */
	public String getDefaultComponentKind()
	{
		if (defaultComponentImplementation == null) {
			defaultComponentImplementation = ExtendedMetaData.INSTANCE.getName(SCAPackage.eINSTANCE.getSCACoreRoot_ImplementationAbstract());
		}
		return defaultComponentImplementation;
	}

	/**
	 * Returns the QName string for the kind of the specified component.
	 * 
	 * @param theComponent
	 *            Component being queried
	 * @return Component kind string
	 */
	private String getComponentKindOf(Component theComponent)
	{
		return theComponent.getType();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getHandlerFor(org.eclipse.stp.core.sca.Component)
	 */
	public IComponentHandler getHandlerFor(Component theComponent)
	{
		if (theComponent == null) {
			return null;
		}
		
		String componentKind = getComponentKindOf(theComponent);
		return (IComponentHandler)componentHandlers.get(componentKind);

	}


	// =====================
	// Create Implementation
	// =====================

	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getExternalServiceKindsFor(org.eclipse.stp.core.saf.Context, java.lang.Object)
	 */
	public IExternalServiceHandler[] getExternalServiceHandlersFor(Context context, Object service)
	{
		
		ArrayList result = new ArrayList();
		Collection handlers = getFilteredKinds(context, externalServiceHandlers, externalServiceKindFilters);
		for (Iterator iter = handlers.iterator(); iter.hasNext();) {
			IExternalServiceHandler handler = (IExternalServiceHandler)iter.next();
			try {
				if (handler.canCreateExternalServiceFor(service)) {
					result.add(handler);
				}
			}
			catch(Exception exc) {
				SAFActivator.log(this, "getImportTypesFor", IStatus.WARNING, exc); //$NON-NLS-1$
			}
		}
		return (IExternalServiceHandler[])result.toArray(new IExternalServiceHandler[result.size()]);

	}
	
	/**
	 * Returns the QName string for the binding kind of the specified import.
	 * @param theExternalService ExternalService being queried
	 * @return Type string for the binding in the import object
	 */
	private List getExternalServiceKindOf(ExternalService theExternalService) {

		List kinds = new ArrayList();
		if (theExternalService != null
				&& theExternalService.getBindings() != null
				&& !theExternalService.getBindings().isEmpty()) {

			List bindings = theExternalService.getBindings();

			for (Iterator iter = bindings.iterator(); iter.hasNext();) {
				Binding binding = (Binding) iter.next();
				kinds.add(SAFManagerUtils.getSubstitutionGroupName(binding));
			}
		}
		return kinds;

	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getHandlersFor(org.eclipse.stp.core.sca.ExternalService)
	 */
	public IExternalServiceHandler[] getHandlersFor(ExternalService theExternalService)
	{
		if (theExternalService == null) {
			return null;
		}
		
		List descriptors = new ArrayList();
		List bindingTypes = getExternalServiceKindOf(theExternalService);
		for (Iterator iter = bindingTypes.iterator(); iter.hasNext();) {
			String bindingType = (String) iter.next();
			IExternalServiceHandler descriptor = (IExternalServiceHandler)externalServiceHandlers.get(bindingType);
			if(descriptors != null && !descriptors.contains(descriptor))
				descriptors.add(descriptor);
		}

		return (IExternalServiceHandler[]) descriptors.toArray(new IExternalServiceHandler[descriptors.size()]);

	}


	// =============
	// Create export
	// =============

	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getEntryPointKindsFor(org.eclipse.stp.core.saf.Context, java.lang.Object)
	 */
	public IEntryPointHandler[] getEntryPointHandlersFor(Context context, Object service)
	{
		ArrayList result = new ArrayList();
		Collection handlers = getFilteredKinds(context, entryPointHandlers, entryPointKindFilters);
		for (Iterator iter = handlers.iterator(); iter.hasNext();) {
			IEntryPointHandler handler = (IEntryPointHandler)iter.next();
			try {
				if (handler.canCreateEntryPointFor(service)) {
					result.add(handler);
				}
			}
			catch(Exception exc) {
				SAFActivator.log(this, "getEntryPointKindsFor", IStatus.WARNING, exc); //$NON-NLS-1$
			}
			
		}
		return (IEntryPointHandler[])result.toArray(new IEntryPointHandler[result.size()]);
	}
	
	/**
	 * Returns the QName strings for the binding kinds of the specified entry point.
	 * @param theEntryPoint Export being queried
	 * @return Kind strings for the bindings in the entry point object
	 */
	private List getEntryPointKindOf(EntryPoint theEntryPoint) {
		List kinds = new ArrayList();
		if (theEntryPoint != null && theEntryPoint.getBindings() != null
				&& !theEntryPoint.getBindings().isEmpty()) {

			List bindings = theEntryPoint.getBindings();

			for (Iterator iter = bindings.iterator(); iter.hasNext();) {
				Binding binding = (Binding) iter.next();
				kinds.add(SAFManagerUtils.getSubstitutionGroupName(binding));
			}
		}
		return kinds;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getHandlersFor(org.eclipse.stp.core.sca.EntryPoint)
	 */
	public IEntryPointHandler[] getHandlersFor(EntryPoint theEntryPoint)
	{
		if (theEntryPoint == null) {
			return null;
		}
		
		List descriptors = new ArrayList();
		List bindingTypes = getEntryPointKindOf(theEntryPoint);
		for (Iterator iter = bindingTypes.iterator(); iter.hasNext();) {
			String bindingType = (String) iter.next();
			IEntryPointHandler handler = (IEntryPointHandler)entryPointHandlers.get(bindingType);
			if(handler != null && !descriptors.contains(handler))
				descriptors.add(handler);
		}

		return (IEntryPointHandler[]) descriptors.toArray(new IEntryPointHandler[descriptors.size()]);
	}



	// ===========
	// Auto-wiring
	// ===========

	/**
	 * Returns the QName string for the kind of the specified interface.
	 * 
	 * @param theComponent
	 *            Component being queried
	 * @return Component kind string
	 */
	private String getInterfaceKindOf(Interface theInterface)
	{
		return SAFManagerUtils.getSubstitutionGroupName(theInterface);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getHandlerFor(org.eclipse.stp.core.sca.Interface)
	 */
	public IInterfaceHandler getHandlerFor(Interface theInterface) {
		
		if (theInterface == null) {
			return null;
		}
		
		String interfaceKind = getInterfaceKindOf(theInterface);
		return (IInterfaceHandler)interfaceHandlers.get(interfaceKind);

	}


	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.ISAFManager#getInterfaceHandlersFor(java.lang.Object)
	 */
	public IInterfaceHandler[] getInterfaceHandlersFor(Object interfaceObj) {
		ArrayList result = new ArrayList();
		Collection handlers = getFilteredKinds(null, interfaceHandlers, null);
		for (Iterator iter = handlers.iterator(); iter.hasNext();) {
			IInterfaceHandler handler = (IInterfaceHandler) iter.next();
			try {
				if (handler.canCreateInterfaceFor(interfaceObj)) {
					result.add(handler);
				}
			} catch (Exception exc) {
				SAFActivator.log(this, "getInterfaceKindsFor", IStatus.WARNING, exc); //$NON-NLS-1$
			}
		}
		return (IInterfaceHandler[]) result.toArray(new IInterfaceHandler[result.size()]);
	}
	
}

