/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.ui.wizards;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.datatools.connectivity.internal.ui.wizards.BaseWizard;
import org.eclipse.datatools.connectivity.internal.ui.wizards.ISummaryDataSource;
import org.eclipse.datatools.connectivity.internal.ui.wizards.SummaryWizardPage;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.stp.soas.deploy.core.DeploymentDefinitionBuilder;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.operations.CreateDeployFileOperation;
import org.eclipse.stp.soas.deploy.core.ui.actions.ExecuteDeployAction;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployFilePackage;
import org.eclipse.stp.soas.deploy.models.deployfile.Root;
import org.eclipse.stp.soas.internal.deploy.ui.wizards.WizardNewFileCreationPage;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWizard;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.ide.IDE;

/**
 * This is a sample new wizard. Its role is to create a new file resource in the
 * provided container. If the container resource (a folder or a project) is
 * selected in the workspace when the wizard is opened, it will accept it as the
 * target container. The wizard creates one file with the extension "deploy". If
 * a sample multi-page editor (also available as a template) is registered for
 * the same extension, it will be able to open it.
 */

public class DeployFileNewWizard extends BaseWizard implements INewWizard,
		ISummaryDataSource {

	public final static String DEPLOY_FILE_EXTENSION = "deploy"; //$NON-NLS-1$

	private final static ImageDescriptor NEW_DEPLOY_FILE_DEFAULT_PAGE_IMAGE;
	private final static String NEW_DEPLOY_FILE_WIZARD_TITLE;

	private final static String FILE_PAGE_NAME;
	private final static String FILE_PAGE_TITLE;
	private final static ImageDescriptor FILE_PAGE_IMAGE;
	private final static String FILE_PAGE_DESCRIPTION;

	private final static String DESCRIPTION_PAGE_NAME;
	private final static String DESCRIPTION_PAGE_TITLE;
	private final static ImageDescriptor DESCRIPTION_PAGE_IMAGE;
	private final static String DESCRIPTION_PAGE_DESCRIPTION;

	private final static String PACKAGES_PAGE_NAME;
	private final static String PACKAGES_PAGE_TITLE;
	private final static ImageDescriptor PACKAGES_PAGE_IMAGE;
	private final static String PACKAGES_PAGE_DESCRIPTION;

	private final static String TARGETS_PAGE_NAME;
	private final static String TARGETS_PAGE_TITLE;
	private final static ImageDescriptor TARGETS_PAGE_IMAGE;
	private final static String TARGETS_PAGE_DESCRIPTION;

	private WizardNewFileCreationPage mFilePage;
	private DeployFileDescriptionWizardPage mDescriptionPage;
	private PackageSelectionWizardPage mPackagesPage;
	private PackageTargetWizardPage mTargetsPage;
	private SummaryWizardPage mSummaryPage;
	private ISelection mSelection;
	private String mInitialFileName = new String();
	private IContainer mInitialContainer;
	private Root mRoot;

	/**
	 * Constructor for DeployFileNewWizard.
	 */
	public DeployFileNewWizard() {
		super();

		setHelpAvailable(false);

		setDefaultPageImageDescriptor(NEW_DEPLOY_FILE_DEFAULT_PAGE_IMAGE);

		setDialogSettings(null);

		setNeedsProgressMonitor(true);

		setWindowTitle(NEW_DEPLOY_FILE_WIZARD_TITLE);

		setDialogSettings(null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.wizard.IWizard#getNextPage(org.eclipse.jface.wizard.IWizardPage)
	 */
	public IWizardPage getNextPage(IWizardPage page) {
		if (page == mPackagesPage) {
			if (mPackagesPage.getSelection().length == 0) {
				page = mTargetsPage;
			}
		}
		else if (page == mFilePage) {
			IFile file = mFilePage.getOutputFile();
			if (file != null) {
				mRoot.eResource().setURI(
						URI.createPlatformResourceURI(file.getFullPath()
								.toString()));
			}
		}
		return super.getNextPage(page);
	}

	/**
	 * We will accept the selection in the workbench to see if we can initialize
	 * from it.
	 * 
	 * @see IWorkbenchWizard#init(IWorkbench, IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		mSelection = selection;

		Object object = selection.getFirstElement();
		if (object != null) {
			IResource resource;
			if (object instanceof IResource) {
				resource = (IResource) object;
			}
			else if (object instanceof IAdaptable) {
				resource = (IResource) ((IAdaptable) object)
						.getAdapter(IResource.class);
			}
			else {
				resource = null;
			}
			if (resource != null) {
				IProject project = resource.getProject();
				mInitialContainer = resource.getType() == IResource.PROJECT ? (IProject) resource
						: resource.getParent();
				mInitialFileName = project.getName();
				if (selection.size() == 1) {
					if (resource.getType() == IResource.FOLDER) {
						mInitialContainer = (IFolder) resource;
						mInitialFileName = resource.getName();
					}
					else if (resource.getType() == IResource.FILE) {
						mInitialFileName = resource.getFullPath()
								.removeFileExtension().lastSegment();
					}
				}
			}
		}
		ResourceSet rs = new ResourceSetImpl();
		Resource dummyResource = rs.createResource(URI
				.createPlatformResourceURI("/dummy/dummy.deploy")); //$NON-NLS-1$
		mRoot = DeployFilePackage.eINSTANCE.getDeployFileFactory().createRoot();
		dummyResource.getContents().add(mRoot);
	}

	public IFile[] getSelectedPackages() {
		return mPackagesPage.getSelection();
	}

	public IFile getOutputFile() {
		return mFilePage.getOutputFile();
	}

	/**
	 * Adding the page to the wizard.
	 */
	public void addPages() {
		mFilePage = new WizardNewFileCreationPage(FILE_PAGE_NAME,
				FILE_PAGE_TITLE, FILE_PAGE_IMAGE);
		mFilePage.setDescription(FILE_PAGE_DESCRIPTION);
		mFilePage.init(DEPLOY_FILE_EXTENSION,
				mInitialContainer == null ? StructuredSelection.EMPTY
						: new StructuredSelection(mInitialContainer));
		mFilePage.setFileName(mInitialFileName);
		addPage(mFilePage);

		mDescriptionPage = new DeployFileDescriptionWizardPage(
				DESCRIPTION_PAGE_NAME, DESCRIPTION_PAGE_TITLE,
				DESCRIPTION_PAGE_IMAGE);
		mDescriptionPage.setDescription(DESCRIPTION_PAGE_DESCRIPTION);
		mDescriptionPage.init(mRoot);
		addPage(mDescriptionPage);

		mPackagesPage = new PackageSelectionWizardPage(PACKAGES_PAGE_NAME,
				PACKAGES_PAGE_TITLE, PACKAGES_PAGE_IMAGE);
		mPackagesPage.setDescription(PACKAGES_PAGE_DESCRIPTION);
		mPackagesPage.init(mRoot, mSelection);
		addPage(mPackagesPage);

		mTargetsPage = new PackageTargetWizardPage(TARGETS_PAGE_NAME,
				TARGETS_PAGE_TITLE, TARGETS_PAGE_IMAGE);
		mTargetsPage.setDescription(TARGETS_PAGE_DESCRIPTION);
		mTargetsPage.init(mRoot);
		addPage(mTargetsPage);

		mSummaryPage = new SummaryWizardPage(this);
		addPage(mSummaryPage);
	}

	/**
	 * This method is called when 'Finish' button is pressed in the wizard. We
	 * will create an operation and run it using wizard as execution context.
	 */
	public boolean performFinish() {
		final IFile file = getOutputFile();
		final String name = mDescriptionPage.getDeployName();
		final String description = mDescriptionPage.getDeployDescription();
		final boolean execute = mTargetsPage.getExecuteDeployment();
		IRunnableWithProgress op = new WorkspaceModifyOperation() {

			public void execute(IProgressMonitor monitor)
					throws InvocationTargetException, InterruptedException {
				installBuilder(monitor);

				monitor.beginTask(DeployCorePlugin.getDefault()
						.getResourceString("TASK.CreatingFile", //$NON-NLS-1$
								new Object[] { file.getName()}), 2);
				mRoot.setName(name);
				mRoot.setDescription(description);
				new CreateDeployFileOperation(file, mRoot).run(monitor);
				if (!execute) {
					monitor.worked(1);
					monitor.setTaskName(DeployCorePlugin.getDefault()
							.getResourceString("TASK.OpeningFile")); //$NON-NLS-1$
					getShell().getDisplay().asyncExec(new Runnable() {

						public void run() {
							IWorkbenchPage page = PlatformUI.getWorkbench()
									.getActiveWorkbenchWindow().getActivePage();
							try {
								IDE.openEditor(page, file, true);
							}
							catch (PartInitException e) {
								e.printStackTrace();
							}
						}
					});
					monitor.worked(1);
				}
				monitor.done();
			}
		};
		try {
			getContainer().run(false, true, op);
		}
		catch (InterruptedException e) {
			return false;
		}
		catch (InvocationTargetException e) {
			Throwable realException = e.getTargetException();
			MessageDialog.openError(getShell(), DeployCorePlugin
					.getDefault().getResourceString("DLGTITLE.Error"), //$NON-NLS-1$
					realException.getMessage());
			return false;
		}
		if (execute) {
			ExecuteDeployAction action = new ExecuteDeployAction(new String(),
					getShell());
			action.selectionChanged(new StructuredSelection(mRoot));
			action.run();
		}
		return true;
	}

	private void installBuilder(IProgressMonitor monitor) {
		try {
			boolean foundBuilder = false;
			IProject project = getOutputFile().getProject();
			IProjectDescription projDesc = project.getDescription();
			List builders = new ArrayList(Arrays
					.asList(projDesc.getBuildSpec()));
			for (Iterator it = builders.iterator(); !foundBuilder
					&& it.hasNext();) {
				foundBuilder = DeploymentDefinitionBuilder.BUILDER_ID
						.equals(((ICommand) it.next()).getBuilderName());
			}
			if (!foundBuilder) {
				ICommand packageBuilder = projDesc.newCommand();
				packageBuilder
						.setBuilderName(DeploymentDefinitionBuilder.BUILDER_ID);
				builders.add(packageBuilder);
				projDesc.setBuildSpec((ICommand[]) builders
						.toArray(new ICommand[builders.size()]));
				project.setDescription(projDesc, monitor);
			}
		}
		catch (CoreException e) {
			e.printStackTrace();
		}
	}

	static {
		NEW_DEPLOY_FILE_DEFAULT_PAGE_IMAGE = ImageDescriptor
				.createFromURL(DeployCorePlugin.getDefault().getBundle()
						.getEntry("icons/new_wiz.gif")); //$NON-NLS-1$
		NEW_DEPLOY_FILE_WIZARD_TITLE = DeployCorePlugin.getDefault()
				.getResourceString("DeployFileNewWizard.DLGTITLE"); //$NON-NLS-1$

		FILE_PAGE_NAME = "NewFilePage.name"; //$NON-NLS-1$
		FILE_PAGE_TITLE = DeployCorePlugin.getDefault().getResourceString(
				"DeployFileNewWizard.PAGETITLE.FilePage"); //$NON-NLS-1$
		FILE_PAGE_IMAGE = NEW_DEPLOY_FILE_DEFAULT_PAGE_IMAGE;
		FILE_PAGE_DESCRIPTION = DeployCorePlugin.getDefault()
				.getResourceString("DeployFileNewWizard.PAGEDESC.FilePage"); //$NON-NLS-1$

		DESCRIPTION_PAGE_NAME = "DescriptionPage.name"; //$NON-NLS-1$
		DESCRIPTION_PAGE_TITLE = DeployCorePlugin.getDefault()
				.getResourceString(
						"DeployFileNewWizard.PAGETITLE.DescriptionPage"); //$NON-NLS-1$
		DESCRIPTION_PAGE_IMAGE = NEW_DEPLOY_FILE_DEFAULT_PAGE_IMAGE;
		DESCRIPTION_PAGE_DESCRIPTION = DeployCorePlugin.getDefault()
				.getResourceString(
						"DeployFileNewWizard.PAGEDESC.DescriptionPage"); //$NON-NLS-1$

		PACKAGES_PAGE_NAME = "PackagesPage.name"; //$NON-NLS-1$
		PACKAGES_PAGE_TITLE = DeployCorePlugin
				.getDefault()
				.getResourceString("DeployFileNewWizard.PAGETITLE.PackagesPage"); //$NON-NLS-1$
		PACKAGES_PAGE_IMAGE = NEW_DEPLOY_FILE_DEFAULT_PAGE_IMAGE;
		PACKAGES_PAGE_DESCRIPTION = DeployCorePlugin.getDefault()
				.getResourceString("DeployFileNewWizard.PAGEDESC.PackagesPage"); //$NON-NLS-1$

		TARGETS_PAGE_NAME = "TargetsPage.name"; //$NON-NLS-1$
		TARGETS_PAGE_TITLE = DeployCorePlugin.getDefault()
				.getResourceString("DeployFileNewWizard.PAGETITLE.TargetsPage"); //$NON-NLS-1$
		TARGETS_PAGE_IMAGE = NEW_DEPLOY_FILE_DEFAULT_PAGE_IMAGE;
		TARGETS_PAGE_DESCRIPTION = DeployCorePlugin.getDefault()
				.getResourceString("DeployFileNewWizard.PAGEDESC.TargetsPage"); //$NON-NLS-1$
	}

}