/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.deploy.core;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.util.Assert;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployConfiguration;
import org.eclipse.stp.soas.internal.deploy.core.ConfigurablePackageExtension;
import org.eclipse.stp.soas.internal.deploy.core.DeployDriverExtension;
import org.eclipse.stp.soas.internal.deploy.core.LogicalPackageExtension;
import org.eclipse.stp.soas.internal.deploy.core.PackageExtension;
import org.eclipse.stp.soas.internal.deploy.core.PackageExtensionException;
import org.eclipse.stp.soas.internal.deploy.core.PhysicalPackageExtension;
import org.eclipse.stp.soas.internal.deploy.core.ServerDefinition;
import org.eclipse.stp.soas.internal.deploy.core.TechnologyDefinition;
import org.eclipse.stp.soas.internal.deploy.core.TechnologyMap;
import org.eclipse.stp.soas.internal.deploy.ui.properties.DeployOutputFolderPropertyPage;

/**
 * @author rcernich
 * 
 * Created on Mar 18, 2004
 */
public class DeploymentExtensionManager {

	public static final QualifiedName sPackageExtensionPropertyKey = new QualifiedName(
			DeployCorePlugin.getDefault().getBundle().getSymbolicName(),
			"packageExtensionID"); //$NON-NLS-1$

	public static final String EXTENSION_ID = "deploymentExtension"; //$NON-NLS-1$

	public static final String EXT_ELEM_LOGICAL_PACKAGE = "logicalPackage"; //$NON-NLS-1$
	public static final String EXT_ELEM_CONFIGURABLE_PACKAGE = "configurablePackage"; //$NON-NLS-1$
	public static final String EXT_ELEM_PHYSICAL_PACKAGE = "physicalPackage"; //$NON-NLS-1$
	public static final String EXT_ELEM_DEPLOY_DRIVER = "deployDriver"; //$NON-NLS-1$
	public static final String EXT_ELEM_SUPPORTED_SERVER = "supportedServer"; //$NON-NLS-1$
	public static final String EXT_ELEM_SUPPORTED_TECHNOLOGY = "supportedTechnology"; //$NON-NLS-1$
	public static final String EXT_ELEM_TECHNOLOGY_DEFINITION = "technologyDefinition"; //$NON-NLS-1$
	public static final String EXT_ELEM_SERVER_DEFINITION = "serverDefinition"; //$NON-NLS-1$
	public static final String EXT_ELEM_TECHNOLOGY_MAP = "technologyMap"; //$NON-NLS-1$

	public static boolean DEBUG_DEPLOYMENT_EXTENSION = false;

	private static final String OPTION_DEBUG_DEPLOYMENT_EXTENSION = "org.eclipse.stp.soas.deploy.core/deploymentextension"; //$NON-NLS-1$
	private static DeploymentExtensionManager sInstance;

	private Map mTechnologyDefsByID;
	private Map mServerDefsByID;
	private Map mIdToLogicalExtension;
	private SortedSet mLogicalExtensionByFileExtension;
	private Map mIdToConfigurableExtension;
	private SortedSet mConfigurableExtensionByFileExtension;
	private Map mIdToPhysicalExtension;
	private Map mPhysicalExtensionByFileExtension;
	private Map mIdToDeployDriver;
	private Map mProfileIdToDeployDriverList;

	public static DeploymentExtensionManager getInstance() {
		if (sInstance == null) {
			String debug = Platform
					.getDebugOption(OPTION_DEBUG_DEPLOYMENT_EXTENSION);
			DEBUG_DEPLOYMENT_EXTENSION = debug == null ? false : (debug
					.equalsIgnoreCase("true") ? true : false); //$NON-NLS-1$
			sInstance = new DeploymentExtensionManager();
			sInstance.init();
		}
		return sInstance;
	}

	private DeploymentExtensionManager() {
		super();
	}

	public boolean isPackage(IFile file) {
		return getPackageExtension(file) != null;
	}

	public boolean isLogicalPackage(IFile file) {
		return getLogicalPackageExtension(file) != null;
	}

	public boolean isConfigurablePackage(IFile file) {
		return getConfigurablePackageExtension(file) != null;
	}

	public boolean isPhysicalPackage(IFile file) {
		return getPhysicalPackageExtension(file) != null;
	}

	public IPackage getPackage(IFile file) {
		IPackage pkg;
		IPackageExtension ipe = getPackageExtension(file);
		if (ipe == null) {
			pkg = null;
		}
		else if (ipe instanceof IConfigurablePackageExtension) {
			pkg = ((IConfigurablePackageExtension) ipe).adaptFile(file);
		}
		else if (ipe instanceof ILogicalPackageExtension) {
			pkg = ((ILogicalPackageExtension) ipe).adaptFile(file);
		}
		else if (ipe instanceof IPhysicalPackageExtension) {
			pkg = ((IPhysicalPackageExtension) ipe).adaptFile(file);
		}
		else {
			if (DeployCorePlugin.getDefault().isDebugging()) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.unknownPackageExtension", //$NON-NLS-1$
										new Object[] { ipe.getClass()}));
			}
			pkg = null;
		}
		return pkg;
	}

	public IPhysicalPackage getPhysicalPackage(IFile file) {
		IPackage pkg = getPackage(file);
		if (pkg != null && pkg instanceof IPhysicalPackage) {
			return (IPhysicalPackage) pkg;
		}
		return null;
	}

	/**
	 * Returns the output folder for a specified logical package and deployment
	 * configuration. If the configuration of the package has not been
	 * overridden, the default location for the package output is returned. If
	 * the configuration of the package has been overridden, the output location
	 * is specific to the deployment configuration. For example:
	 * <p>
	 * No package configuration: /&lt;package project&gt;/&lt;deployment
	 * output&gt;/&lt;project relative path to package&gt;/&lt;deploy driver
	 * name&gt;
	 * </p>
	 * <p>
	 * With package configuration: /&lt;package project&gt;/&lt;deployment
	 * output&gt;/&lt;project relative path to deploy file>/&lt;server name&gt;
	 * </p>
	 * 
	 * @param pkg
	 * @param deployConfig
	 * @return
	 */
	public IFolder getOutputFolder(ILogicalPackage pkg,
			DeployConfiguration deployConfig) {
		IPath outputFolderPath;
		IWorkspaceRoot wsRoot = pkg.getFile().getWorkspace().getRoot();
		IDeployTarget target = Utilities.adaptToIDeployTarget(deployConfig);
		IPackageConfiguration pkgConfig = getPackageConfiguration(pkg,
				deployConfig);
		if (pkgConfig == null) {
			IFile pkgFile = pkg.getFile();
			IPath pkgFilePath = pkgFile.getProjectRelativePath();
			outputFolderPath = new Path(DeployOutputFolderPropertyPage
					.getOutputContainerPath(pkgFile.getProject()));
			outputFolderPath = outputFolderPath.append(pkgFilePath);
			outputFolderPath = outputFolderPath.append(target.getDeployDriver()
					.getName());
		}
		else {
			// If a configuration override is associated with the package,
			// the deployment profile must exist in the workspace.
			IFile deployFile = wsRoot.getFile(new Path(deployConfig.eResource()
					.getURI().path()).removeFirstSegments(1));
			outputFolderPath = new Path(DeployOutputFolderPropertyPage
					.getOutputContainerPath(deployFile.getProject()));
			outputFolderPath = outputFolderPath.append(deployFile
					.getProjectRelativePath());
			outputFolderPath = outputFolderPath.append(target
					.getConnectionProfile().getName());
		}
		return wsRoot.getFolder(outputFolderPath);
	}

	public IFolder getOutputFolder(ILogicalPackage pkg,
			IDeployDriverExtension deployDriver) {
		IFile pkgFile = pkg.getFile();
		IPath pkgFilePath = pkgFile.getProjectRelativePath();

		IPath outputFolderPath = new Path(DeployOutputFolderPropertyPage
				.getOutputContainerPath(pkgFile.getProject()));
		outputFolderPath = outputFolderPath.append(pkgFilePath);
		outputFolderPath = outputFolderPath.append(deployDriver.getName());
		
		return pkgFile.getWorkspace().getRoot().getFolder(outputFolderPath);
	}

	private IPackageConfiguration getPackageConfiguration(ILogicalPackage ilp,
			DeployConfiguration deployConfig) {
		IPackageConfiguration retVal;
		byte[] configData = deployConfig.getConfigOverride();
		if (ilp instanceof IConfigurablePackage) {
			if (configData == null || configData.length == 0) {
				retVal = null;
			}
			else {
				ByteArrayInputStream bais = new ByteArrayInputStream(configData);
				try {
					retVal = ((IConfigurablePackageExtension) ilp
							.getExtension()).getPackageConfigurationManager()
							.createPackageConfiguration(ilp, bais);
				}
				catch (IOException e) {
					e.printStackTrace();
					retVal = ((IConfigurablePackageExtension) ilp
							.getExtension()).getPackageConfigurationManager()
							.createPackageConfiguration(ilp);
				}
				finally {
					try {
						bais.close();
					}
					catch (IOException e) {
					}
				}
			}
		}
		else {
			if (configData != null && configData.length == 0) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"CreateDeployPackagesJob.error.configurationSpecifiedForLogicalPackage")); //$NON-NLS-1$
			}
			retVal = null;
		}
		return retVal;
	}

	public IConfigurablePackage getConfigurablePackage(IFile file) {
		IPackage pkg = getPackage(file);
		if (pkg != null && pkg instanceof IConfigurablePackage) {
			return (IConfigurablePackage) pkg;
		}
		return null;
	}

	public ILogicalPackage getLogicalPackage(IFile file) {
		IPackage pkg = getPackage(file);
		if (pkg != null && pkg instanceof ILogicalPackage) {
			return (ILogicalPackage) pkg;
		}
		return null;
	}

	public IPackageExtension getPackageExtension(IFile file) {
		IPackageExtension ipe = getConfigurablePackageExtension(file);
		if (ipe == null) {
			ipe = getLogicalPackageExtension(file);
			if (ipe == null) {
				ipe = getPhysicalPackageExtension(file);
			}
		}
		return ipe;
	}

	public ILogicalPackageExtension getLogicalPackageExtension(IFile file) {
		ILogicalPackageExtension ilpe = null;
		String extensionID = null;
		try {
			extensionID = file
					.getPersistentProperty(sPackageExtensionPropertyKey);
			if (extensionID != null) {
				ilpe = getLogicalPackageExtension(extensionID);
			}
		}
		catch (CoreException e) {
			// This is just a cached value. Should be safe to ignore.
		}
		if (ilpe == null) {
			String fileExt = file.getFileExtension();
			if (fileExt != null) {
				List extensions = getLogicalExtensions(fileExt);
				for (Iterator it = extensions.iterator(); ilpe == null
						&& it.hasNext();) {
					ILogicalPackageExtension extension = (ILogicalPackageExtension) it
							.next();
					if (extension.supportsFile(file)) {
						ilpe = extension;
					}
				}
			}
			if (ilpe != null && extensionID == null) {
				try {
					// the extensionID may represent a physical package
					// extension only set the property if extensionID is not
					// set.
					file.setPersistentProperty(sPackageExtensionPropertyKey,
							ilpe.getID());
				}
				catch (CoreException e) {
					// This is only for caching, so it should be safe to
					// ignore.
				}
			}
		}
		return ilpe;
	}

	public IConfigurablePackageExtension getConfigurablePackageExtension(
			IFile file) {
		IConfigurablePackageExtension icpe = null;
		String extensionID = null;
		try {
			extensionID = file
					.getPersistentProperty(sPackageExtensionPropertyKey);
			if (extensionID != null) {
				icpe = getConfigurablePackageExtension(extensionID);
			}
		}
		catch (CoreException e) {
			// This is just a cached value. Should be safe to ignore.
		}
		if (icpe == null) {
			String fileExt = file.getFileExtension();
			if (fileExt != null) {
				List extensions = getConfigurableExtensions(fileExt);
				for (Iterator it = extensions.iterator(); icpe == null
						&& it.hasNext();) {
					IConfigurablePackageExtension extension = (IConfigurablePackageExtension) it
							.next();
					if (extension.supportsFile(file)) {
						icpe = extension;
					}
				}
			}
			if (icpe != null && extensionID == null) {
				try {
					// the extensionID may represent a physical package
					// extension only set the property if extensionID is not
					// set.
					file.setPersistentProperty(sPackageExtensionPropertyKey,
							icpe.getID());
				}
				catch (CoreException e) {
					// This is only for caching, so it should be safe to
					// ignore.
				}
			}
		}
		return icpe;
	}

	public IPhysicalPackageExtension getPhysicalPackageExtension(IFile file) {
		IPhysicalPackageExtension ippe = null;
		String extensionID = null;
		try {
			extensionID = file
					.getPersistentProperty(sPackageExtensionPropertyKey);
			if (extensionID != null) {
				ippe = getPhysicalPackageExtension(extensionID);
			}
		}
		catch (CoreException e) {
			// This is just a cached value. Should be safe to ignore.
		}
		if (ippe == null) {
			String fileExt = file.getFileExtension();
			List extensions = getPhysicalPackageExtensions(fileExt);
			for (Iterator it = extensions.iterator(); ippe == null
					&& it.hasNext();) {
				IPhysicalPackageExtension extension = (IPhysicalPackageExtension) it
						.next();
				if (extension.supportsFile(file)) {
					ippe = extension;
				}
			}
			if (ippe != null) {
				try {
					file.setPersistentProperty(sPackageExtensionPropertyKey,
							ippe.getID());
				}
				catch (CoreException e) {
					// This is only for caching, so it should be safe to
					// ignore.
				}
			}
		}
		return ippe;
	}

	public ILogicalPackageExtension getLogicalPackageExtension(String id) {
		ILogicalPackageExtension ilpe;
		if (id == null || !mIdToLogicalExtension.containsKey(id)) {
			ilpe = null;
		}
		else {
			ilpe = (ILogicalPackageExtension) mIdToLogicalExtension.get(id);
		}
		return ilpe;
	}

	public List getLogicalExtensions(String fileExtension) {
		return new ArrayList(mLogicalExtensionByFileExtension.subSet(
				fileExtension, fileExtension + '\0'));
	}

	public IConfigurablePackageExtension getConfigurablePackageExtension(
			String id) {
		IConfigurablePackageExtension icpe;
		if (id == null || !mIdToLogicalExtension.containsKey(id)) {
			icpe = null;
		}
		else {
			icpe = (IConfigurablePackageExtension) mIdToConfigurableExtension
					.get(id);
		}
		return icpe;
	}

	public List getConfigurableExtensions(String fileExtension) {
		return new ArrayList(mConfigurableExtensionByFileExtension.subSet(
				fileExtension, fileExtension + '\0'));
	}

	public IPhysicalPackageExtension getPhysicalPackageExtension(String id) {
		IPhysicalPackageExtension ippe;
		if (id == null || !mIdToPhysicalExtension.containsKey(id)) {
			ippe = null;
		}
		else {
			ippe = (IPhysicalPackageExtension) mIdToPhysicalExtension.get(id);
		}
		return ippe;
	}

	public List getPhysicalPackageExtensions(String fileExtension) {
		List retVal;
		if (fileExtension == null) {
			retVal = new ArrayList();
		}
		else {
            if(mPhysicalExtensionByFileExtension.containsKey(fileExtension)){
                retVal = (List)mPhysicalExtensionByFileExtension.get(fileExtension);
            }
            else{
                retVal = new ArrayList();
            }
		}
		return retVal;
	}

	public ITechnologyDefinition getTechnologyDefinition(String type) {
		ITechnologyDefinition itd;
		if (type == null || !mTechnologyDefsByID.containsKey(type)) {
			itd = null;
		}
		else {
			itd = (ITechnologyDefinition) mTechnologyDefsByID.get(type);
		}
		return itd;
	}

	public IServerDefinition getServerDefinition(String type) {
		IServerDefinition isd;
		if (type == null || !mServerDefsByID.containsKey(type)) {
			isd = null;
		}
		else {
			isd = (IServerDefinition) mServerDefsByID.get(type);
		}
		return isd;
	}

	public List getDeployDrivers() {
		return new ArrayList(mIdToDeployDriver.values());
	}

	public List getDeployDrivers(String profileID) {
		List drivers;
		if (profileID == null
				|| !mProfileIdToDeployDriverList.containsKey(profileID)) {
			drivers = new ArrayList();
		}
		else {
			drivers = new ArrayList((List) mProfileIdToDeployDriverList
					.get(profileID));
		}
		return drivers;
	}

	public IDeployDriverExtension getDeployDriver(IPackage pkg, String profileID) {
		IDeployDriverExtension idde = null;
		if (pkg != null) {
			if (pkg instanceof ILogicalPackage) {
				idde = getDeployDriver((ILogicalPackage) pkg, profileID);
			}
			else if (pkg instanceof IPhysicalPackage) {
				idde = getDeployDriver((IPhysicalPackage) pkg, profileID);
			}
			else {
				throw new IllegalArgumentException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.unknownPackageType", //$NON-NLS-1$
										new Object[] { pkg.getClass()}));
			}
		}
		return idde;
	}

	public IDeployDriverExtension getDeployDriver(IPhysicalPackage pkg,
			String profileID) {
		IDeployDriverExtension idde = null;
		IServerType serverType = pkg.getServerType();
		for (Iterator it = getDeployDrivers(profileID).iterator(); idde == null
				&& it.hasNext();) {
			IDeployDriverExtension tmp = (IDeployDriverExtension) it.next();
			if (tmp.supportsServer(serverType)) {
				idde = tmp;
			}
		}
		return idde;
	}

	public IDeployDriverExtension getDeployDriver(ILogicalPackage pkg,
			String profileID) {
		IDeployDriverExtension idde = null;
		ITechnologyType technologyType = pkg.getTechnologyType();
		for (Iterator it = getDeployDrivers(profileID).iterator(); idde == null
				&& it.hasNext();) {
			IDeployDriverExtension tmp = (IDeployDriverExtension) it.next();
			IPackageConstructorExtension constructor = tmp
					.getPackageConstructor(technologyType);
			if (constructor != null && constructor.supportsPackage(pkg)) {
				idde = tmp;
			}
		}
		return idde;
	}

	private void init() {
		mTechnologyDefsByID = new TreeMap();
		mServerDefsByID = new TreeMap();
		mIdToLogicalExtension = new TreeMap(new IDComparator());
		mLogicalExtensionByFileExtension = new TreeSet(
				new FileExtensionComparator());
		mIdToConfigurableExtension = new TreeMap(new IDComparator());
		mConfigurableExtensionByFileExtension = new TreeSet(
				new FileExtensionComparator());
		mIdToPhysicalExtension = new TreeMap(new IDComparator());
		mPhysicalExtensionByFileExtension = new HashMap();
		mIdToDeployDriver = new TreeMap();
		mProfileIdToDeployDriverList = new TreeMap();
		processExtensions();
	}

	private void processExtensions() {
		IExtensionPoint exp = Platform.getExtensionRegistry()
				.getExtensionPoint(
						DeployCorePlugin.getDefault().getBundle()
								.getSymbolicName(), EXTENSION_ID);
		IExtension[] exts = exp.getExtensions();
		Collection logicalPackages = new ArrayList();
		Collection configurablePackages = new ArrayList();
		Collection physicalPackages = new ArrayList();
		Collection deployDrivers = new ArrayList();
		Collection technologyDefs = new ArrayList();
		Collection serverDefs = new ArrayList();
		Collection technologyMaps = new ArrayList();
		for (Iterator xit = Arrays.asList(exts).iterator(); xit.hasNext();) {
			IExtension ext = (IExtension) xit.next();
			IConfigurationElement[] elems = ext.getConfigurationElements();
			for (Iterator eit = Arrays.asList(elems).iterator(); eit.hasNext();) {
				IConfigurationElement elem = (IConfigurationElement) eit.next();
				String elemName = elem.getName();
				if (EXT_ELEM_LOGICAL_PACKAGE.equals(elemName)) {
					logicalPackages.add(elem);
				}
				else if (EXT_ELEM_CONFIGURABLE_PACKAGE.equals(elemName)) {
					configurablePackages.add(elem);
				}
				else if (EXT_ELEM_PHYSICAL_PACKAGE.equals(elemName)) {
					physicalPackages.add(elem);
				}
				else if (EXT_ELEM_DEPLOY_DRIVER.equals(elemName)) {
					deployDrivers.add(elem);
				}
				else if (EXT_ELEM_TECHNOLOGY_DEFINITION.equals(elemName)) {
					technologyDefs.add(elem);
				}
				else if (EXT_ELEM_SERVER_DEFINITION.equals(elemName)) {
					serverDefs.add(elem);
				}
				else if (EXT_ELEM_TECHNOLOGY_MAP.equals(elemName)) {
					technologyMaps.add(elem);
				}
				else {
					if (DEBUG_DEPLOYMENT_EXTENSION) {
						System.err
								.println(DeployCorePlugin
										.getDefault()
										.getResourceString(
												"DeploymentExtensionManager.trace.error.unknownElement", //$NON-NLS-1$
												new Object[] { elemName,
														ext.getContributor().getName()}));
						System.err.flush();
					}
				}
			}
		}
		// Process the extension elements
		// technology types first (depended upon by packages, technology maps)
		processTechnologyDefs(technologyDefs);
		// server types (depended upon by deploy drivers, physical packages)
		processServerDefs(serverDefs);
		// server technology maps
		processTechnologyMaps(technologyMaps);
		// logical packages (depended upon by physical packages)
		processLogicalPackages(logicalPackages);
		// configurable packages
		processConfigurablePackages(configurablePackages);
		// physical packages
		processPhysicalPackages(physicalPackages);
		// deployment drivers
		processDeployDrivers(deployDrivers);
	}

	private void processTechnologyDefs(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processTechnologyDef((IConfigurationElement) it.next());
		}
	}

	private void processTechnologyDef(IConfigurationElement element) {
		try {
			ITechnologyDefinition itd = new TechnologyDefinition(element);
			if (mTechnologyDefsByID.containsKey(itd.getID())) {
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.duplicateTechnologyDefinitionId")); //$NON-NLS-1$
			}
			mTechnologyDefsByID.put(itd.getID(), itd);
		}
		catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.technologyDefinitionParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(ServerDefinition.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor().getName()}));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private void processServerDefs(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processServerDef((IConfigurationElement) it.next());
		}
	}

	private void processServerDef(IConfigurationElement element) {
		try {
			IServerDefinition isd = new ServerDefinition(element);
			if (mServerDefsByID.containsKey(isd.getID())) {
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.duplicateServerDefinitionId")); //$NON-NLS-1$
			}
			mServerDefsByID.put(isd.getID(), isd);
		}
		catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.serverDefinitionParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(ServerDefinition.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor().getName()}));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private void processTechnologyMaps(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processTechnologyMap((IConfigurationElement) it.next());
		}
	}

	private void processTechnologyMap(IConfigurationElement element) {
		try {
			new TechnologyMap(element);
		}
		catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.technologyMapParseError", //$NON-NLS-1$
										new Object[] { element
												.getDeclaringExtension()
												.getContributor().getName()}));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private void processLogicalPackages(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processLogicalPackage((IConfigurationElement) it.next());
		}
	}

	private void processLogicalPackage(IConfigurationElement element) {
		try {
			LogicalPackageExtension lpp = new LogicalPackageExtension(element);
			if (mIdToLogicalExtension.containsKey(lpp.getID())) {
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.duplicateLogicalPackageId")); //$NON-NLS-1$
			}
			mIdToLogicalExtension.put(lpp.getID(), lpp);
			mLogicalExtensionByFileExtension.add(lpp);
		}
		catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.logicalPackageParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(PackageExtension.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor().getName()}));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private void processConfigurablePackages(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processConfigurablePackage((IConfigurationElement) it.next());
		}
	}

	private void processConfigurablePackage(IConfigurationElement element) {
		try {
			ConfigurablePackageExtension cpe = new ConfigurablePackageExtension(
					element);
			if (mIdToConfigurableExtension.containsKey(cpe.getID())) {
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.duplicateConfigurablePackageId")); //$NON-NLS-1$
			}
			mIdToConfigurableExtension.put(cpe.getID(), cpe);
			mConfigurableExtensionByFileExtension.add(cpe);
		}
		catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.configurablePackageParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(PackageExtension.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor().getName()}));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private void processPhysicalPackages(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processPhysicalPackage((IConfigurationElement) it.next());
		}
	}

	private void processPhysicalPackage(IConfigurationElement element) {
		try {
			PhysicalPackageExtension ppp = new PhysicalPackageExtension(element);
			if (mIdToPhysicalExtension.containsKey(ppp.getID())) {
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.duplicatePhysicalPackageId")); //$NON-NLS-1$
			}
			mIdToPhysicalExtension.put(ppp.getID(), ppp);
            
            List<PhysicalPackageExtension> pkgExts = null;
            if(mPhysicalExtensionByFileExtension.containsKey(ppp.getFileExtension())){
                pkgExts = (List)mPhysicalExtensionByFileExtension.get(ppp.getFileExtension());
                pkgExts.add(ppp);
            }
            else{
                pkgExts = new ArrayList<PhysicalPackageExtension>();
                pkgExts.add(ppp);
                mPhysicalExtensionByFileExtension.put(ppp.getFileExtension(), pkgExts);
            }
			
		}
		catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.physicalPackageParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(PackageExtension.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor().getName()}));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private void processDeployDrivers(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processDeployDriver((IConfigurationElement) it.next());
		}
	}

	private void processDeployDriver(IConfigurationElement element) {
		try {
			DeployDriverExtension de = new DeployDriverExtension(element);
			Assert
					.isTrue(
							!mIdToDeployDriver.containsKey(de.getID()),
							DeployCorePlugin
									.getDefault()
									.getResourceString(
											"DeploymentExtensionManager.assert.duplicateDeployDriver", //$NON-NLS-1$
											new Object[] { de.getID(),
													element.toString(),
													de.getID()}));
			mIdToDeployDriver.put(de.getID(), de);
			String profileID = de.getConnectionProfileProvider().getId();
			List drivers;
			if (mProfileIdToDeployDriverList.containsKey(profileID)) {
				drivers = (List) mProfileIdToDeployDriverList.get(profileID);
			}
			else {
				drivers = new ArrayList();
				mProfileIdToDeployDriverList.put(profileID, drivers);
			}
			drivers.add(de);
		}
		catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.deployDriverParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(DeployDriverExtension.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor().getName()}));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private static class IDComparator implements Comparator {

		public int compare(Object o1, Object o2) {
			String s1;
			String s2;
			if (o1 instanceof String) {
				s1 = (String) o1;
			}
			else {
				s1 = ((IPackageExtension) o1).getID();
			}
			if (o2 instanceof String) {
				s2 = (String) o2;
			}
			else {
				s2 = ((IPackageExtension) o2).getID();
			}
			return s1.compareTo(s2);
		}
	}

	private static class FileExtensionComparator implements Comparator {

		public int compare(Object o1, Object o2) {
			int retVal;
			String s1;
			String s2;
			if (o1 instanceof String) {
				s1 = (String) o1;
				if (o2 instanceof String) {
					s2 = (String) o2;
				}
				else {
					s2 = ((IPackageExtension) o2).getFileExtension();
				}
				retVal = s1.compareTo(s2);
			}
			else if (o2 instanceof String) {
				s2 = (String) o2;
				if (o1 instanceof String) {
					s1 = (String) o1;
				}
				else {
					s1 = ((IPackageExtension) o1).getFileExtension();
				}
				retVal = s1.compareTo(s2);
			}
			else {
				IPackageExtension ipe1 = (IPackageExtension) o1;
				IPackageExtension ipe2 = (IPackageExtension) o2;
				retVal = ipe1.getFileExtension().compareTo(
						ipe2.getFileExtension());
				if (retVal == 0) {
					retVal = ipe1.getID().compareTo(ipe2.getID());
				}
			}
			return retVal;
		}
	}

}