/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler.xpath;

import java.lang.reflect.Method;
import java.util.ArrayList;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.ScopeStack;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Switches;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap.WSDLMap;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceTranslator;
import org.eclipse.stp.b2j.core.jengine.internal.utils.TranslatorUtils;
import org.eclipse.stp.b2j.core.jengine.internal.utils.UIDPool;
import org.eclipse.stp.b2j.core.jengine.internal.utils.XPU;
import org.eclipse.stp.b2j.core.publicapi.extension.wsdlbinding.XSDTypeTranslator;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Node;

/**
 * 
 * @author amiguel
 * 
 * Translates XPATH expressions into engine Program parts (e.g. methods)
 * 
 * First converts the XPATH expression into a tree of XPathNodes, then runs certain 
 * optimisation classes on those nodes.
 */
public class XPATHTreeTranslator {

	public static boolean XPATH_DEBUG = false;

	private static final String INITIAL_NODESET = "initialNodeSet";
	
	int depth = 1;
	int depthMax = 1;
	
	int SCOPE = 0;
	
	static UIDPool uidpool = new UIDPool();
	int UID = uidpool.getUID();

	StringBuffer decl = new StringBuffer();
	ArrayList var_refs = new ArrayList();
	boolean force_no_cache = false;
	
	boolean variables_are_links = false;
	
	StringBuffer prefix = new StringBuffer();
	StringBuffer postfix = new StringBuffer();
	
	StringBuffer sb = new StringBuffer();

	///////////////////////////////////////////////////////
	// UTILITY FUNCTIONS
	///////////////////////////////////////////////////////

	private String removeQuotes(String s) {
		if (	s.length() > 1
				&& 
				(
					(s.startsWith("\"") && s.endsWith("\""))
					|| (s.startsWith("'") && s.endsWith("'"))
				)
			) {
			return s.substring(1,s.length()-1);
		}
		return s;
	}
	
	private int getDepth() {
		return depth;
	}

	private int getParentDepth() {
		return depth-1;
	}

	private void incrementDepth() {
		depth++;
		if (depth > depthMax) depthMax = depth;
	}
	
	private void decrementDepth() {
		depth--;
	}
	
	private void incrementScope() {
		SCOPE++;
	}
	private void decrementScope() {
		if (SCOPE >= 1) SCOPE--;	
	}
	
	public String getVarRef(int index) {
		return "xpath_variable"+index+"_"+UID;
	}
	public String getLinkRef(int index) {
		return "xpath_link"+index+"_"+UID;
	}
	
	public String getRetRef() {
		return "xpath_return_"+UID;
	}

	public String getDirtyRef() {
		return "xpath_dirty_"+UID;
	}
	
	private void appendDeclaration(String s) {
		decl.append(s).append('\n');
	}

	private void appendLine(String s) {
		for (int i = 0; i < SCOPE; i++) {
			sb.append("  ");	
		}
		sb.append(s).append('\n');
	}
	
	private void appendNoLine(String s) {
		for (int i = 0; i < SCOPE; i++) {
			sb.append("  ");	
		}
		sb.append(s);
	}
	
	public ArrayList getReferencedVariables() {
		return var_refs;
	}
	
	ArrayList exceptions = new ArrayList();
	
	private void checkForExceptions(String expression, Node source) throws Exception {
		for (int i = 0; i < exceptions.size(); i++) {
			throw XPATHTranslatorException.parserError((Exception)exceptions.get(i),expression,source);
		}
	}
	private void error(String s) {
		exceptions.add(new Exception(s));
	}
	
	///////////////////////////////////////////////////////
	// CODE GENERATION
	///////////////////////////////////////////////////////
	
	public String resolveToBoolean(String method, String expression, Switches switches, NamespaceTranslator nt, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		appendLine("boolean "+getRetRef()+";");
		appendLine("public boolean "+method+"() throws Exception {");

		//caching prefix
		int prefix_index = sb.length();

		resolve(expression,switches,nt,scopes,flow_scopes,wsdlmap,source,false);
		appendLine("  "+getRetRef()+" = XPU.toBoolean(stack.peek()).booleanValue();\n");
//		appendLine("  System.out.println(\""+method+" resolving to \"+"+getRetRef()+"+\"/\"+stack.peek());");
		
		//caching postfix
		sb.insert(prefix_index,prefix.toString());
		appendNoLine(postfix.toString());
		
		appendLine("    return "+getRetRef()+";");
		appendLine("}");
		return decl.toString()+sb.toString();
	}
	
	public String resolveLinksToBoolean(String method, String expression, Switches switches, NamespaceTranslator nt, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		
		variables_are_links = true;
		return resolveToBoolean(method,expression,switches,nt,scopes,flow_scopes,wsdlmap,source);
	}

	public String resolveToDateOrDateTimeInMillis(String method, String expression, Switches switches, NamespaceTranslator nt, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		appendLine("long "+getRetRef()+";");
		appendLine("public long "+method+"() throws Exception {");

		//caching prefix
		int prefix_index = sb.length();

		resolve(expression,switches,nt,scopes,flow_scopes,wsdlmap,source,false);

		String d_type = nt.qualify(XSDTypeTranslator.XSD_TYPE_DATE,false);
		String dt_type = nt.qualify(XSDTypeTranslator.XSD_TYPE_DATETIME,false);
		
		appendLine("  String xpath_tmpreturn = XPU.toString(stack.peek());");
		appendLine("  try {");
		appendLine("  	//parse as a dateTime");
		appendLine("    "+getRetRef()+" = (("+dt_type+") "+dt_type+".fromXML(xpath_tmpreturn)).INTERNAL_VALUE.getTimeInMillis();");
		appendLine("  } catch (Exception e) {");
		appendLine("  	//not a dateTime - must be a date");
		appendLine("    "+getRetRef()+" = (("+d_type+") "+d_type+".fromXML(xpath_tmpreturn)).INTERNAL_VALUE.getTimeInMillis();");
		appendLine("  }");
		
		//caching postfix
		sb.insert(prefix_index,prefix.toString());
		appendNoLine(postfix.toString());

		appendLine("    return "+getRetRef()+";");
//		appendLine("  System.out.println(\""+method+" resolving to \"+xpath_return+\"/\"+stack.peek());");
		appendLine("}");
		return decl.toString()+sb.toString();
	}

	public String resolveToDurationInMillis(String method, String expression, Switches switches, NamespaceTranslator nt, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		appendLine("long "+getRetRef()+";");
		appendLine("public long "+method+"() throws Exception {");

		//caching prefix
		int prefix_index = sb.length();

		String dtype = nt.qualify(XSDTypeTranslator.XSD_TYPE_DURATION,false);
		
		resolve(expression,switches,nt,scopes,flow_scopes,wsdlmap,source,false);
		appendNoLine("  "+getRetRef()+" = ");
		appendNoLine("Math.max(1, (("+dtype+") "+dtype+".fromXML(");
		appendNoLine("XPU.toString(stack.peek())");
		appendNoLine(")).INTERNAL_VALUE.longValue() )");
		appendLine(";");
//		appendLine("  "+getRetRef()+" = XPU.toString(stack.peek());");

//		appendLine("  System.out.println(\""+method+" resolving to \"+xpath_return+\"/\"+stack.peek());");
		
		//caching postfix
		sb.insert(prefix_index,prefix.toString());
		appendNoLine(postfix.toString());

		appendLine("    return "+getRetRef()+";");
		appendLine("}");
		return decl.toString()+sb.toString();
	}
	
	public String resolveToType(String method, String expression, Switches switches, NamespaceTranslator nt, String type, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		appendLine(""+type+" "+getRetRef()+";");
		appendLine("public "+type+" "+method+"() throws Exception {");

		//caching prefix
		int prefix_index = sb.length();

		resolve(expression,switches,nt,scopes,flow_scopes,wsdlmap,source,false);

		appendLine("  "+getRetRef()+" = ("+type+") "+type+".fromXML(XPU.toString(stack.peek()));");
	
//		appendLine("  System.out.println(\""+method+" resolving to \"+xpath_return+\"/\"+stack.peek());");
		
		//caching postfix
		sb.insert(prefix_index,prefix.toString());
		appendNoLine(postfix.toString());

		appendLine("    return "+getRetRef()+";");
		appendLine("}");
		return decl.toString()+sb.toString();
	}

	public String resolveToTypeUsingInstance(String method, String expression, Switches switches, NamespaceTranslator nt, String type, String instance, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		appendLine(""+type+" "+getRetRef()+";");
		appendLine("public "+type+" "+method+"("+type+" typeInstance) throws Exception {");

		//caching prefix
		int prefix_index = sb.length();

		resolve(expression,switches,nt,scopes,flow_scopes,wsdlmap,source,false);

		appendLine("  "+getRetRef()+" = ("+type+") typeInstance.ifromXML(XPU.toString(stack.peek()));");
		
//		appendLine("  System.out.println(\""+method+" resolving to \"+xpath_return+\"/\"+stack.peek());");
		
		//caching postfix
		sb.insert(prefix_index,prefix.toString());
		appendNoLine(postfix.toString());

		appendLine("    return "+getRetRef()+";");
		appendLine("}");
		return decl.toString()+sb.toString();
	}
	
	public String resolveToNodeSet(String method, String expression, Switches switches, NamespaceTranslator nt, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		appendLine("XNodeSet "+getRetRef()+";");
		appendLine("public XNodeSet "+method+"() throws Exception {");

		//caching prefix
		int prefix_index = sb.length();

		resolve(expression,switches,nt,scopes,flow_scopes,wsdlmap,source,false);

		appendLine("  "+getRetRef()+" = (XNodeSet)stack.peek();");
		
		//caching postfix
		sb.insert(prefix_index,prefix.toString());
		appendNoLine(postfix.toString());

		appendLine("    return "+getRetRef()+";");
		appendLine("}");
		return decl.toString()+sb.toString();
	}

	public String resolveVariableQueryToNodeSet(String method, String expression, Util compiler_util, NamespaceTranslator nt, WSDLMap wsdlmap, Node source) throws Exception {
		
		Switches switches = compiler_util.getSwitches();
		
		//There are no variables when resolving variable queries
//		HashMap variable_to_qtype = new HashMap();
		//There are no scopes when resolving variable queries
		ScopeStack scopes = new ScopeStack(compiler_util);
		//There are no scopes when resolving variable queries
		ScopeStack flow_scopes = new ScopeStack(compiler_util);
		
		appendLine("XNodeSet "+getRetRef()+";");
		appendLine("public XNodeSet "+method+"(XNodeSet "+INITIAL_NODESET+") throws Exception {");

		//caching prefix 
		int prefix_index = sb.length();

		resolve(expression,switches,nt,scopes,flow_scopes,wsdlmap,source,true);

		//this is probably the correct way to do it
		appendLine("  "+getRetRef()+" = (XNodeSet)stack.peek();");
		
		//caching postfix
		sb.insert(prefix_index,prefix.toString());
		appendNoLine(postfix.toString());

		appendLine("    return "+getRetRef()+";");
		appendLine("}");
		return decl.toString()+sb.toString();
	}
	
	public String resolveToXmlString(String method, String expression, Switches switches, NamespaceTranslator nt, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		appendLine("String "+getRetRef()+";");
		appendLine("public String "+method+"() throws Exception {");

		//caching prefix
		int prefix_index = sb.length();

		resolve(expression,switches,nt,scopes,flow_scopes,wsdlmap,source,false);
		appendLine("  "+getRetRef()+" = String.valueOf(((XNodeSet)stack.peek()).get(0).getValue());");
		
//		appendLine("  System.out.println(\""+method+" resolving to \"+xpath_return+\"/\"+stack.peek());");
		
		//caching postfix
		sb.insert(prefix_index,prefix.toString());
		appendNoLine(postfix.toString());

		appendLine("    return "+getRetRef()+";");
		appendLine("}");
		return decl.toString()+sb.toString();
	}

	
	public String resolveToString(String method, String expression, Switches switches, NamespaceTranslator nt, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		appendLine("String "+getRetRef()+";");
		appendLine("public String "+method+"() throws Exception {");

		//caching prefix
		int prefix_index = sb.length();

		resolve(expression,switches,nt,scopes,flow_scopes,wsdlmap,source,false);
		appendLine("  "+getRetRef()+" = XPU.toString(stack.peek());");
		
//		appendLine("  System.out.println(\""+method+" resolving to \"+xpath_return+\"/\"+stack.peek());");
		
		//caching postfix
		sb.insert(prefix_index,prefix.toString());
		appendNoLine(postfix.toString());

		appendLine("    return "+getRetRef()+";");
		appendLine("}");
		return decl.toString()+sb.toString();
	}

	public String resolveToNumber(String method, String expression, Switches switches, NamespaceTranslator nt, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		appendLine("double "+getRetRef()+";");
		appendLine("public double "+method+"() throws Exception {");
		
		//caching prefix
		int prefix_index = sb.length();

		resolve(expression,switches,nt,scopes,flow_scopes,wsdlmap,source,false);
		appendLine("  "+getRetRef()+" = XPU.toNumber(stack.peek()).doubleValue();");
//		appendLine("  System.out.println(\""+method+" resolving to \"+xpath_return+\"/\"+stack.peek());");
		
		//caching postfix
		sb.insert(prefix_index,prefix.toString());
		appendNoLine(postfix.toString());

		appendLine("    return "+getRetRef()+";");
		appendLine("}");
		return decl.toString()+sb.toString();
	}

	public String resolveToAny(String method, String expression, Switches switches, NamespaceTranslator nt, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		appendLine("Object "+getRetRef()+";");
		appendLine("public Object "+method+"() throws Exception {");
		
		//caching prefix
		int prefix_index = sb.length();

		resolve(expression,switches,nt,scopes,flow_scopes,wsdlmap,source,false);
		appendLine("  "+getRetRef()+" = stack.peek();");
//		appendLine("  System.out.println(\""+method+" resolving to \"+xpath_return+\"/\"+stack.peek());");
		
		//caching postfix
		sb.insert(prefix_index,prefix.toString());
		appendNoLine(postfix.toString());

		appendLine("    return "+getRetRef()+";");
		appendLine("}");
		return decl.toString()+sb.toString();
	}

	private String resolve(String expression, Switches switches, NamespaceTranslator nt, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source, boolean expectInitialNodeSet) throws Exception {
		
		force_no_cache = false; 
		
		incrementScope();
		incrementScope();
		
		appendDeclaration("//////////////////////////////////");
		appendDeclaration("// XPATH EXPRESSION - "+expression.replace('\n',' ').replace('\r',' '));
		appendDeclaration("//////////////////////////////////");
		

		if (switches.APPEND_BPEL_TO_CALLSTACK) {
			appendLine("    Runner.pushStack(\"XPATH expression "+TranslatorUtils.toJavaString(expression)+"\");");
		}
		appendLine("");
		
		appendLine("Object func_ret = null;");
		appendLine("XNodeSet nsContext = null;");
		appendLine("FStack stack = new FStack();");
//		appendLine("XNodeSet nsTmp"+getParentDepth()+" = new XNodeSet();");
		appendLine("nsTmp"+getParentDepth()+" = new XNodeSet();");
		appendLine("int xpi"+getParentDepth()+" = 0;\n");

		if (expectInitialNodeSet) {
			appendLine("nsContext = "+INITIAL_NODESET+";");
			appendLine("stack.push(nsContext);\n");
		}
		
		
		//debug - print out the expression before we try to parse it
		try {
			XPathNode roots;
			
			//
			// Build a tree from this expression
			//
			XPATHTreeBuilder treebuilder = new XPATHTreeBuilder(nt);
			treebuilder.resolveToTree(expression,scopes,source);

			roots = treebuilder.getRootExpressionsContainer();
			
			//
			// Precompute whatever we can
			//
			if (switches.XPATH_EXPRESSION_PRECOMPUTING) {
				XPATHTreePrecomputer precomputer = new XPATHTreePrecomputer();
				for (int i = 0; i < roots.getChildCount(); i++) {
					precomputer.precompute(roots.getChild(i));
				}
				
				roots = treebuilder.getRootExpressionsContainer();
			}
			
			//
			// Generate code from the tree
			//
			if (roots.getChildCount() != 1) throw new XPATHTranslatorException("error - found more than one expression to translate");
			generateCode(roots.getChild(0),scopes,flow_scopes,wsdlmap,source);
			
/*			
			XPathParser parser = new XPathParser(new StringReader(expression));
			if (XPATH_DEBUG) {
				//run it through the standard visitor that prints stuff out
				parser.XPath(new XPathVisitorAdapter(true));
				parser = new XPathParser(new StringReader(expression));
			}
			parser.XPath(this);
*/			
		} catch (XPATHTranslatorException t) {
			throw t;
		} catch (Throwable t) {
			throw XPATHTranslatorException.parserError(t,expression,source);
		}
		
		decrementScope();
		decrementScope();
		
		if (switches.XPATH_EXPRESSION_CACHING && !force_no_cache) {
			appendDeclaration("boolean "+getDirtyRef()+" = true;");
		}
		
		for (int i = 0; i < var_refs.size(); i++) {
			String varname = (String)var_refs.get(i);
			String ref = getVarRef(i);

			appendDeclaration("SharedVariable "+ref+" = null;");
			appendDeclaration("Message "+ref+"_val = null;");

			if (i == 0) {			

				prefix.append("  if ("+ref+" == null) {\n");
				prefix.append("    //initialise variable references\n");
			}
			prefix.append("    "+ref+" = BPELVariable.getReference(engine,\""+scopes.getVariableName(varname,source)+"\");\n");
			if (i == (var_refs.size()-1)) {
				prefix.append("  }\n");
			}
		}
		
		prefix.append("  Message tmp_value;\n");
		
		//create references to all needed variables
		for (int i = 0; i < var_refs.size(); i++) {
			String varname = (String)var_refs.get(i);
			String ref = getVarRef(i);

			if (switches.XPATH_EXPRESSION_CACHING && !force_no_cache) {
				prefix.append("  tmp_value = BPELVariable.getMessageFromRef(engine,"+ref+");\n");
				prefix.append("  if (tmp_value != "+ref+"_val) {\n");
				prefix.append("    "+ref+"_val = tmp_value;\n");
				prefix.append("    "+getDirtyRef()+" = true;\n");
				prefix.append("  }\n");
			} else {
				prefix.append("    "+ref+"_val = BPELVariable.getMessageFromRef(engine,"+ref+");\n");
			}
		}

		for (int i = 0; i < var_refs.size(); i++) {
			String varname = (String)var_refs.get(i);
			String ref = getVarRef(i);
		}
		
		if (switches.XPATH_EXPRESSION_CACHING && !force_no_cache) {
			prefix.append("  if ("+getDirtyRef()+") {\n");
			prefix.append("    "+getDirtyRef()+" = false;\n");
			postfix.append("  }\n");
		}
		
		for (int i = 0; i <= depthMax; i++) {
			prefix.append("    XNodeSet nsRet"+i+";\n");
			prefix.append("    XNodeSet nsTmp"+i+";\n");
		}
		
		if (switches.APPEND_BPEL_TO_CALLSTACK) {
			appendLine("    Runner.popStack();");
		}
		//did we have any problems while we were parsing?
		checkForExceptions(expression,source);
		
		return sb.toString();
	}

	public void generate(XPathNode node, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		generateCode(node,scopes,flow_scopes,wsdlmap,source);
	}
	
	public void generateCode(XPathNode node, ScopeStack scopes, ScopeStack flow_scopes, WSDLMap wsdlmap, Node source) throws Exception {
		
		switch (node.getType()) {
			case XPathNode.TYPE_PREDICATE:
			{
				//Predicate START
				appendLine("stack.pop();");
				//these are declared at the beginning of the function
				appendLine("nsRet"+getDepth()+" = new XNodeSet();");
				appendLine("nsTmp"+getDepth()+" = nsContext;");
				appendLine("for (int xpi"+getDepth()+" = 0; xpi"+getDepth()+" < nsTmp"+getDepth()+".size(); xpi"+getDepth()+"++) {");
				appendLine("  nsContext = XPU.getSingleNodeSet(nsTmp"+getDepth()+".get(xpi"+getDepth()+"));");
				appendLine("  //PREDICATE EXPRESSION");
		
				incrementDepth();
				incrementScope();
				break;
			}
		}
		
		//depth first
		for (int i = 0; i < node.getChildCount(); i++) {
			generateCode(node.getChild(i),scopes,flow_scopes,wsdlmap,source);
		}
		
		//generate code for this node
		switch (node.getType()) {
			case XPathNode.TYPE_PREDICATE:
			{
				//Predicate END
				decrementScope();
				decrementDepth();
				
				appendLine("  //END PREDICATE EXPRESSION");
				appendLine("  Object tmp"+getDepth()+" = stack.pop();");
				appendLine("  if (tmp"+getDepth()+" instanceof Number) {");
				appendLine("    if (((Number)tmp"+getDepth()+").intValue() == xpi"+getDepth()+") {");
				appendLine("      nsRet"+getDepth()+".addAll(nsContext);");
				appendLine("    }");
				appendLine("  } else if (XPU.toBoolean(tmp"+getDepth()+").booleanValue()) {");
				appendLine("    nsRet"+getDepth()+".addAll(nsContext);");
				appendLine("  }");
				appendLine("}");
				appendLine("nsContext = nsRet"+getDepth()+";");
				appendLine("stack.push(nsContext);");
				break;
			}
			case XPathNode.TYPE_NUMBER:
			{
				String number = String.valueOf(node.getArg(0));
			
				appendLine("stack.push(new Double("+number+"));");
				break;
			}
			case XPathNode.TYPE_BOOLEAN:
			{
				String value = String.valueOf(node.getArg(0));
			
				if (value.equalsIgnoreCase("true")) {
					appendLine("stack.push(Boolean.TRUE);");
				} else {
					appendLine("stack.push(Boolean.FALSE);");
				}
				break;
			}
			case XPathNode.TYPE_LITERAL:
			{
				String literal = String.valueOf(node.getArg(0));
		
				if (!literal.startsWith("\"")) literal = "\""+literal;
				if (!literal.endsWith("\"")) literal = literal+"\"";
				
				//translate the bit between the quotes " to a java string (\ == \\ etc)
				literal = TranslatorUtils.toJavaString(literal,1,literal.length()-2);

				appendLine("stack.push("+literal+");");
				break;
			}
			case XPathNode.TYPE_ADDITIVE:
			{
				String operator = String.valueOf(node.getArg(0));
		
				if (operator.equals("+")) {
					operator = "add";
				} else if (operator.equals("-")) {
					operator = "subtract";
				} else {
					error("additive operator "+operator+" not found");
					operator = "add";
				}		
				if (XPATH_DEBUG) appendLine("engine.print(\"XPR - \"+stack.peek(2)+\" ["+operator+"] \"+stack.peek());");
				appendLine("stack.discardAndPush(2,XPU."+operator+"(stack.peek(2),stack.peek()));");
				if (XPATH_DEBUG) appendLine("engine.print(\"XPR - "+operator+" = \"+stack.peek());");
				break;
			}
			case XPathNode.TYPE_MULTIPLICATIVE:
			{
				String operator = String.valueOf(node.getArg(0));
				
				if (operator.equals("*")) {
					operator = "multiply";
				} else if (operator.equals("div")) {
					operator = "div";
				} else if (operator.equals("mod")) {
					operator = "mod";
				} else {
					error("multiplicative operator "+operator+" not found");
					operator = "multiply";
				}		
				if (XPATH_DEBUG) appendLine("engine.print(\"XPR - \"+stack.peek(2)+\" ["+operator+"] \"+stack.peek());");
				appendLine("stack.discardAndPush(2,XPU."+operator+"(stack.peek(2),stack.peek()));");
				if (XPATH_DEBUG) appendLine("engine.print(\"XPR - "+operator+" = \"+stack.peek());");
				break;
			}
			case XPathNode.TYPE_UNARY:
			{
				String operator = String.valueOf(node.getArg(0));
		
				if (operator.equals("-")) {
					operator = "negate";
				} else {
					error("unary operator "+operator+" not found");
					operator = "negate";
				}
					
				
				if (XPATH_DEBUG) appendLine("engine.print(\"XPR - ["+operator+"] \"+stack.peek());");
				appendLine("stack.push(XPU."+operator+"(stack.pop()));");
				if (XPATH_DEBUG) appendLine("engine.print(\"XPR - "+operator+" = \"+stack.peek());");
				break;
			}
			case XPathNode.TYPE_RELATIONAL:
			{
				String operator = String.valueOf(node.getArg(0));
			
				if (operator.equals(">")) {
					operator = "gt";
				} else if (operator.equals(">=")) {
					operator = "ge";
				} else if (operator.equals("<")) {
					operator = "lt";
				} else if (operator.equals("<=")) {
					operator = "le";
				} else {
					error("relational operator "+operator+" not found");
					operator = "gt";
				}
					
				if (XPATH_DEBUG) appendLine("engine.print(\"XPR \"+stack.peek(2)+\" ["+operator+"] \"+stack.peek());");
				appendLine("stack.discardAndPush(2,XPU."+operator+"(stack.peek(2),stack.peek()));");
				if (XPATH_DEBUG) appendLine("engine.print(\"XPR "+operator+" = \"+stack.peek());");
				break;
			}
			case XPathNode.TYPE_EQUALITY:
			{
				String operator = String.valueOf(node.getArg(0));
				if (XPATH_DEBUG) System.out.println("XPR - equality expr - "+operator);
				
				if (operator.equals("=")) {
					operator = "eq";
				} else if (operator.equals("!=")) {
					operator = "ne";
				} else {
					error("equality operator "+operator+" not found");
					operator = "ne";
				}
				
				if (XPATH_DEBUG) appendLine("engine.print(\"XPR \"+stack.peek(2)+\" ["+operator+"] \"+stack.peek());");
				appendLine("stack.discardAndPush(2,XPU."+operator+"(stack.peek(2),stack.peek()));");
				if (XPATH_DEBUG) appendLine("engine.print(\"XPR "+operator+" = \"+stack.peek());");
				break;
			}
			case XPathNode.TYPE_ANDEXPR:
			{
				String operator = String.valueOf(node.getArg(0));
				appendLine("stack.push(XPU.and(stack.pop(),stack.pop()));");
				break;
			}
			case XPathNode.TYPE_OREXPR:
			{
				String operator = String.valueOf(node.getArg(0));
				appendLine("stack.push(XPU.or(stack.pop(),stack.pop()));");
				break;
			}
			case XPathNode.TYPE_VARIABLEREF:
			{
				String tovariable = ""+node.getArg(0);

				if (variables_are_links) {
					
					//special case function where the argument is checked to be a literal and added as an ARG
					//instead of as a child node (which would be added to the stack etc)

					force_no_cache = true;

					appendLine("stack.push(new Boolean(BPELLink.incoming(engine,\""+flow_scopes.getLinkName(tovariable,source)+"\")));");
				
				} else {

					String topart = null;
					
					int dotindex = tovariable.indexOf('.');
					if (dotindex != -1) {
						topart = tovariable.substring(dotindex+1);
						tovariable = tovariable.substring(0,dotindex);
					}
					
					String qtype = scopes.getVariableQtype(tovariable,source);
					
					if (qtype == null) error("XPath: could not find variable reference "+tovariable);
					
					int index = var_refs.size();
					var_refs.add(tovariable);
					
					String REF = getVarRef(index);

					appendLine("nsContext = XPU.getSingleNodeSet( \""+tovariable+"\", ("+qtype+") "+qtype+".fromEngineMessage("+REF+"_val), null, 0);");
					appendLine("stack.push(nsContext);");
						
					if (topart != null) {
						
						if (wsdlmap.getMessage(qtype) == null) {
							error("Could not find WSDL message "+qtype);
						}
						
						String qpartType = wsdlmap.getMessage(qtype).getPartType(topart);
					
						if (qpartType == null) {
							error("No part type "+topart+" found in WSDL message "+qtype);
						}
						
						//auto create the array to hold the part
						appendLine("if ((("+qtype+") nsContext.get(0).getValue())."+NamespaceTranslator.getElement(topart)+".length == 0) {");
						appendLine("(("+qtype+") nsContext.get(0).getValue())."+NamespaceTranslator.getElement(topart)+" = new "+qpartType+"[1];");
						appendLine("(("+qtype+") nsContext.get(0).getValue())."+NamespaceTranslator.getElement(topart)+"[0] = new "+qpartType+"();");
						appendLine("}");
						/*
						//auto populate it with a default empty one to assign into
						appendLine("if ((("+qtype+") nsContext.get(0).getValue())."+NamespaceTranslator.getElement(topart)+"[0] == null) {");
						appendLine("(("+qtype+") nsContext.get(0).getValue())."+NamespaceTranslator.getElement(topart)+"[0] = new "+qpartType+"();");
						appendLine("}");*/
						
						//fetch the part, and link it to the parent node set which is the variable (this allows assignment back into the variable)
						appendLine("nsContext = XPU.getSingleNodeSet( \""+topart+"\", (("+qtype+") nsContext.get(0).getValue())."+NamespaceTranslator.getElement(topart)+"[0], (("+qtype+") nsContext.get(0).getValue())."+NamespaceTranslator.getElement(topart)+", 0, nsContext.get(0));");
						appendLine("stack.discardAndPush(1,nsContext);");
					}
				}
				break;
			}
			case XPathNode.TYPE_FUNCTION_CALL:
			{
				String function = node.getArg(0);
				int args = node.getChildCount();
				
				//ignore function namespace for now
				function = NamespaceTranslator.getName(function);
		
				if (XPATH_DEBUG) System.out.println("XPR - function call - "+function+"("+args+")");
				
				//special cases
				if (function.equals("getThreadChildIndex")) {
					
					appendLine("stack.push(new Double(bpelThreadChildIndex));");
					
				} else if (function.equals("concat")) {
					
					int id = uidpool.getUID();
					
					appendLine("StringBuffer concat"+id+" = new StringBuffer();");
					for (int i = 0; i < args; i++) {
						appendLine("XPU.toString( stack.peek("+(args-i)+"), concat"+id+" );");
					}
					appendLine("stack.discardAndPush("+args+",concat"+id+".toString());");
					
				} else if (function.equals("substring")) {
					appendNoLine("stack.discardAndPush( ");
					if (args > 2) {
						appendNoLine("3,");
					} else {
						appendNoLine("2,");
					}
					appendNoLine("((String)stack.peek(3)).substring(");
					if (args > 2) {
						appendNoLine("((Number)stack.peek(2)).intValue()");
						appendNoLine(",((Number)stack.peek()).intValue()");
					} else {
						appendNoLine("((Number)stack.peek()).intValue()");
					}
					appendNoLine(")");
					appendLine(");");
		
				} else if (function.equals("getVariableProperty")) {
					
					String propname = String.valueOf(node.getArg(2));
					appendLine("stack.discardAndPush(1,((XNodeSet)stack.peek()).get(0).getValue().getBpelProperty(\""+propname+"\"));");
					/*
				} else if (function.equals("getLinkStatus")) {
					
					//special case function where the argument is checked to be a literal and added as an ARG
					//instead of as a child node (which would be added to the stack etc)

					force_no_cache = true;
//					appendLine("stack.push(new Boolean(BPELLink.incoming(engine,(String)stack.pop())));");
					appendLine("stack.push(new Boolean(BPELLink.incoming(engine,\""+flow_scopes.getLinkName(String.valueOf(node.getArg(1)),source)+"\")));");
								*/
				} else {
					
					String origFunction = function;
					
					//naming conversions
					if (function.equals("string")) {
						function = "toString";
					} else if (function.equals("number")) {
						function = "toNumber";
					} else if (function.equals("boolean")) {
						function = "toBoolean";
					} else if (function.equals("contains")) {
						function = "stringContains";
					} else if (function.equals("starts-with")) {
						function = "stringStartsWith";
					} else if (function.equals("string-length")) {
						function = "stringLength";
					} else if (function.equals("true")) {
						function = "booleanTrue";
					} else if (function.equals("false")) {
						function = "booleanFalse";
					} else if (function.equals("not")) {
						function = "booleanNot";
					} else if (function.equals("floor")) {
						function = "numberFloor";
					} else if (function.equals("ceiling")) {
						function = "numberCeiling";
					} else if (function.equals("round")) {
						function = "numberRound";
					} else if (function.equals("sum")) {
						function = "numberSum";
					}
					
					boolean foundName = false;
					boolean foundArgs = false;
					
					int extra_arg_count = 3;
					
					Method[] methods = XPU.class.getDeclaredMethods();
					for (int k = 0; k < methods.length; k++) {
						if (methods[k].getName().equals(function)) {
							foundName = true;
							if (methods[k].getParameterTypes().length - extra_arg_count == args) {
								foundArgs = true;
								break;
							}
						}
					}
					
					if (!foundName) {
						error("function '"+origFunction+"' not known");
					}
					if (!foundArgs) {
						error("function '"+origFunction+"' cannot accept "+args+" arguments");
					}
					
					//WARNING - change any of this and you have to change the constant 'extra_arg_count' above
					
					appendNoLine("func_ret = XPU."+function+"(nsContext,");
					for (int i = 0; i < args; i++) {
						appendNoLine("stack.peek("+(args-i)+"),");
					}
					appendLine("xpi"+getParentDepth()+",nsTmp"+getParentDepth()+");");
					appendLine("stack.discard("+args+");");
					//if the function returns something, push it onto the stack
					appendLine("if (func_ret != null) stack.push(func_ret);");
				}
		
				break;
			}
			case XPathNode.TYPE_NODE_TYPE_TEST:
			{
				String function = String.valueOf(node.getArg(0));
				
				//TODO need to evaluate node type test here
				break;
			}
			case XPathNode.TYPE_NAME_TEST:
			{
				String name = String.valueOf(node.getArg(0));
				
				appendLine("stack.pop();");
				appendLine("nsContext = XPU.nameTest(nsContext,\""+name+"\");");
				appendLine("stack.push(nsContext);");
				break;
			}
			case XPathNode.TYPE_AXIS_SPECIFIER:
			{
				String axis = String.valueOf(node.getArg(0));

				//TODO need to evaluate all kinds of axis here
					
				appendLine("stack.pop();");
				appendLine("nsContext = XPU."+axis+"(nsContext,xpi"+getParentDepth()+",nsTmp"+getParentDepth()+");");
				appendLine("stack.push(nsContext);");
					
				break;
			}
		}
	}
}