/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.transport;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;

import org.eclipse.stp.b2j.core.jengine.internal.extensions.sessiontransport.tcpip.TCPIPTransport;
import org.eclipse.stp.b2j.core.jengine.internal.transport.session.Session;
import org.eclipse.stp.b2j.core.jengine.internal.transport.session.SessionFactory;
import org.eclipse.stp.b2j.core.jengine.internal.utils.Logger;
import org.eclipse.stp.b2j.core.jengine.internal.utils.StreamUtils;
import org.eclipse.stp.b2j.core.publicapi.transport.session.SessionAddress;

/**
 * 
 * @author amiguel
 *
 * A class to test the session and transport layer
 */
public class TransportTest {
	
	static final boolean KILL_SOCKETS = false;

	static final boolean TEST_UNENCRYPTED = true;
	static final boolean TEST_ENCRYPTED = false;

	static final boolean TEST_BASIC = false;
	static final boolean TEST_MULTIPLEXED = false;
	static final boolean TEST_RECONNECTION = true;
	
	public static void main(String[] args) {
		Logger.PRINT_INFO = true;
		
		String initiator = "localhost";
		String listener = "localhost";
		boolean am_initiator = false;
		boolean am_both = true;
		
		if (args.length > 0) {
			am_both = false; 
			
			initiator = args[0];
			listener = args[1];
			if (!initiator.equalsIgnoreCase("localhost")) {
				am_initiator = false;
			} else {
				am_initiator = true;
			}
		}
		
		System.out.println("B2J engine transport test");
		SessionAddress address;

		int port = 29000;
		
		int RUNS = 30;
		
		long[] times = new long[30];
		Arrays.fill(times,-1);
		
		boolean forever = true;
		
		int count = 0;
		
//		for (int i = 0; i < RUNS; i++) {
		while (forever) {
			count++;
			
			int time_index = 0;
			
			if (port >= 29900) {
				port = 29000;
			}
			
			if (TEST_UNENCRYPTED) {
				if (TEST_BASIC) {
					port++;
					address = new SessionAddress(initiator,port,port,listener,port,port);
					times[time_index++] += testSession(count,address,am_both,am_initiator);
				}
				
				if (TEST_MULTIPLEXED) {
					port++;
					address = new SessionAddress(initiator,port,port,listener,port,port);
					address.setRequiresMultipleStreams(true);
					times[time_index++] += testSession(count,address,am_both,am_initiator);
				}
	
				if (TEST_RECONNECTION) {
					port++;
					address = new SessionAddress(initiator,port,port,listener,port,port);
					address.setRequiresMultipleStreams(true);
					address.setRequiresLinkReconnection(true);
					address.setStartupFailureAbortTimeout(60000);
					address.setReconnectionFailureAbortTimeout(120000);
					times[time_index++] += testSession(count,address,am_both,am_initiator);
				}
			}
			
			if (TEST_ENCRYPTED) {
				if (TEST_BASIC) {
					port++;
					address = new SessionAddress(initiator,port,port,listener,port,port);
					address.setRequiresEncryption(true);
					times[time_index++] += testSession(count,address,am_both,am_initiator);
				}
	
				if (TEST_MULTIPLEXED) {
					port++;
					address = new SessionAddress(initiator,port,port,listener,port,port);
					address.setRequiresEncryption(true);
					address.setRequiresMultipleStreams(true);
					times[time_index++] += testSession(count,address,am_both,am_initiator);
				}
	
				if (TEST_RECONNECTION) {
					port++;
					address = new SessionAddress(initiator,port,port,listener,port,port);
					address.setRequiresEncryption(true);
					address.setRequiresMultipleStreams(true);
					address.setRequiresLinkReconnection(true);
					address.setStartupFailureAbortTimeout(60000);
					address.setReconnectionFailureAbortTimeout(120000);
					times[time_index++] += testSession(count,address,am_both,am_initiator);
				}
			}
		}

		System.out.println("\n\n----- ALL RUNS COMPLETED SUCCESSFULLY, WAITING FOR ANY FURTHER PRINTOUTS...");
		
		try {
			Thread.sleep(6000);
		} catch (Exception e) {
		}
		
		System.out.println("\n\n----- TIMING INFO FOR ALL RUNS:");
		for (int k = 0; k < times.length; k++) {
			if (times[k] != -1) {
				times[k] = times[k] / RUNS;
				System.out.println("----- Timer for "+k+": "+times[k]+"ms");
			}
		}
	}
	
	public static void testSessionAddressSerialisation(SessionAddress address) {
		try {
			String serialised = SessionAddress.toString(address);
			SessionAddress deserialised = SessionAddress.fromString(serialised);
			
			if (deserialised.equals(address)) {
				System.out.println("----- Session serialised and deserialised OK");
			} else {
				System.out.println("----- Session serialised and deserialised incorrectly");
				System.out.println(serialised);
				System.exit(0);
			}
		} catch (Exception e) {
			System.out.println("----- An error occurred while serialising and deserialising the Session: "+e);
			e.printStackTrace();
		}
	}

	public static long testSession(int count, SessionAddress address, boolean am_both, boolean am_initiator) {

		System.out.println("\n\n----- Test session "+count+" for "+address+"\n");

		testSessionAddressSerialisation(address);
		
		TestThread server = null;
		TestThread client = null;
		
		if (address.getListenerPortMaximum() != address.getListenerPortMinimum()) {
			System.out.println("----- Address requires port scanning");
			
			if (am_both) {
				System.out.println("----- Address requires port scanning but cannot pass bound address to unknown process");
				return 0;
			}
			
			server = new TestThread(address,"Listener",false);

			//wait until the session comes up
			while (server.err == null && server.session == null) {
				try {
					Thread.sleep(500);
				} catch (Exception e) {}
			}
			
			//wait until the session is bound (actual address is there)
			while (server.err == null && server.session.getActualAddress() == null) {
				try {
					Thread.sleep(500);
				} catch (Exception e) {}
			}
			
			System.out.println("----- Actual address found: "+server.session.getActualAddress());
			
			client = new TestThread(server.session.getActualAddress(),"Initiator",true);
			
		} else {
			System.out.println("----- Address specifies an exact port");
			
			if (am_both || !am_initiator) { 
				server = new TestThread(address,"Listener",false);
			}
			try {
				//give the listener time to create the serversocket
				Thread.sleep(2000);
			} catch (Exception e) {
			}
			if (am_both || am_initiator) { 
				client = new TestThread(address,"Initiator",true);
			}
		}
			
		
		try {
			if (am_both || !am_initiator) { 
				server.join();
			}
			if (am_both || am_initiator) { 
				client.join();
			}
		} catch (Exception e) {
		}
		
		long time_taken = 0;
		if (am_both) {
			time_taken = (server.time_ms + client.time_ms)/2;
		} else if (am_initiator) {
			time_taken = client.time_ms;
		} else {
			time_taken = server.time_ms;
		}

		if (am_both) {
			if (server.err != null || client.err != null) {
				System.out.println("\n----- Test session FAILED ("+time_taken+"ms)\n\n");
				if (server.err != null) {
					server.err.printStackTrace();
				}
				if (client.err != null) {
					client.err.printStackTrace();
				}
				System.exit(0);
			}
		}
		if (am_both || !am_initiator) { 
			if (server.err != null) {
				System.out.println("\n----- Test Listener session FAILED ("+time_taken+"ms)\n\n");
				if (server.err != null) {
					server.err.printStackTrace();
				}
				System.exit(0);
			}
		}
		if (am_both || am_initiator) { 
			if (client.err != null) {
				System.out.println("\n----- Test Initiator session FAILED ("+time_taken+"ms)\n\n");
				if (client.err != null) {
					client.err.printStackTrace();
				}
				System.exit(0);
			}
		}
		/*
		if (server.err != null || client.err != null) {
			System.out.println("\n----- Test session FAILED ("+time_taken+"ms)\n\n");
			if (server.err != null) {
				server.err.printStackTrace();
			}
			if (client.err != null) {
				client.err.printStackTrace();
			}
			System.exit(0);
		} else {
			System.out.println("\n----- Test session SUCCEEDED ("+time_taken+"ms)\n\n");
		}*/
		System.out.println("\n----- Test session SUCCEEDED ("+time_taken+"ms)\n\n");
		
		return time_taken;
	}
	
	public static class TestThread extends Thread {
		Throwable err;
		
		SessionAddress address;
		boolean initiator = false;
		
		Session session;
		
		String name;
		
		long time_ms = 0;
		
		public TestThread(SessionAddress address, String name, boolean initiator) {
			this.address = address;
			this.initiator = initiator;
			this.name = name;
			start();
		}
		public void run() {
			try {
				session = SessionFactory.newSession(address,initiator);
				runEx(session);
			} catch (Throwable t) {
				t.printStackTrace();
				err = t;
			}
		}
		public void runEx(Session session) throws Exception {
			System.out.println("----- "+name+" session transport "+session.getTransportImplementationName());
			session.begin();
			
			System.out.println("----- "+name+" session connected "+session.getActualAddress());
			
			time_ms = System.currentTimeMillis();
			
			for (int k = 0; k < 20000; k++) {
				writeData(name,session);
				if (KILL_SOCKETS) {
					if (!session.getActualAddress().getRequiresEncryption() && session.getActualAddress().getRequiresLinkReconnection() && k == 9000) {
						try {
							System.out.println("----- ********** KILLING SOCKET **********");
							TCPIPTransport transport = (TCPIPTransport)session.transport;
							transport.socket.close();
							System.out.println("----- ********** KILLED SOCKET **********");
						} catch (Exception e) {
						}
					}
				}

				readData(name,session);
				if (KILL_SOCKETS) {
					if (!session.getActualAddress().getRequiresEncryption() && session.getActualAddress().getRequiresLinkReconnection() && k == 9000) {
						try {
							System.out.println("----- ********** KILLING SOCKET **********");
							TCPIPTransport transport = (TCPIPTransport)session.transport;
							transport.socket.close();
							System.out.println("----- ********** KILLED SOCKET **********");
						} catch (Exception e) {
						}
					}
				}
			}

			System.out.println("----- "+name+" Wrote and Read all data OK");
			
			writeFinish(name,session);
			readFinish(name,session);
			
			time_ms = System.currentTimeMillis()-time_ms;
			
			
			session.end(200,200);
		}
	}
	
	static double[] test_doubles = new double[]{
			0.6d,
			1.0d,
			99999999d,
			2137984327098432473284.12300079843279843298d,
			0.00000000000000000000000000000000000000014632d,
	};
	public static void writeFinish(String name, Session session) throws Exception {
		StreamUtils.writeBoolean(session.getOutputStream(Session.DEFAULT_STREAM_INDEX),true);
		session.getOutputStream(Session.DEFAULT_STREAM_INDEX).flush();
	}
	public static void readFinish(String name, Session session) throws Exception {
		StreamUtils.readBoolean(session.getInputStream(Session.DEFAULT_STREAM_INDEX));
	}
	public static void writeData(String name, Session session) throws Exception {
		
		OutputStream outstream = session.getOutputStream(Session.DEFAULT_STREAM_INDEX);
		
		for (int i = 0; i < test_doubles.length; i++) {
			StreamUtils.writeDouble(outstream,test_doubles[i]);
		}
		outstream.flush();

	}
	public static void readData(String name, Session session) throws Exception {
		
		InputStream instream = session.getInputStream(Session.DEFAULT_STREAM_INDEX);
		
		for (int i = 0; i < test_doubles.length; i++) {
			double tmp = StreamUtils.readDouble(instream);
			if (tmp != test_doubles[i]) {
				throw new Exception("----- double mismatch, "+tmp+" != "+test_doubles[i]);
			}
		}
	}
}