/**
 * Copyright (C) 2000-2007, Intalio Inc.
 *
 * The program(s) herein may be used and/or copied only with the
 * written permission of Intalio Inc. or in accordance with the terms
 * and conditions stipulated in the agreement/contract under which the
 * program(s) have been supplied.
 *
 * Dates       		 Author              Changes
 * Feb 21, 2007      Antoine Toulmé   Creation
 */
package org.eclipse.stp.bpmn.policies;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Insets;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.Request;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.gef.requests.ChangeBoundsRequest;
import org.eclipse.gef.requests.CreateRequest;
import org.eclipse.gmf.runtime.common.ui.services.action.global.GetGlobalActionHandlerOperation;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editpolicies.XYLayoutEditPolicy;
import org.eclipse.gmf.runtime.diagram.ui.requests.CreateViewRequest;
import org.eclipse.gmf.runtime.diagram.ui.requests.RequestConstants;
import org.eclipse.gmf.runtime.diagram.ui.requests.CreateViewRequest.ViewDescriptor;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.stp.bpmn.ActivityType;
import org.eclipse.stp.bpmn.commands.ElementTypeEx;
import org.eclipse.stp.bpmn.commands.IElementTypeEx;
import org.eclipse.stp.bpmn.diagram.edit.parts.ActivityEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.PoolEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.SubProcessEditPart;
import org.eclipse.stp.bpmn.diagram.providers.BpmnElementTypes;
import org.eclipse.stp.bpmn.figures.BpmnShapesDefaultSizes;

/**
 * Does the big job of having the compartment been set to the size of the 
 * children.
 * Methods were copied from the PoolPoolCompartmentXYLayoutEditPolicy
 * and adapted to sub processes. That might be a source of problems.
 * @author <a href="mailto:atoulme@intalio.com">Antoine Toulmé</a>
 * @author <a href="http://www.intalio.com">&copy; Intalio, Inc.</a>
 */
public class SubProcessSubProcessCompartmentXYLayoutEditPolicy extends XYLayoutEditPolicy {

	private static final Dimension MIN_DIMENSION = SubProcessEditPart.EXPANDED_SIZE.getCopy();
    /**
     * @generated not
     */
    public static final Insets INSETS = SubProcessEditPart.INSETS;

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.gef.editpolicies.ConstrainedLayoutEditPolicy#getResizeChildrenCommand(org.eclipse.gef.requests.ChangeBoundsRequest)
     */
    protected Command getResizeChildrenCommand(ChangeBoundsRequest request) {
    	List<Request> requests = chunkNegativeMove(request);
    	
    	CompoundCommand command = new CompoundCommand();
    	command.add(super.getResizeChildrenCommand(request));
    	if (requests != null && (!requests.isEmpty())) {
    		for (Request negativeReq : requests) {
    			Command co = getHost().getParent().getParent().
    					getCommand(negativeReq);
    			command.add(co);
    		}

    	} else {
//  		make sure that the children won't be bigger than the pool.
        	// to do that, build a request for the pool to be resized, 
        	// and address it to the diagram.
        	ChangeBoundsRequest req = new ChangeBoundsRequest(
        			RequestConstants.REQ_RESIZE_CHILDREN);
        	req.setEditParts(getHost().getParent());
        	req.setCenteredResize(true);
        	req.setSizeDelta(getMinSizeForSP(request));
    		command.add(getHost().getParent().getParent().getCommand(req));
    	}
    	return command;
    }

    /**
     * This mysterious method checks that the move of the element in the
     * subprocess wouldn't place at negative coordinates, which would
     * mess up pretty much everything.
     * It resizes the parent in the direction of the move,
     * and it moves the children, so that they do not seem to have moved.
     * @param request
     */
    private List<Request> chunkNegativeMove(
    		ChangeBoundsRequest request) {
    	if (request == null) {
    		return null;
    	}
    	boolean isNegative = false;
		Point maxMove = new Point(request.getMoveDelta().x, 
				request.getMoveDelta().y);
		for (Object ep : request.getEditParts()) {
			Point loc = ((GraphicalEditPart) ep).getFigure().getBounds().
				getLocation().getCopy();
			int xmove = loc.x + request.getMoveDelta().x - INSETS.left;
			if (xmove <= 0) {
				maxMove.x =  -loc.x + INSETS.left ;
				isNegative = true;
			}
			int ymove = loc.y + request.getMoveDelta().y - INSETS.top;
			if (ymove <= 0) {
				maxMove.y = -loc.y + INSETS.top;
				isNegative = true;
			}
		}
		Point p = new Point(request.getMoveDelta().x - maxMove.x, 
				request.getMoveDelta().y - maxMove.y);
		request.setMoveDelta(maxMove);
//		 one last test. Maybe the move is negative and going too far.
		// we have modified the request already
		if (isNegative) {
			// test this case :
			// |
			// |    /
			// |   /
			// | |/_
			// |___________________
			
			int toobigy = request.getMoveDelta().y + request.getSizeDelta().height;
			int maxHeight = 0;
			for (Object ep : request.getEditParts()) {
				maxHeight = Math.max(maxHeight, ((GraphicalEditPart) ep).
						getFigure().getBounds().height +((GraphicalEditPart) ep).
						getFigure().getBounds().y);
			}
			if (getHostFigure().getBounds().height < (toobigy + maxHeight)) {
				isNegative = false;
			}
		}
		if (isNegative) {
//			now test this case :
			/*
			 * _______________
			 * 				   /  |
			 *                   /   |
			 *                  /    |
			 * 
			 */
			
			int toobigx = request.getMoveDelta().x + request.getSizeDelta().width;
			int maxWidth = 0;
			for (Object ep : request.getEditParts()) {
				maxWidth = Math.max(maxWidth, ((GraphicalEditPart) ep).
						getFigure().getBounds().width +((GraphicalEditPart) ep).
						getFigure().getBounds().x);
			}
			if (getHostFigure().getBounds().width < (toobigx + maxWidth)) {
				isNegative = false;
			}
		}
		if (!isNegative) {
			return null;
		}
		
		ChangeBoundsRequest compartmentRequest = 
			new ChangeBoundsRequest(RequestConstants.REQ_RESIZE_CHILDREN);
		
		Dimension d = new Dimension(Math.abs(p.x), Math.abs(p.y));
		compartmentRequest.setMoveDelta(p);
		compartmentRequest.setSizeDelta(d);
		compartmentRequest.setEditParts(getHost().getParent());
		
		
		
		ChangeBoundsRequest childrenRequest = 
			new ChangeBoundsRequest(RequestConstants.REQ_RESIZE_CHILDREN);
		Point childrenMove = new Point(- p.x, -p.y);
		childrenRequest.setMoveDelta(childrenMove);
		childrenRequest.setEditParts(new LinkedList(getHost().getChildren()));
		childrenRequest.getEditParts().removeAll(request.getEditParts());
		// we use a copy of the array so that the new edit part will not be moved
		
		List<Request> requests = new LinkedList<Request>();
		requests.add(compartmentRequest);
		if (!childrenRequest.getEditParts().isEmpty()) {
			requests.add(childrenRequest);
		}
		return requests;
	}

	/**
     * Gets the minimum size when creating a new element in the pool.
     * If the element is created on a border, it will
     * add some width and/or length to have it in the compartment.
     * @param request
     * @return
     */
    private Dimension getMinSizeForSP(CreateRequest request) {
    	ChangeBoundsRequest req = new ChangeBoundsRequest(
    			RequestConstants.REQ_RESIZE_CHILDREN);
    	req.setEditParts(Collections.EMPTY_LIST);
    	Dimension dim = getMinSizeForSP(req);
    	Dimension requestSize = request.getSize() == null ? null :
    		request.getSize().getCopy();
    	if (requestSize == null || requestSize.width == -1 && requestSize.height == -1) {
    		// default minimum size for adding a new shape.
//    		Rectangle r = (Rectangle) getConstraintFor(request);
    		requestSize = new Dimension(150, 100);
    		if (request instanceof CreateViewRequest) {
    			List descriptors = ((CreateViewRequest) request).getViewDescriptors();
    			if (!descriptors.isEmpty()) {
    				ViewDescriptor desc = (ViewDescriptor) descriptors.get(0);
    				IElementType type = (IElementType) desc.getElementAdapter().
    					getAdapter(IElementType.class);
    				requestSize = BpmnShapesDefaultSizes.getDefaultSize(type).getCopy();
    			}
    		}
    	}
    	Dimension initialdim = ((GraphicalEditPart) getHost()).getFigure().getSize().getCopy();
    	if (initialdim.height == 0 && initialdim.width == 0) {
    		return initialdim; // meaning no resize, the subprocess is 
    		//being initialized at the same time.
    	}
    	
    	Point loc = request.getLocation().getCopy();
    	getHostFigure().translateToAbsolute(loc);
    	getHostFigure().translateToRelative(loc);
    	Rectangle rect = getHostFigure().getBounds().getCopy();
    	getHostFigure().translateToAbsolute(rect);
    	loc.x = loc.x - rect.x;
    	loc.y = loc.y - rect.y;
    	dim.width = Math.max(dim.width, loc.x + requestSize.width 
    			 + INSETS.getWidth());
    	dim.height = Math.max(dim.height, loc.y + requestSize.height 
    			+ INSETS.getHeight());
    	
    	dim.width = dim.width - initialdim.width + INSETS.getWidth();
    	dim.height = dim.height - initialdim.height + INSETS.getHeight();
    	if (dim.width < 0) {
    		dim.width = 0;
    	}
    	if (dim.height < 0) {
    		dim.height =  0;
    	}
    	return dim;
    }
    
    /**
     * Finds the minimum size for the subprocess by examining the children,
     * if they are modified by the request, then they are resized and or moved, 
     * so that the min size is changed accordingly.
     * 
     * Returns a delta, not the actual size for the subprocess.
     * @param request
     * @return
     */
    private Dimension getMinSizeForSP(ChangeBoundsRequest request) {
    	Dimension initialdim = ((GraphicalEditPart) getHost()).getFigure().getSize().getCopy();
    	Dimension dim = MIN_DIMENSION.getCopy();
    	for (Object child : getHost().getChildren()) {
    		IGraphicalEditPart part = (IGraphicalEditPart) child;
    		Rectangle r = part.getFigure().getBounds().getCopy();
//    		getHostFigure().translateToAbsolute(r);
//    		getHostFigure().translateToRelative(r); // no don't 
    		int w = r.x + r.width;
    		if (request.getEditParts().contains(part)) {
    			w += request.getSizeDelta().width + request.getMoveDelta().x;
    		}
    		if (dim.width < w) {
    			dim.width = w;
    		}
    		int h = r.y + r.height;
    		if (request.getEditParts().contains(part)) {
    			h += request.getSizeDelta().height + request.getMoveDelta().y;
    		}
    		if (dim.height < h) {
    			dim.height = h;
    		}
    	}
    	
    	// now calculate the delta.
    	dim.width = dim.width - initialdim.width + INSETS.left + INSETS.right;
    	dim.height = dim.height - initialdim.height + INSETS.bottom + INSETS.top;
    	if (dim.width < 0) {
    		dim.width = 0;
    	}
    	if (dim.height < 0) {
    		dim.height =  0;
    	}
    	return dim;
    }
    
    /**
     * When creating something, create some place for it.
     */
    @Override
    protected Command getCreateCommand(CreateRequest request) {
    	//  negative location on creation.
    	Point loc = request.getLocation().getCopy();
    	Point here = getHostFigure().getParent().getBounds().getCopy().getLocation();
    	if (here.x == 0 && here.y == 0) {
    		// just initializing the subprocess itself !
    		// we shouldn't change the location of the task,
    		// as the location of the subprocess is subject to change
    	} else {
    		getHostFigure().getParent().translateToAbsolute(here);
    		getHostFigure().translateToAbsolute(loc);
    		getHostFigure().translateToRelative(loc);
    		loc.x = loc.x - here.x - INSETS.left;
    		loc.y = loc.y - here.y - INSETS.top;
    		if (loc.x <= 0) {
    			int correction = - loc.x + 1;
    			request.getLocation().x += correction;
    		}
    		if (loc.y <= 0) {
    			int correction = - loc.y + 1;
    			request.getLocation().y += correction;
    		}
    	}
     	CompoundCommand command = new CompoundCommand();
        command.add(super.getCreateCommand(request));
        ChangeBoundsRequest thisSizeReq = new ChangeBoundsRequest(
    			RequestConstants.REQ_RESIZE_CHILDREN);
        thisSizeReq.setCenteredResize(true);
        thisSizeReq.setEditParts(getHost().getParent());
        thisSizeReq.setSizeDelta(getMinSizeForSP(request));
        if (thisSizeReq.getSizeDelta().height == 0 && thisSizeReq.getSizeDelta().width == 0) {
        	return command;
        }
    	Command co = getHost().getParent().getParent().getCommand(thisSizeReq);
    	command.add(co);
    	return command;
    }    
    
}