/*******************************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.edit.operations;

import java.util.List;
import java.util.Set;

import org.apache.xerces.util.XMLChar;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.stp.core.infrastructure.operations.AbstractScribblerDataModelProvider;
import org.eclipse.stp.core.internal.Messages;
import org.eclipse.stp.core.internal.STPCorePlugin;
import org.eclipse.stp.core.sca.SCAObject;
import org.eclipse.wst.common.frameworks.datamodel.IDataModelOperation;

public class ChangeSCAObjectNameDataModelProvider extends
      AbstractScribblerDataModelProvider implements
      IChangeSCAObjectNameDataModelProperties {

   public Set getPropertyNames() {
      Set propertyNames = super.getPropertyNames();
      propertyNames.add(SCAOBJECT);
      propertyNames.add(NEW_NAME);
      propertyNames.add(EXISTING_SCAOBJECTS);
      return propertyNames;
   }

   public IStatus validate(String propertyName) {
      if (propertyName.equals(SCAOBJECT)) {
         validateSCAObject();
      } else if (propertyName.equals(NEW_NAME)) {
         return validateNewName();
      }

      return super.validate(propertyName);
   }

   public IDataModelOperation getDefaultOperation() {
      return new ChangeSCAObjectNameOperation(model);
   }

   private IStatus validateSCAObject() {
      Object part = getProperty(SCAOBJECT);
      if (!(part instanceof SCAObject))
         return STPCorePlugin.createErrorStatus(
               Messages.invalidObjectPassed_errorText, null);

      return OK_STATUS;
   }

   private IStatus checkForDuplicate() {
      List list = (List) getProperty(IChangeSCAObjectNameDataModelProperties.EXISTING_SCAOBJECTS);
      if (list != null) {
         SCAObject scaObject = (SCAObject) getProperty(IChangeSCAObjectNameDataModelProperties.SCAOBJECT);
         String newName = getStringProperty(NEW_NAME);
         for (int x = 0; x < list.size(); ++x) {
            SCAObject object = (SCAObject) list.get(x);
            if (scaObject == object)
               continue;
            if (newName != null) {
               if (newName.equals(object.getName()))
                  return STPCorePlugin.createErrorStatus(
                        Messages.Name_Already_Exists, null);
            }
         }
      }
      return Status.OK_STATUS;
   }

   private IStatus validateNewName() {
      String newName = getStringProperty(NEW_NAME);

      if (newName == null || newName.length() <= 0)
         return STPCorePlugin.createErrorStatus(
               Messages.newNameBlank_errorText, null);

      // make sure the new name is not the same as the old name.
      SCAObject scaObject = (SCAObject) getProperty(IChangeSCAObjectNameDataModelProperties.SCAOBJECT);
      if (scaObject.getName().equals(newName))
         return STPCorePlugin.createErrorStatus(
               Messages.Existing_name_is_same_as_, null);

      // TODO check if a part with same name exists or not?
      // if yes
      // return STPCorePlugin.createErrorStatus("A Part with the same name
      // already exists. Try a different name", null);

      // validate for a valid file name
      IStatus result = ResourcesPlugin.getWorkspace().validateName(newName,
            IResource.FILE);

      if (result.isOK()) {
         result = checkForSpecialURICharacters(newName);
      }
      if (result.isOK()) {
         result = checkForDuplicate();
      }

      return result;
   }

   // The following characters are restricted characters of
   // org.eclipse.emf.common.util.URI which is
   // used while loading the extension model (.modulex) file.
   protected static final String URI_RESTRICTED_CHARACTERS = ":?#@!"; //$NON-NLS-1$

   private IStatus checkForSpecialURICharacters(String name) {
      if (!XMLChar.isValidNCName(name)) {
         return STPCorePlugin.createErrorStatus(Messages.bind(
               Messages.nameNotValid_errorText, name), null);
      }
      for (int i = 0; i < URI_RESTRICTED_CHARACTERS.length(); i++) {
         if (name.indexOf(URI_RESTRICTED_CHARACTERS.charAt(i)) != -1) {
            return STPCorePlugin.createErrorStatus(Messages.bind(
                  Messages.portionOfNameNotValid_errorText, name, String
                        .valueOf(URI_RESTRICTED_CHARACTERS.charAt(i))), null);
         }
      }

      return Status.OK_STATUS;
   }
}
