/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.deploy.core.operations;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.datatools.connectivity.IConnectionProfile;
import org.eclipse.emf.common.util.URI;
import org.eclipse.stp.soas.deploy.core.DeploymentExtensionManager;
import org.eclipse.stp.soas.deploy.core.FilePackageOutputDescriptor;
import org.eclipse.stp.soas.deploy.core.IConfigurablePackage;
import org.eclipse.stp.soas.deploy.core.IConfigurablePackageExtension;
import org.eclipse.stp.soas.deploy.core.IDeployDriverExtension;
import org.eclipse.stp.soas.deploy.core.ILogicalPackage;
import org.eclipse.stp.soas.deploy.core.IPackage;
import org.eclipse.stp.soas.deploy.core.IPackageConfiguration;
import org.eclipse.stp.soas.deploy.core.IPackageConstructor;
import org.eclipse.stp.soas.deploy.core.IPackageConstructorExtension;
import org.eclipse.stp.soas.deploy.core.IPackageCreationContext;
import org.eclipse.stp.soas.deploy.core.IPackageOutputDescriptor;
import org.eclipse.stp.soas.deploy.core.IPhysicalPackage;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.Utilities;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployConfiguration;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployPackage;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployServer;
import org.eclipse.stp.soas.deploy.models.deployfile.Root;
import org.eclipse.stp.soas.internal.deploy.core.PackageCreationContext;
import org.eclipse.stp.soas.internal.deploy.core.cp.StpProfileManager;

/**
 * @author rcernich
 * 
 * Created on Jul 21, 2004
 */
public class CreateDeployPackagesJob extends Job {

	private IFile mDeployFile;
	private Root mRoot;
	private File mTempFolder;
	private PackageCreationContext.OverwriteCancelQuery mOverwriteCancelQuery;
	private Map mTargetToPackageDescriptorMap;
	private Map mLogicalPackageToPackageDescriptorMap;

	/**
	 * 
	 */
	public CreateDeployPackagesJob(Root root) {
		super(DeployCorePlugin.getDefault().getResourceString(
				"CreateDeployPackagesJob.task.name")); //$NON-NLS-1$
		mRoot = root;
		mOverwriteCancelQuery = new PackageCreationContext.OverwriteCancelQuery();
		mTargetToPackageDescriptorMap = new HashMap();
		mLogicalPackageToPackageDescriptorMap = new HashMap();

		URI uri = mRoot.eResource().getURI();
		if ("platform".equals(uri.scheme())) { //$NON-NLS-1$
			try {
				URL url = FileLocator.toFileURL(new URL(uri.scheme(), uri
						.authority(), uri.path()));
				uri = URI.createFileURI(url.getFile());
			}
			catch (Exception e) {
				throw new RuntimeException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"CreateDeployPackagesJob.error.runtime.fileNotFound") //$NON-NLS-1$
								+ uri.toString(), e);
			}
		}
		mDeployFile = ResourcesPlugin.getWorkspace().getRoot()
				.getFileForLocation(new Path(uri.toFileString()));
		setRule(mDeployFile.getWorkspace().getRoot());
		setUser(true);
		setSystem(false);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public IStatus run(IProgressMonitor monitor) {
		monitor
				.beginTask(
						DeployCorePlugin.getDefault().getResourceString(
								"CreateDeployPackagesJob.task.name"), getBuildCount() * 100 + 600); //$NON-NLS-1$
		IStatus status = Status.OK_STATUS;

		try {
			buildWorkspace(monitor);
			checkCancelled(monitor);

			checkForErrors(monitor);
			checkCancelled(monitor);

			createPackages(monitor);

			refreshWorkspace(monitor);
		}
		catch (InterruptedException e) {
			status = Status.CANCEL_STATUS;
		}
		catch (CoreException e) {
			status = e.getStatus();
		}
		finally {
			deleteTempFolders();
		}
		monitor.done();

		return status;
	}

	public Map getPackageDescriptors() {
		return mTargetToPackageDescriptorMap;
	}

	private void deleteTempFolders() {
		if (mTempFolder != null && mTempFolder.isDirectory()) {
			deleteTempFile(mTempFolder);
			mTempFolder = null;
		}
	}

	private void checkCancelled(IProgressMonitor monitor)
			throws InterruptedException {
		if (monitor.isCanceled()) {
			throw new InterruptedException();
		}
	}

	private void buildWorkspace(final IProgressMonitor monitor)
			throws CoreException {
		IProgressMonitor buildMonitor = new SubProgressMonitor(monitor, 400);

		monitor.subTask(DeployCorePlugin.getDefault().getResourceString(
				"CreateDeployPackagesJob.subtask.buildWorkspace")); //$NON-NLS-1$

		mDeployFile.getWorkspace().build(
				IncrementalProjectBuilder.INCREMENTAL_BUILD, buildMonitor);
		buildMonitor.done();
	}

	private void checkForErrors(IProgressMonitor monitor) throws CoreException {
		monitor.subTask(DeployCorePlugin.getDefault().getResourceString(
				"CreateDeployPackagesJob.subtask.errorCheck")); //$NON-NLS-1$
		if (mDeployFile.exists()) {
			for (Iterator it = Arrays.asList(
					mDeployFile.findMarkers(IMarker.PROBLEM, true,
							IResource.DEPTH_ZERO)).iterator(); it.hasNext();) {
				IMarker marker = (IMarker) it.next();
				int severity = marker.getAttribute(IMarker.SEVERITY,
						IMarker.SEVERITY_INFO);
				if (IMarker.SEVERITY_ERROR == severity) {
					String exceptionMessage = DeployCorePlugin
							.getDefault()
							.getResourceString(
									"CreateDeployPackagesJob.error.errorsFound"); //$NON-NLS-1$ 
					IStatus status = new Status(
							Status.ERROR,
							DeployCorePlugin.getDefault().getBundle()
									.getSymbolicName(),
							-1,
							DeployCorePlugin
									.getDefault()
									.getResourceString(
											"CreateDeployPackagesJob.error.log.message"), //$NON-NLS-1$
							new Exception(exceptionMessage));
					throw new CoreException(
							new MultiStatus(
									DeployCorePlugin.getDefault()
											.getBundle().getSymbolicName(),
									-1,
									new IStatus[] { status},
									DeployCorePlugin
											.getDefault()
											.getResourceString(
													"CreateDeployPackagesJob.message.log.buildPackage"), //$NON-NLS-1$
									null));
				}
				else if (IMarker.SEVERITY_WARNING == severity) {
				}
				// ignore other severities
			}
		}
		else {
			DeployPackage dp = (DeployPackage) mRoot.getPackage().get(0);
			// special case for "on the fly" deployment
			IPackage ip = Utilities.adaptToIPackage(dp);
			if (ip instanceof ILogicalPackage) {
				ILogicalPackage ilp = (ILogicalPackage) ip;
				IPackageConstructorExtension ipce = getPackageConstructor(ilp,
						(DeployConfiguration) dp.getTargetConfiguration()
								.get(0));
				IStatus[] validateStatus = ipce
						.validatePackage(ilp, null, null);

				if ((validateStatus != null) && (validateStatus.length > 0)) {
					MultiStatus status = new MultiStatus(
							DeployCorePlugin.getDefault().getBundle()
									.getSymbolicName(),
							-1,
							validateStatus,
							DeployCorePlugin
									.getDefault()
									.getResourceString(
											"CreateDeployPackagesJob.message.log.buildPackage"), //$NON-NLS-1$
							null);
					if (status.getSeverity() == IStatus.ERROR) {
						throw new CoreException(status);
					}
				}
			}
		}

		monitor.worked(100);
	}

	private void createPackages(IProgressMonitor monitor) throws CoreException,
			InterruptedException {
		List errors = new ArrayList();

		try {
			monitor
					.subTask(DeployCorePlugin
							.getDefault()
							.getResourceString(
									"CreateDeployPackagesJob.subtask.createWorkingFolders")); //$NON-NLS-1$
			createTempFolders();
			monitor.worked(100);

			for (Iterator pkgIt = mRoot.getPackage().iterator(); pkgIt
					.hasNext();) {
				DeployPackage deployPackage = (DeployPackage) pkgIt.next();
				IPackage ip = Utilities.adaptToIPackage(deployPackage);

				if (ip instanceof ILogicalPackage) {
					ILogicalPackage ilp = (ILogicalPackage) ip;					
					for (Iterator trgIt = deployPackage
							.getTargetConfiguration().iterator(); trgIt
							.hasNext();) {
						DeployConfiguration deployConfig = (DeployConfiguration) trgIt
								.next();

						monitor
								.subTask(DeployCorePlugin
										.getDefault()
										.getResourceString(
												"CreateDeployPackagesJob.subtask.buildPackageForDeployment", //$NON-NLS-1$
												new Object[] {
														deployConfig
																.getSourcePackage()
																.getPackageFile(),
														deployConfig
																.getTargetServer()
																.getProfileName()}));

						IPackageConfiguration config = getPackageConfiguration(
								ilp, deployConfig);
						LogicalPackageConfigurationKey key = new LogicalPackageConfigurationKey(
								ilp, deployConfig.getTargetServer());
						if (mLogicalPackageToPackageDescriptorMap
								.containsKey(key)) {
							mTargetToPackageDescriptorMap.put(deployConfig,
									mLogicalPackageToPackageDescriptorMap
											.get(key));
						}
						else {
							IPackageCreationContext context = createPackageCreationContext(
									ilp, deployConfig);
							IPackageConstructor ipc = getPackageConstructor(
									ilp, deployConfig);
							try {
								// TODO: in the future, check to see if the
								// package already exists. if so, check to
								// see if it needs to be recreated.
								IPackageOutputDescriptor ipod = ipc
										.createPackage(ilp, context, config);
								mTargetToPackageDescriptorMap.put(deployConfig,
										ipod);
								mLogicalPackageToPackageDescriptorMap.put(key, ipod);
							}
							catch (CoreException e) {
								if (e.getStatus().getSeverity() == IStatus.CANCEL) {
									throw new InterruptedException();
								}
								MultiStatus status = new MultiStatus(
										DeployCorePlugin.getDefault()
												.getBundle().getSymbolicName(),
										-1,
										DeployCorePlugin
												.getDefault()
												.getResourceString(
														"CreateDeployPackagesJob.error.createPackageFailed", //$NON-NLS-1$
														new Object[] {
																ilp
																		.getFile()
																		.getFullPath(),
																deployConfig
																		.getTargetServer()
																		.getProfileName()}),
										e);
								status.add(e.getStatus());
								errors.add(status);
							}
							finally {
								monitor.worked(100);
								checkCancelled(monitor);
							}
						}
					}
				}
				else {
					File packageFile = ip.getFile().getLocation().toFile();
					IPackageOutputDescriptor ipod = new FilePackageOutputDescriptor(
							packageFile, null, ((IPhysicalPackage) ip)
									.getServerType(), packageFile.toString(),
							ip.getName());
					for (Iterator trgIt = deployPackage
							.getTargetConfiguration().iterator(); trgIt
							.hasNext();) {
						DeployConfiguration deployConfig = (DeployConfiguration) trgIt
								.next();
						monitor
								.subTask(DeployCorePlugin
										.getDefault()
										.getResourceString(
												"CreateDeployPackagesJob.subtask.processingPackage", //$NON-NLS-1$
												new Object[] {
														ip.getName(),
														deployConfig
																.getTargetServer()
																.getProfileName()}));
						mTargetToPackageDescriptorMap.put(deployConfig, ipod);
						monitor.worked(100);
						checkCancelled(monitor);
					}
				}
			}
		}
		catch (IOException e) {
			IStatus status = new Status(Status.ERROR, DeployCorePlugin
					.getDefault().getBundle().getSymbolicName(), -1,
					DeployCorePlugin.getDefault().getResourceString(
							"CreateDeployPackagesJob.error.log.message"), //$NON-NLS-1$
					e);
			IStatus mutiStatus = new MultiStatus(
					DeployCorePlugin.getDefault().getBundle()
							.getSymbolicName(),
					-1,
					new IStatus[] { status},
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"CreateDeployPackagesJob.message.log.buildPackage"), //$NON-NLS-1$
					null);
			throw new CoreException(mutiStatus);
		}
		finally {
			if (mTempFolder != null) {
				monitor
						.subTask(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"CreateDeployPackagesJob.subtask.deleteWorkingFolders")); //$NON-NLS-1$
				deleteTempFile(mTempFolder);
				monitor.worked(100);
			}
			if (errors.size() != 0) {
				IStatus mutiStatus = new MultiStatus(
						DeployCorePlugin.getDefault().getBundle()
								.getSymbolicName(),
						-1,
						(IStatus[]) errors.toArray(new IStatus[errors.size()]),
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"CreateDeployPackagesJob.error.errorsOcurredCreatingPackages"), null); //$NON-NLS-1$
				throw new CoreException(mutiStatus);
			}
		}
	}

	private void refreshWorkspace(IProgressMonitor monitor)
			throws CoreException {
		monitor = new SubProgressMonitor(monitor, 100);
		mDeployFile.getWorkspace().getRoot().refreshLocal(
				IResource.DEPTH_INFINITE, monitor);
	}

	private void deleteTempFile(File file) {
		File[] files = file.listFiles();
		if (files != null && files.length > 0) {
			for (int index = 0, count = files.length; index < count; ++index) {
				deleteTempFile(files[index]);
			}
		}
		// TODO: consider adding a log entry if delete() returns false
		//
		file.delete();
	}

	private IPackageConfiguration getPackageConfiguration(ILogicalPackage ilp,
			DeployConfiguration deployConfig) {
		IPackageConfiguration retVal;
		byte[] configData = deployConfig.getConfigOverride();
		if (ilp instanceof IConfigurablePackage) {
			if (configData == null || configData.length == 0) {
				retVal = null;
			}
			else {
				ByteArrayInputStream bais = new ByteArrayInputStream(configData);
				try {
					retVal = ((IConfigurablePackageExtension) ilp
							.getExtension()).getPackageConfigurationManager()
							.createPackageConfiguration(ilp, bais);
				}
				catch (IOException e) {
					e.printStackTrace();
					retVal = ((IConfigurablePackageExtension) ilp
							.getExtension()).getPackageConfigurationManager()
							.createPackageConfiguration(ilp);
				}
				finally {
					try {
						bais.close();
					}
					catch (IOException e) {
					}
				}
			}
		}
		else {
			if (configData != null && configData.length == 0) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"CreateDeployPackagesJob.error.configurationSpecifiedForLogicalPackage")); //$NON-NLS-1$
			}
			retVal = null;
		}
		return retVal;
	}

	private IPackageConstructorExtension getPackageConstructor(
			ILogicalPackage pkg, DeployConfiguration deployConfig) {
		IPackageConstructorExtension retVal;
		IDeployDriverExtension driver;
		IConnectionProfile profile = StpProfileManager.getInstance()
				.getProfileByName(
						deployConfig.getTargetServer().getProfileName());
		if (profile == null) {
			System.err
					.println(DeployCorePlugin
							.getDefault()
							.getResourceString(
									"CreateDeployPackagesJob.error.connectionProfileNotFound", //$NON-NLS-1$
									new Object[] { deployConfig
											.getTargetServer().getProfileName()}));
			driver = null;
		}
		else {
			driver = DeploymentExtensionManager.getInstance().getDeployDriver(
					pkg, profile.getProviderId());
		}
		if (driver == null) {
			System.err
					.println(DeployCorePlugin
							.getDefault()
							.getResourceString(
									"CreateDeployPackagesJob.error.deployDriverNotFound", //$NON-NLS-1$
									new Object[] { pkg.getFile().getFullPath()}));
			retVal = null;
		}
		else {
			retVal = driver.getPackageConstructor(pkg.getTechnologyType());
			if (retVal == null) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"CreateDeployPackagesJob.error.packageConstructorNotFound", //$NON-NLS-1$
										new Object[] {
												pkg.getFile().getFullPath(),
												driver.getID()}));
			}
		}
		return retVal;
	}

	private IPackageCreationContext createPackageCreationContext(
			ILogicalPackage ilp, DeployConfiguration deployConfig) {
		String childFolderName = deployConfig.getTargetServer()
				.getProfileName();
		IConnectionProfile target = Utilities
				.adaptToIConnectionProfile(deployConfig);
		File outputFolder = createFolder(DeploymentExtensionManager
				.getInstance().getOutputFolder(ilp, deployConfig).getLocation()
				.toFile());
		File tempFolder = createFolder(new File(new File(mTempFolder,
				childFolderName), ilp.getName()));

		if (target == null) {
			System.err
					.println(DeployCorePlugin
							.getDefault()
							.getResourceString(
									"CreateDeployPackagesJob.error.targetServerNotFound", //$NON-NLS-1$
									new Object[] { deployConfig
											.getTargetServer().getProfileName()}));
		}

		return new PackageCreationContext(target, outputFolder, tempFolder,
				mOverwriteCancelQuery);
	}

	private File createFolder(File folder) {
		if (!folder.exists()) {
			folder.mkdirs();
		}
		return folder;
	}

	private void createTempFolders() throws IOException {
		File tmpFile = File.createTempFile("deploy_temp", ".tmp"); //$NON-NLS-1$ //$NON-NLS-2$
		tmpFile.deleteOnExit();
		mTempFolder = new File(tmpFile.getParentFile(), tmpFile.getName()
				.substring(0, tmpFile.getName().lastIndexOf(".tmp"))); //$NON-NLS-1$
		mTempFolder.mkdir();
		mTempFolder.deleteOnExit();
	}

	private int getBuildCount() {
		int retVal = 2; // One for creating the temp folders, one for deleting
		// them
		for (Iterator it = mRoot.getPackage().iterator(); it.hasNext();) {
			retVal += ((DeployPackage) it.next()).getTargetConfiguration()
					.size();
		}
		return retVal;
	}

	private static class LogicalPackageConfigurationKey {

		private ILogicalPackage mLogicalPackage;
		private DeployServer mServer;

		public LogicalPackageConfigurationKey(ILogicalPackage ilp,
				DeployServer server) {
			mLogicalPackage = ilp;
			mServer = server;
		}

		public boolean equals(Object obj) {
			if (obj instanceof LogicalPackageConfigurationKey) {
				LogicalPackageConfigurationKey other = (LogicalPackageConfigurationKey) obj;
				return mLogicalPackage.equals(other.mLogicalPackage)
						&& ((mServer != null && mServer
								.equals(other.mServer)) || (mServer == null && other.mServer == null));
			}
			return false;
		}

		public int hashCode() {
			return mLogicalPackage.hashCode();
		}

	}
}
