/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.ui.viewers;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;

/**
 * Provides support for automatically updating checkstate of parents and
 * children.
 * 
 * @author rcernich
 * 
 * Created on Jan 28, 2004
 */
public class CheckboxTreeViewer extends
		org.eclipse.jface.viewers.CheckboxTreeViewer {

	public CheckboxTreeViewer(Composite parent) {
		super(parent);
	}

	/**
	 * @param parent
	 * @param style
	 */
	public CheckboxTreeViewer(Composite parent, int style) {
		super(parent, style);
	}

	/**
	 * @param tree
	 */
	public CheckboxTreeViewer(Tree tree) {
		super(tree);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.CheckboxTreeViewer#setCheckedElements(java.lang.Object[])
	 */
	public void setCheckedElements(Object[] elements) {
		super.setCheckedElements(elements);
		if (elements.length > 0) {
			for (int index = 0, count = elements.length; index < count; ++index) {
				handleItemChecked(elements[index], true);
			}
		}
		else {
			super.setGrayedElements(new Object[0]);
		}
	}

	/**
	 * Returns only those checked objects that have no children. This
	 * essentially filters container elements from the list of returned
	 * elements.
	 * 
	 * @return
	 */
	public Object[] getCheckedLeafElements() {
		List elements = new ArrayList();
		for (Iterator it = Arrays.asList(getCheckedElements()).iterator(); it
				.hasNext();) {
			Object element = it.next();
			Widget widget = internalExpand(element, false);
			Item[] kids = getChildren(widget);
			if (kids == null || kids.length == 0) {
				elements.add(element);
			}
		}
		return elements.toArray();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.CheckboxTreeViewer#fireCheckStateChanged(org.eclipse.jface.viewers.CheckStateChangedEvent)
	 */
	protected void fireCheckStateChanged(CheckStateChangedEvent event) {
		setSelection(new StructuredSelection(event.getElement()));
		handleItemChecked(event.getElement(), event.getChecked());
		super.fireCheckStateChanged(event);
	}

	// Check state handling
	public void handleItemChecked(Object element, boolean checked) {
		ITreeContentProvider tp = (ITreeContentProvider) getContentProvider();
		checkChildren(element, checked);
		grayChildren(element, false);
		grayParents(tp.getParent(element), true);
		setGrayed(element, false);
		if (checked) {
			checkParents(tp.getParent(element), true);
			determineParentGrayState(element);
		}
		else {
			determineParentCheckState(element);
		}
	}

	private void checkChildren(Object element, boolean state) {
		setSubtreeChecked(element, state);
	}

	private void checkParents(Object element, boolean state) {
		Widget widget = internalExpand(element, false);
		if (widget instanceof TreeItem) {
			for (TreeItem item = (TreeItem) widget; item != null; item = item
					.getParentItem()) {
				item.setChecked(state);
			}
		}
	}

	private void grayParents(Object element, boolean state) {
		if (element != null) {
			setParentsGrayed(element, state);
		}
	}

	private void grayChildren(Object element, boolean grayed) {
		Widget widget = internalExpand(element, false);
		if (widget instanceof TreeItem) {
			TreeItem item = (TreeItem) widget;
			item.setGrayed(grayed);
			setGrayedChildren(item, grayed);
		}
	}

	private void determineParentGrayState(Object element) {
		ITreeContentProvider tp = (ITreeContentProvider) getContentProvider();
		Object parent = tp.getParent(element);
		Widget widget = internalExpand(parent, false);

		Item sibs[] = getChildren(widget);
		if (sibs == null) {
			return;
		}

		boolean checked = parent != null;
		for (int index = 0, count = sibs.length; index < count && checked; ++index) {
			TreeItem sib = (TreeItem) sibs[index];
			checked = sib.getChecked() && !sib.getGrayed();
		}
		if (checked) {
			setGrayed(parent, false);
			determineParentGrayState(parent);
		}
	}

	private void determineParentCheckState(Object element) {
		ITreeContentProvider tp = (ITreeContentProvider) getContentProvider();
		Object parent = tp.getParent(element);
		Widget widget = internalExpand(parent, false);

		Object sibs[] = getChildren(widget);
		if (sibs == null) {
			return;
		}

		boolean notChecked = parent != null;
		for (int index = 0, count = sibs.length; index < count && notChecked; ++index) {
			TreeItem sib = (TreeItem) sibs[index];
			notChecked = !sib.getChecked();
		}
		if (notChecked) {
			handleItemChecked(parent, false);
		}
	}

	private void setGrayedChildren(TreeItem item, boolean grayed) {
		createChildren(item);
		Item[] items = getChildren(item);
		if (items != null) {
			for (int i = 0; i < items.length; i++) {
				Item it = items[i];
				if (it.getData() != null && (it instanceof TreeItem)) {
					TreeItem treeItem = (TreeItem) it;
					treeItem.setGrayed(grayed);
					setGrayedChildren(treeItem, grayed);
				}
			}
		}
	}

}