/*******************************************************************************
 * Copyright (c) 2003, 2005 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.core.infrastructure.emf;

import org.eclipse.core.resources.IProject;
import org.eclipse.stp.core.internal.infrastructure.emf.EditModelFactory;

/**
 * IEditModelFactory creates instances of IEditModels and IEditModelScribblers.
 * Clients may access the default implementation of the factory via
 * {@link #eINSTANCE}.
 * 
 * <p>
 * Clients are recommended to build most of the customization logic (such as
 * determining the interesting resources, and the appropriate label for their
 * edit models through the methods made available in
 * {@link org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler}. Then
 * clients may rely on
 * {@link #createScribblerForRead(IProject, IScribblerDomain)} and
 * {@link #createScribblerForWrite(IProject, IScribblerDomain)}.
 * </p>
 */
public interface IEditModelFactory {

   /**
    * The public singleton instance of the IEditModelFactory.
    */
   public static final IEditModelFactory eINSTANCE = EditModelFactory.init();

   /**
    * Return the IEditModel for the given project with the given label. The
    * editModelLabel parameter allows clients to coordinate access to a
    * particular edit model across editors, operations, and views. The label
    * should be something that is scoped to a particular collection of
    * resources, and even targeted to a specific kind of resources (for
    * instance, the EJB resources for a specific module contained by the given
    * project).
    * 
    * @param aProject
    *           The scope of the IEditModel; general IEditModels are scoped to a
    *           project
    * @param editModelLabel
    *           The customized, client-chosen label for the edit model.
    * @return A created (or cached) IEditModel
    */
   IEditModel getEditModel(IProject aProject, String editModelLabel);

   /**
    * Clients may create a scribbler using an array of IScribblerDomains for a
    * specific edit model (identified by {aProject, editModelLabel}. Clients can
    * also decide whether the returned scribbler should be made available for
    * write operations.
    * 
    * @param aProject
    *           Used with editModelLabel to identify the relevant IEditModel.
    * @param editModelLabel
    *           Used with aProject to identify the relevant IEditModel.
    * @param domains
    *           An array of IScribblerDomains that identify the interesting
    *           resources.
    * @param toMakeReadonly
    *           True indicates the returned scribbler should not allow save()
    *           operations.
    * @return An instance of IEditModelScribbler to be used to access any of the
    *         files described by the array of IScribblerDomains.
    * @throws EditModelException
    *            If the project is inAccessible() or if another problem occurs
    *            while locating the edit model or creating the scribbler.
    */
   IEditModelScribbler createScribbler(IProject aProject,
         String editModelLabel, IScribblerDomain[] domains,
         boolean toMakeReadonly) throws EditModelException;

   /**
    * Create a scribbler using the project and the given domain. The appropriate
    * IEditModel will be located using
    * {@link IScribblerDomain#getEditModelLabel()}. The scribbler returned will
    * not allow write operations.
    * 
    * @param aProject
    *           Used with {@link IScribblerDomain#getEditModelLabel()} to
    *           identify the relevant IEditModel.
    * @param domain
    *           An IScribblerDomain that will identify the interesting
    *           resources.
    * @return An instance of IEditModelScribbler to be used to access any of the
    *         files described domain.
    * @throws EditModelException
    *            If the project is inAccessible() or if another problem occurs
    *            while locating the edit model or creating the scribbler.
    */
   IEditModelScribbler createScribblerForRead(IProject aProject,
         IScribblerDomain domain) throws EditModelException;

   /**
    * Create a scribbler using the project and the given domain. The appropriate
    * IEditModel will be located using
    * {@link IScribblerDomain#getEditModelLabel()}. The scribbler returned will
    * allow write operations.
    * 
    * @param aProject
    *           Used with {@link IScribblerDomain#getEditModelLabel()} to
    *           identify the relevant IEditModel.
    * @param domain
    *           An IScribblerDomain that will identify the interesting
    *           resources.
    * @return An instance of IEditModelScribbler to be used to access any of the
    *         files described domain.
    * @throws EditModelException
    *            If the project is inAccessible() or if another problem occurs
    *            while locating the edit model or creating the scribbler.
    */
   IEditModelScribbler createScribblerForWrite(IProject aProject,
         IScribblerDomain domain) throws EditModelException;

   /**
    * Clients may create a scribbler using an array of IScribblerDomains for a
    * specific edit model (identified by {aProject, editModelLabel}. The
    * scribbler returned will not allow write operations.
    * 
    * @param aProject
    *           Used with editModelLabel to identify the relevant IEditModel.
    * @param editModelLabel
    *           Used with aProject to identify the relevant IEditModel.
    * @param domains
    *           An array of IScribblerDomains that identify the interesting
    *           resources.
    * @return An instance of IEditModelScribbler to be used to access any of the
    *         files described by the array of IScribblerDomains.
    * @throws EditModelException
    *            If the project is inAccessible() or if another problem occurs
    *            while locating the edit model or creating the scribbler.
    */
   IEditModelScribbler createScribblerForRead(IProject aProject,
         String editModelLabel, IScribblerDomain[] domains)
         throws EditModelException;

   /**
    * Clients may create a scribbler using an array of IScribblerDomains for a
    * specific edit model (identified by {aProject, editModelLabel}. The
    * scribbler returned will allow write operations.
    * 
    * @param aProject
    *           Used with editModelLabel to identify the relevant IEditModel.
    * @param editModelLabel
    *           Used with aProject to identify the relevant IEditModel.
    * @param domains
    *           An array of IScribblerDomains that identify the interesting
    *           resources.
    * @return An instance of IEditModelScribbler to be used to access any of the
    *         files described by the array of IScribblerDomains.
    * @throws EditModelException
    *            If the project is inAccessible() or if another problem occurs
    *            while locating the edit model or creating the scribbler.
    */
   IEditModelScribbler createScribblerForWrite(IProject aProject,
         String editModelLabel, IScribblerDomain[] domains)
         throws EditModelException;

}
