/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.compiler.wsdlmap;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.HashMap;

import org.eclipse.stp.b2j.core.jengine.internal.compiler.TranslatorLog;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.Util;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceException;
import org.eclipse.stp.b2j.core.jengine.internal.compiler.xmlns.NamespaceTranslator;
import org.eclipse.stp.b2j.core.misc.internal.XMLConfigUtil;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Document;
import org.eclipse.stp.b2j.core.xml.internal.w3c.DocumentBuilder;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Element;
import org.eclipse.stp.b2j.core.xml.internal.w3c.Node;
import org.eclipse.stp.b2j.core.xml.internal.w3c.NodeList;

/**
 * 
 * @author amiguel
 *
 * A class to represent a complete WSDL file, used in the translation 
 * of BPEL source files into engine Programs
 */
public class WSDLMap {

	String LOG_SOURCE = "WSDL Translator";
	TranslatorLog log;
	NamespaceTranslator nt;
	
	Util compiler_util;
	
	HashMap qname_to_portType = new HashMap();
	HashMap qname_to_message = new HashMap();
	HashMap qname_to_bpel_partnerLinkType = new HashMap();
	HashMap qname_to_bpel_property = new HashMap();
	ArrayList bpel_propertyAliasList = new ArrayList();

	public WSDLMessage getMessage(String qname) {
		return (WSDLMessage)qname_to_message.get(qname);	
	}

	public WSDLPortType getPortType(String qname) {
		return (WSDLPortType)qname_to_portType.get(qname);	
	}
	
	public BPELPartnerLinkType getPartnerLinkType(String qname) {
		return (BPELPartnerLinkType)qname_to_bpel_partnerLinkType.get(qname);
	}
	
	public BPELProperty getProperty(String qname) {
		return (BPELProperty)qname_to_bpel_property.get(qname);
	}
	
	public String[] getAllPropertyNames() {
		ArrayList list = new ArrayList();
		list.addAll(qname_to_bpel_property.keySet());
		String[] tmp = new String[list.size()];
		list.toArray(tmp);
		return tmp;
	}
	
	public BPELPropertyAlias[] getPropertyAliases() {
		BPELPropertyAlias[] tmp = new BPELPropertyAlias[bpel_propertyAliasList.size()];
		bpel_propertyAliasList.toArray(tmp);
		return tmp;
	}

	private void readPart(WSDLMessage m, Element part) throws NamespaceException, InvalidWsdlException {

		nt.addNamespaces(part);

		String name = part.getAttribute("name");
		String type = part.getAttribute("type");
		if (type.length() == 0) {
			type = part.getAttribute("element");
		}
		String qtype = nt.qualify(type,false);

		log.logInfo(LOG_SOURCE,"Found Message Part "+name+" ("+compiler_util.getQNameFromQName(qtype)+")");
//		B2jPlugin.DBG.info("Found Message Part "+m.qname+" -> "+name+" ("+qtype+")");
		m.part_to_xsdtype.put(name,qtype);

		nt.removeNamespaces(part);
	}
	
	private void readMessage(Element message) throws NamespaceException, InvalidWsdlException {

		nt.addNamespaces(message);

		String name = message.getAttribute("name");
		String qname = nt.qualify(name,true);
		
		WSDLMessage m = new WSDLMessage();
		m.qname = qname;
		
		log.logInfo(LOG_SOURCE,"message "+compiler_util.getQNameFromQName(qname));

		ArrayList parts = Util.getAllElements(message);
		for (int i = 0; i < parts.size(); i++) {
			Element elem = (Element)parts.get(i);
			
			if (NamespaceTranslator.getName(elem).equals("part")) {
				readPart(m, elem);	
			}	
		}
		
//		B2jPlugin.DBG.info("Found Message "+qname);
		qname_to_message.put(qname,m);

		nt.removeNamespaces(message);
	}

	private void readPortType(Element port) throws NamespaceException, InvalidWsdlException {

		nt.addNamespaces(port);

		String type = port.getAttribute("name");

		WSDLPortType portType = new WSDLPortType();
		portType.name = nt.qualify(type,true);
		
		log.logInfo(LOG_SOURCE,"portType "+compiler_util.getQNameFromQName(portType.name));
//		B2jPlugin.DBG.info("Found PortType "+portType.name);
		qname_to_portType.put(portType.name,portType);

		ArrayList ops = Util.getAllElements(port);
		for (int i = 0; i < ops.size(); i++) {
			Element op = (Element)ops.get(i);
			
			nt.addNamespaces(op);
			
			if (NamespaceTranslator.getName(op).equals("operation")) {
			
				String opname = op.getAttribute("name");

				WSDLOperation operation = new WSDLOperation();
				
				boolean input_detected = false;
				boolean output_detected = false;
		
				ArrayList msgs = Util.getAllElements(op);
				for (int k = 0; k < msgs.size(); k++) {
					Element msg = (Element)msgs.get(k);

					nt.addNamespaces(msg);

					String msgtag = NamespaceTranslator.getName(msg);
					String msgname = msg.getAttribute("name");
					String msgtype = msg.getAttribute("message");
					msgtype = nt.qualify(msgtype,false);

					if (msgtag.equals("input")) {
						input_detected = true;
						operation.message_types.put("input",msgtype);
						if (output_detected) {
							//output then input
							operation.type = WSDLOperation.TYPE_SOLICIT_RESPONSE;
						} else {
							//input only
							operation.type = WSDLOperation.TYPE_ONEWAY;
						}
					} else if (msgtag.equals("output")) {
						output_detected = true;
						operation.message_types.put("output",msgtype);
						if (input_detected) {
							//input then output
							operation.type = WSDLOperation.TYPE_REQUEST_RESPONSE;
						} else {
							//output
							operation.type = WSDLOperation.TYPE_NOTIFICATION;
						}
					} else if (msgtag.equals("fault")) {
						operation.message_types.put(msgname,msgtype);
					}
					
					nt.removeNamespaces(msg);
				}
			
				log.logInfo(LOG_SOURCE,"  operation "+opname);
//				B2jPlugin.DBG.info("Found Operation "+portType.name+" -> "+opname);
				portType.name_to_operation.put(opname,operation);
			
			}
			nt.removeNamespaces(op);
		}

		nt.removeNamespaces(port);
	}
	
	private void readPropertyAlias(Element pal) throws NamespaceException, InvalidWsdlException {
		nt.addNamespaces(pal);
		
		String propertyName = pal.getAttribute("propertyName");

		String messageType = pal.getAttribute("messageType");
		String messagePart = pal.getAttribute("part");
		//OR
		String xsdType = pal.getAttribute("type");
		//OR
		String xsdElement = pal.getAttribute("element");
		
		if (messageType != null) {
			if (messagePart == null) {
				throw new InvalidWsdlException("<propertyAlias> element with 'messageType' must also contain 'part' attribute");
			}
		} else if (xsdType != null) {
			//ok
		} else if (xsdElement != null) {
			//ok
		} else {
			throw new InvalidWsdlException("<propertyAlias> element must contain one of 'messageType'+'part', 'type' or 'element' attributes");
		}
		
		String query = null;
		
		ArrayList queries = Util.getAllElements(pal);
		
		for (int i = 0; i < queries.size(); i++) {
			Element elem = (Element)queries.get(i);
			
			if (NamespaceTranslator.getName(elem).equals("query")) {
				query = Util.getTextDirectlyUnder(elem).trim();
				break;
			}
		}
		
		if (query == null) {
			query = "";
//			throw new InvalidWsdlException("<propertyAlias ...> element must contain a <query> element");
		}
		
		
		BPELPropertyAlias alias = new BPELPropertyAlias();
		alias.qpropertyName = nt.qualify(propertyName,false);

		if (messageType != null) {
			if (messageType.length() != 0) {
				alias.qmessageType = nt.qualify(messageType,false);
				alias.messagePart = messagePart;
				log.logInfo(LOG_SOURCE,"BPEL property alias for "+compiler_util.getQNameFromQName(alias.qpropertyName)+" ("+compiler_util.getQNameFromQName(alias.qmessageType)+" -> "+alias.messagePart+")");
//				B2jPlugin.DBG.info("Found BPEL property alias for "+alias.qpropertyName+"  "+alias.qmessageType+" -> "+alias.messagePart);
			}
		}
		if (xsdType != null) {
			if (xsdType.length() != 0) {
				alias.qxsdType = nt.qualify(xsdType,false);
				log.logInfo(LOG_SOURCE,"BPEL property alias for "+compiler_util.getQNameFromQName(alias.qpropertyName)+" ("+compiler_util.getQNameFromQName(alias.qxsdType)+")");
//				B2jPlugin.DBG.info("Found BPEL property alias for "+alias.qpropertyName+"  "+alias.qxsdType);
			}
		}
		if (xsdElement != null) {
			if (xsdElement.length() != 0) {
				alias.qxsdElement = nt.qualify(xsdElement,false);
				log.logInfo(LOG_SOURCE,"BPEL property alias for "+compiler_util.getQNameFromQName(alias.qpropertyName)+" ("+compiler_util.getQNameFromQName(alias.qxsdElement)+")");
//				B2jPlugin.DBG.info("Found BPEL property alias for "+alias.qpropertyName+"  "+alias.qxsdElement);
			}
		}
		
		alias.query = query;
		
		bpel_propertyAliasList.add(alias);
		
		nt.removeNamespaces(pal);
	}
	
	private void readProperty(Element prop) throws NamespaceException, InvalidWsdlException {
		nt.addNamespaces(prop);
		
		String name = prop.getAttribute("name");
		String type = prop.getAttribute("type");
		
		BPELProperty varprop = new BPELProperty();
		varprop.qname = nt.qualify(name,true);
		varprop.qType = nt.qualify(type,false);
		
		log.logInfo(LOG_SOURCE,"BPEL variable property "+compiler_util.getQNameFromQName(varprop.qname));
//		B2jPlugin.DBG.info("Found BPEL variable property "+varprop.qname);
		qname_to_bpel_property.put(varprop.qname,varprop);
		
		nt.removeNamespaces(prop);
	}
	
	private void readPartnerLinkType(Element plink) throws NamespaceException, InvalidWsdlException {

		nt.addNamespaces(plink);

		String name = plink.getAttribute("name");
		
		BPELPartnerLinkType plinkType = new BPELPartnerLinkType();
		plinkType.name = nt.qualify(name,true);
		
		log.logInfo(LOG_SOURCE,"BPEL partner link type "+compiler_util.getQNameFromQName(plinkType.name));
//		B2jPlugin.DBG.info("Found BPEL partner link type "+plinkType.name);
		qname_to_bpel_partnerLinkType.put(plinkType.name,plinkType);
		
		ArrayList roles = Util.getAllElements(plink);
		
		for (int i = 0; i < roles.size(); i++) {
			Element role = (Element)roles.get(i);

			nt.addNamespaces(role);

			if (NamespaceTranslator.getName(role).equals("role")) {
				BPELPartnerLinkRole plinkRole = new BPELPartnerLinkRole();
				//note - this name is later referenced as an ncname, NOT as a qname, so
				//we dont qualify it with the current target namespace
				plinkRole.name = role.getAttribute("name");
				plinkRole.qportType = nt.qualify(role.getAttribute("portType"),false);

				log.logInfo(LOG_SOURCE,"  role "+plinkRole.name+" ("+compiler_util.getQNameFromQName(plinkRole.qportType)+")");
//				B2jPlugin.DBG.info("Found BPEL partner link type role "+plinkRole.name+" ("+plinkRole.qportType+")");
				plinkType.addRole(plinkRole);
				
				if (Util.getFirstElement(role,"portType") != null) {
					log.logWarning(LOG_SOURCE,"element <portType> under partner link type element <role> has been replaced with the attribute @portType on element <role>, the element <portType> will be ignored");
				}
			}
			
			nt.removeNamespaces(role);
		}

		nt.removeNamespaces(plink);
	}
	

	private void readMessages(Element definitions) throws NamespaceException, InvalidWsdlException {
		
		nt = compiler_util.createNamespaceTranslator();
		nt.addNamespaces(definitions);
		
		ArrayList elems = Util.getAllElements(definitions);

		for (int i = 0; i < elems.size(); i++) {
			Element elem = (Element)elems.get(i);
			String tagname = NamespaceTranslator.getName(elem);
			
			if (tagname.equals("message")) {
				readMessage(elem);
			}
		}
		nt.removeNamespaces(definitions);
	}

	private void readPortTypes(Element definitions) throws NamespaceException, InvalidWsdlException {
		nt = compiler_util.createNamespaceTranslator();
		nt.addNamespaces(definitions);
		
		ArrayList elems = Util.getAllElements(definitions);

		for (int i = 0; i < elems.size(); i++) {
			Element elem = (Element)elems.get(i);
			String tagname = NamespaceTranslator.getName(elem);
			
			if (tagname.equals("portType")) {
				//Add additional information to the Operation classes
				readPortType(elem);
			}
		}
		nt.removeNamespaces(definitions);
	}

	private void readPartnerLinkTypes(Element definitions) throws NamespaceException, InvalidWsdlException {
		nt = compiler_util.createNamespaceTranslator();
		nt.addNamespaces(definitions);
		
		NodeList list = definitions.getChildNodes();
		for (int i = 0; i < list.getLength(); i++) {
			Node n = (Node)list.item(i);
			System.out.println(i+": "+n.getClass().getName()+" = "+n.getNodeName());
		}
		
		ArrayList elems = Util.getAllElements(definitions);

		for (int i = 0; i < elems.size(); i++) {

			Element elem = (Element)elems.get(i);
			String tagname = NamespaceTranslator.getName(elem);

			if (tagname.equals("partnerLinkType")) {
				readPartnerLinkType(elem);
			}
		}
		
		nt.removeNamespaces(definitions);
	}
	
	private void readPropertyAliases(Element definitions) throws NamespaceException, InvalidWsdlException {
		nt = compiler_util.createNamespaceTranslator();
		nt.addNamespaces(definitions);
		
		ArrayList elems = Util.getAllElements(definitions);
		
		for (int i = 0; i < elems.size(); i++) {
			
			Element elem = (Element)elems.get(i);
			String tagname = NamespaceTranslator.getName(elem);
			
			if (tagname.equals("propertyAlias")) {
				readPropertyAlias(elem);
			}
		}
		
		nt.removeNamespaces(definitions);
	}
	
	private void readProperties(Element definitions) throws NamespaceException, InvalidWsdlException {
		nt = compiler_util.createNamespaceTranslator();
		nt.addNamespaces(definitions);
		
		ArrayList elems = Util.getAllElements(definitions);
		
		for (int i = 0; i < elems.size(); i++) {
			
			Element elem = (Element)elems.get(i);
			String tagname = NamespaceTranslator.getName(elem);
			
			if (tagname.equals("property")) {
				readProperty(elem);
			}
		}
		
		nt.removeNamespaces(definitions);
	}


	public WSDLMap(Util compiler_util, TranslatorLog log, String[] wsdlxml) throws Exception {
		this.log = log;
		this.compiler_util = compiler_util;
		Document[] docs = new Document[wsdlxml.length];
		
		nt = compiler_util.createNamespaceTranslator();
		 
		for (int i = 0; i < wsdlxml.length; i++) {
			DocumentBuilder builder = XMLConfigUtil.getDocumentBuilder();
			Document doc = builder.parse(new ByteArrayInputStream(wsdlxml[i].getBytes()));
	
			docs[i] = doc;
		}
		for (int i = 0; i < docs.length; i++) {
			Element definitions = docs[i].getDocumentElement();
			readMessages(definitions);			
		}
		for (int i = 0; i < docs.length; i++) {
			Element definitions = docs[i].getDocumentElement();
			readPortTypes(definitions);			
		}
		for (int i = 0; i < docs.length; i++) {
			Element definitions = docs[i].getDocumentElement();
			readPartnerLinkTypes(definitions);			
		}
		for (int i = 0; i < docs.length; i++) {
			Element definitions = docs[i].getDocumentElement();
			readProperties(definitions);			
		}
		for (int i = 0; i < docs.length; i++) {
			Element definitions = docs[i].getDocumentElement();
			readPropertyAliases(definitions);			
		}
	}
	
}