/**********************************************************************
 * Copyright (c) 2006 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.utils;

import java.util.HashMap;
import java.util.Random;

public class Cache {
	HashMap map = new HashMap();
	LinkedCacheList list = new LinkedCacheList();
	
	int MAX_SIZE = 50;
	
	double hits = 0;
	double misses = 0;
	
	public double getTotalRequests() {
		return hits+misses;
	}
	public double getTotalHits() {
		return hits;
	}
	public double getTotalMisses() {
		return misses;
	}
	public double getHitRatePercent() {
		return 100 * (hits / (hits+misses));
	}
	
	public Cache() {	
	}
	public Cache(int max_size) {
		this.MAX_SIZE = max_size;
	}
	
	public void clear() {
		map = new HashMap();
		list = new LinkedCacheList();
	}
	
	public int size() {
		return map.size();
	}

	public void addToCache(Object key, Object value) {
//		System.out.println("AddToCache: KEY "+key+", VALUE "+value);
		
		LinkedEntry e = (LinkedEntry)map.get(key);
		if (e != null) {
			//already in the cache - just bring it to the front (or do nothing at all?)
//			list.moveToFront(e);
		} else {
			map.put(key,list.add(key,value));
			
			if (map.size() > MAX_SIZE) {
				LinkedEntry rm = list.removeFirst();
	//			System.out.println("Removing First: KEY "+rm.valueKey+", VALUE "+rm.valueValue);
				rm.next = null;
				rm.prev = null;
				if (rm != null) {
					map.remove(rm.valueKey);
				}
			}
		}
	}
	
	public Object getFromCache(Object key) {
		LinkedEntry e = (LinkedEntry)map.get(key);
		
		if (e == null) {
			//cache miss
			misses++;
			
			return null;
		}
		
		//cache hit, boost e to front of list
		hits++;
		
		list.moveToFront(e);
		
		return e.valueValue;
	}
	
	class LinkedCacheList {
		
		LinkedEntry start;
		LinkedEntry end;
		
		public LinkedEntry add(Object key, Object value) {
			LinkedEntry e = new LinkedEntry();
			e.valueKey = key;
			e.valueValue = value;
			e.next = null;
			
			if (end == null) {
				//no elements
				start = e;
				end = e;
				e.prev = null;
			} else {
				//append
				end.next = e;
				e.prev = end;
				end = e;
			}
			return e;
		}
		
		public LinkedEntry removeFirst() {
			if (start != null) {
				if (start == end) {
					//size one
					end = null;
				}
					
				LinkedEntry ret = start;
				start = start.next;
				if (start != null) {
					start.prev = null;
				}
				return ret;
			} else {
				return null;
			}
		}

		public void moveToFront(LinkedEntry e) {
//			System.out.println("--- BEFORE MOVE TO FRONT ("+e+", "+e.prev+"<-e->"+e.next+") ---");
//			dump();
			if (e != end) {
				if (e == start) {
					start = e.next;
					start.prev = null;
				} else {
					e.prev.next = e.next;
					e.next.prev = e.prev;
				}
				end.next = e;
				e.prev = end;
				e.next = null;
				end = e;
			}
//			System.out.println("--- AFTER MOVE TO FRONT ("+e+", "+e.prev+"<-e->"+e.next+") ---");
//			dump();
		}
		
/*		void dump() {
			LinkedEntry e = start;
			System.out.println("START: "+start.valueKey+" / "+start.valueValue);
			System.out.println("END: "+end.valueKey+" / "+end.valueValue);
			while (e != null) {
				System.out.println("ENTRY: "+e.valueKey+" ("+e.prev+"<-e->"+e.next+")");
				e = e.next;
			}
		}*/
	}
	
	class LinkedEntry {
		Object valueKey;
		Object valueValue;
		LinkedEntry prev;
		LinkedEntry next;
		
		public String toString() {
			return ""+valueKey;
		}
	}
	
	public static void main(String[] args) throws Exception {
		Cache cache = new Cache(0);
		
		Random rand = new Random(5);
		
		int hits = 0;
		int misses = 0;
		
		double COUNT = 2000000;
		
		double t = System.currentTimeMillis();
		
		for (int i = 0; i < COUNT; i++) {
			Integer key = new Integer(rand.nextInt(15));
			
//			System.out.println(i+" == "+key);
			
			String value = (String)cache.getFromCache(key);
			
			if (value == null) {
				misses++;
				
//				System.out.println("CACHE MISS "+key);
				value = "#"+key;
				cache.addToCache(key,value);
			} else {
				hits++;
				
//				System.out.println("CACHE HIT "+key);
			}
		}
		
		t = System.currentTimeMillis()-t;
		
		System.out.println(hits+" hits");
		System.out.println(misses+" misses");
		System.out.println("Cache size: "+cache.size());
		
		System.out.println(COUNT+" lookups in "+t+"ms");
		System.out.println((t/COUNT)+" ms per lookups");
	}
}
