/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.ui.xef.schema;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.XMLConstants;

import org.apache.xerces.dom.DOMInputImpl;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSImplementation;
import org.apache.xerces.xs.XSLoader;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSModelGroup;
import org.apache.xerces.xs.XSNamedMap;
import org.apache.xerces.xs.XSNamespaceItem;
import org.apache.xerces.xs.XSNamespaceItemList;
import org.apache.xerces.xs.XSObject;
import org.apache.xerces.xs.XSObjectList;
import org.apache.xerces.xs.XSParticle;
import org.apache.xerces.xs.XSTerm;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.stp.xef.ISchemaProvider;
import org.eclipse.stp.xef.util.InputStreamHelper;

import org.w3c.dom.DOMConfiguration;
import org.w3c.dom.DOMError;
import org.w3c.dom.DOMErrorHandler;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;


/**
 * This class parses XML Schemas and turns them into {@link org.eclipse.stp.ui.xef.schema.SchemaElement} objects.
 * The parser tries to find elements in the schema that are considered top-level. 
 * Top level elements aren't nested inside other elements.
 */
public class SchemaRegistry {
    static {
        System.setProperty(DOMImplementationRegistry.PROPERTY, "org.apache.xerces.dom.DOMXSImplementationSourceImpl");            
    }

    private static final SchemaRegistry SINGLETON = new SchemaRegistry();

    private final SchemaParser parser;
    private Map<String, List<SchemaElement>> schemas = new HashMap<String, List<SchemaElement>>();

    /**
     * Constructor, however take care to share objects of this type. 
     * Therefore consider the static {@link #getDefault()} method in this class.
     */
    public SchemaRegistry() {
        try {
            parser = new SchemaParser();
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }
    
    /**
     * Obtain a default shared singleton instance.
     * @return The singleton.
     */
    public static SchemaRegistry getDefault() {
        return SINGLETON;
    }
    
    /**
     * Clears the schema cache.
     */
    public void clear() {
        schemas = new HashMap<String, List<SchemaElement>>();
    }
    
    /**
     * Resolve the schema elements from the given URL. 
     * @param url The URL that holds the XML-Schema document.
     * @param providers These are used when an imported schema is referenced in a schema. Imported schemas are looked up through their namespace. 
     * @return A list of the top-level schema elements.
     * @throws Exception In case of trouble loading the schema.
     */
    public List<SchemaElement> resolveSchemaFromURL(String url, ISchemaProvider ... providers) throws Exception {
        return parser.parseAndRegister(url, providers); 
    }
    
    /**
     * Resolve the schema from an XML string.
     * @param xml The string holding the actual XML-Schema document.
     * @param cache Whether or not to cache this schema. In certain (testing) cases not caching might be useful.
     * @param providers These are used when an imported schema is referenced in a schema. Imported schemas are looked up through their namespace. 
     * @return A list of the top-level schema elements.
     * @throws Exception In case of trouble loading the schema.
     */
    public List<SchemaElement> resolveSchemaFromXML(String xml, boolean cache, ISchemaProvider ... providers) throws Exception {
        File temp = File.createTempFile("temp", "xsd");
        try {
            OutputStream os = new FileOutputStream(temp);
            try {
                InputStreamHelper.drain(new ByteArrayInputStream(xml.getBytes()), os);
            } finally {
                os.close();
            }
            
            List<SchemaElement> elements = new LinkedList<SchemaElement>();
            String ns = parser.parse(temp.toURI().toString(), elements, providers);
            if (cache) {
                if (!schemas.containsKey(ns)) {
                    registerSchema(ns, elements);
                } else {
                    elements = schemas.get(ns); // use the one previously registered
                }
            }
            return elements;
        } finally {
            temp.delete();
        }
    }
    
    private void registerSchema(String uri, List<SchemaElement> topElements) {
        schemas.put(uri, topElements);
    }
    
    /**
     * Look up a schema by namespace, resolving it through the providers if needed.
     * @param uri The namespace of the schema to look up.
     * @param resolve If <tt>true</tt> resolve the schema through the providers.
     * @param cache Whether or not to cache this schema. In certain (testing) cases not caching might be useful.
     * @param providers Use to resolve schemas that are requested but not yet loaded. 
     * @return A list of the top-level schema elements.
     */
    public List<SchemaElement> getEntryElements(String uri, boolean resolve, boolean cache, ISchemaProvider... providers) {
        List<SchemaElement> els = null; 
        if (cache) {
            els = schemas.get(uri);
        }
        
        if (els == null && resolve) {
            // See if we can download this schema from the URI specified
            try {
                for (ISchemaProvider provider : providers) {
                    String schema = provider.getSchema(uri);
                    if (schema != null) {
                        return resolveSchemaFromXML(schema, cache, providers);
                    }
                }
                return resolveSchemaFromURL(uri, providers);
            } catch (Exception e) {
                e.printStackTrace(); // TODO log properly
            }
        }
        return els;
    }
    
    /**
     * Look up a particular SchemaElement through the namespace of the XML-Schema and the name of the element required. 
     * @param uri The namespace of the schema to look up.
     * @param name The name of the element to look up.
     * @param cache Whether or not to cache this schema. In certain (testing) cases not caching might be useful.
     * @param providers Use to resolve schemas that are requested but not yet loaded. 
     * @return A list of the top-level schema elements.
     */
    public SchemaElement getSchemaElement(String uri, String name, boolean cache, ISchemaProvider... providers) {
        List<SchemaElement> els = getEntryElements(uri, true, cache, providers); 
        if (els != null) {
            for (SchemaElement el : els) {
                if (el.getName().equals(name)) {
                    return el;
                }
            }
        }
        return null;
    }

    private final class SchemaParser implements DOMErrorHandler {        
        private final DOMImplementationRegistry registry;
        private final XSImplementation xsImpl;
        private final XSLoader schemaLoader;
        private final DOMConfiguration domConfig;
        private final SchemaProviderResourceResolver resolver;
        

        private SchemaParser() throws ClassCastException, ClassNotFoundException, InstantiationException, IllegalAccessException {
            registry = DOMImplementationRegistry.newInstance();
            xsImpl = (XSImplementation) registry.getDOMImplementation("XS-Loader");
            schemaLoader = xsImpl.createXSLoader(null);
            resolver = new SchemaProviderResourceResolver();
            domConfig = schemaLoader.getConfig();            

            domConfig.setParameter("error-handler", this);
            domConfig.setParameter("validate", Boolean.TRUE);
            domConfig.setParameter("resource-resolver", resolver);
        }
        
        private synchronized String parse(String uri, List<SchemaElement> elements, ISchemaProvider ... providers) {
            
            try {
                resolver.setProviders(providers);
                XSModel model = schemaLoader.loadURI(uri);
                if (model == null) {
                    throw new NullPointerException("No document found at " + uri);
                }
    
                elements.addAll(initTopLevelEntryElements(model));
    
                XSNamespaceItemList nsItems = model.getNamespaceItems();
                for (int i = 0; i < nsItems.getLength(); i++) {
                    XSNamespaceItem nsItem = nsItems.item(i);
                    if (XMLConstants.W3C_XML_SCHEMA_NS_URI.equals(nsItem.getSchemaNamespace())) {
                        continue; // This is the Schema namespace, we'll ignore that
                    }
                    return nsItem.getSchemaNamespace();
                }
                return null;
            } finally {
                resolver.setProviders(new ISchemaProvider[] {});                
            }
        }
        
        private synchronized List<SchemaElement> parseAndRegister(String uri, ISchemaProvider ... providers) {
            List<SchemaElement> entryElements = new ArrayList<SchemaElement>();
            String ns = parse(uri, entryElements, providers);
            registerSchema(ns, entryElements);
            return entryElements;
        }

        public boolean handleError(DOMError err) {
            System.out.println("*** Error " + err.getMessage());
            ((Throwable) err.getRelatedException()).printStackTrace();
            return false;
        }

        // The following 3 initTopLevelEntryElements are virtually identical to the 
        // getTopLevelEntryElements methods in 
        //     org.eclipse.stp.soa.configgen.tools.GenerateModel. 
        // It would be good if they could be shared
        // somehow, but beware that the org.eclipse.stp.ui.xef plugin shouldn't really have deep 
        // dependencies.
        private List<SchemaElement> initTopLevelEntryElements(XSModel model) {
            List<XSElementDeclaration> topElements = new LinkedList<XSElementDeclaration>();
            Set<XSElementDeclaration> referencedElements = new HashSet<XSElementDeclaration>();
            
            XSNamedMap map = model.getComponents(XSConstants.ELEMENT_DECLARATION);
            for (int i = 0; i < map.getLength(); i++) {
                XSElementDeclaration element = (XSElementDeclaration) map.item(i);
                topElements.add(element);
                initTopLevelEntryElements(element, referencedElements);
            }
            
            for (XSElementDeclaration referenced : referencedElements) {
                topElements.remove(referenced);
            }
            
            List<SchemaElement> l = new ArrayList<SchemaElement>(topElements.size());
            for (XSElementDeclaration element : topElements) {
                SchemaElement se = new SchemaElement(element, null, null);
                l.add(se);
            }
            return l;
        }

        private void initTopLevelEntryElements(XSElementDeclaration element,
                Set<XSElementDeclaration> referencedElements) {
            XSTypeDefinition typeDef = element.getTypeDefinition();
            if (typeDef instanceof XSComplexTypeDefinition) {
                XSComplexTypeDefinition ctd = (XSComplexTypeDefinition) typeDef;
                XSParticle groupParticle = ctd.getParticle();
                if (groupParticle != null) {
                    XSTerm groupTerm = groupParticle.getTerm();
                    if (groupTerm instanceof XSModelGroup) {
                        XSModelGroup group = (XSModelGroup) groupTerm;
                        XSObjectList elements = group.getParticles();
                        initTopLevelEntryElements(elements, referencedElements);
                    }
                }
            }
        }
        
        private void initTopLevelEntryElements(XSObjectList elements, 
            Set<XSElementDeclaration> referencedElements) {
            for (int i = 0; i < elements.getLength(); i++) {
                XSObject obj = elements.item(i);
                if (obj instanceof XSParticle) {
                    XSParticle particle = (XSParticle) obj;
                    XSTerm particleTerm = particle.getTerm();
                    if (particleTerm instanceof XSElementDeclaration) {
                        XSElementDeclaration referencedElement = (XSElementDeclaration) particleTerm;
                        if (referencedElements.contains(referencedElement)) {
                            continue;
                        }
                        referencedElements.add(referencedElement);
                        initTopLevelEntryElements(referencedElement, referencedElements);
                    } else if (particleTerm instanceof XSModelGroup) {
                        XSModelGroup group = (XSModelGroup) particleTerm;
                        XSObjectList groupElements = group.getParticles();
                        initTopLevelEntryElements(groupElements, referencedElements);
                    }
                }
            }
        }
    }
    
    private static class SchemaProviderResourceResolver implements LSResourceResolver {
        private ISchemaProvider [] providers = new ISchemaProvider [] {};
        
        void setProviders(ISchemaProvider ... p) {
            providers = p;
        }
        
        public LSInput resolveResource(String type, String namespaceURI, String publicId, String systemId, String baseURI) {
            for (ISchemaProvider p : providers) {
                String schema = p.getSchema(namespaceURI);
                if (schema != null) {
                    return new DOMInputImpl(publicId, systemId, baseURI, schema, null);
                }
            }
            return null;
        }        
    }
}
