/******************************************************************************
 * Copyright (c) 2006, Intalio Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Intalio Inc. - initial API and implementation
 *******************************************************************************/

/** 
 * Date             Author              Changes 
 * 14 Nov 2006      MPeleshchyshyn      Created 
 **/
package org.eclipse.stp.bpmn.diagram.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.Request;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.tools.ToolUtilities;
import org.eclipse.gmf.runtime.common.core.command.CommandResult;
import org.eclipse.gmf.runtime.diagram.ui.commands.ICommandProxy;
import org.eclipse.gmf.runtime.diagram.ui.editparts.IGraphicalEditPart;
import org.eclipse.gmf.runtime.diagram.ui.editparts.ShapeNodeEditPart;
import org.eclipse.gmf.runtime.diagram.ui.requests.EditCommandRequestWrapper;
import org.eclipse.gmf.runtime.diagram.ui.requests.RequestConstants;
import org.eclipse.gmf.runtime.emf.commands.core.command.AbstractTransactionalCommand;
import org.eclipse.gmf.runtime.emf.type.core.requests.DestroyElementRequest;
import org.eclipse.gmf.runtime.notation.Bounds;
import org.eclipse.gmf.runtime.notation.Edge;
import org.eclipse.gmf.runtime.notation.Location;
import org.eclipse.gmf.runtime.notation.Node;
import org.eclipse.gmf.runtime.notation.View;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.stp.bpmn.Activity;
import org.eclipse.stp.bpmn.Artifact;
import org.eclipse.stp.bpmn.Graph;
import org.eclipse.stp.bpmn.SequenceEdge;
import org.eclipse.stp.bpmn.SubProcess;
import org.eclipse.stp.bpmn.Vertex;
import org.eclipse.stp.bpmn.diagram.edit.parts.SequenceEdgeEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.SubProcessEditPart;
import org.eclipse.stp.bpmn.diagram.edit.parts.SubProcessSubProcessBodyCompartmentEditPart;
import org.eclipse.stp.bpmn.diagram.part.BpmnDiagramEditorPlugin;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;

/**
 * This class implements functionality that moves tasks outside sub-process.
 * Subpocess itself is removed.
 * 
 * @author MPeleshchyshyn
 * @author <a href="http://www.intalio.com">&copy; Intalio, Inc.</a>
 */
public class UngroupAction extends AbstractGroupUngroupAction {
    public static final String ACTION_ID = "ungroupAction";

    public static final String TOOLBAR_ACTION_ID = "toolbarUngroupAction";

    protected static final String ICON_PATH = "icons/Ungroup.gif";

    protected static final String ACTION_TEXT = "Ungroup";

    protected static final String TOOLTIP_TEXT = "Ungroup Shapes(Remove Sub-Process)";

    /**
     * Holds processes to which children should be moved outside.
     */
    private List<SubProcessEditPart> subProcesses = new ArrayList<SubProcessEditPart>();

    public UngroupAction(IWorkbenchPage workbenchPage) {
        super(workbenchPage);
    }

    public UngroupAction(IWorkbenchPart workbenchPart) {
        super(workbenchPart);
    }

    private static UngroupAction createActionWithoutId(
            IWorkbenchPage workbenchPage) {
        UngroupAction action = new UngroupAction(workbenchPage);
        action.setText(ACTION_TEXT);
        action.setToolTipText(TOOLTIP_TEXT);
        action.setImageDescriptor(BpmnDiagramEditorPlugin.
                getBundledImageDescriptor(ICON_PATH));

        return action;
    }

    public static UngroupAction createUngroupAction(IWorkbenchPage workbenchPage) {
        UngroupAction action = createActionWithoutId(workbenchPage);
        action.setId(ACTION_ID);

        return action;
    }

    public static UngroupAction createToolbarUngroupAction(
            IWorkbenchPage workbenchPage) {
        UngroupAction action = createActionWithoutId(workbenchPage);
        action.setId(TOOLBAR_ACTION_ID);

        return action;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.ui.IActionDelegate#run(org.eclipse.jface.action.IAction)
     */
    @Override
    protected void doRun(IProgressMonitor progressMonitor) {
    	SubProcessEditPart p = subProcesses.get(0);
    	
    	AbstractTransactionalCommand undoable = 
    		new AbstractTransactionalCommand(p.getEditingDomain(), 
    				"Ungrouping shapes", null) {

    		@Override
    		protected CommandResult doExecuteWithResult(
    				IProgressMonitor monitor, IAdaptable info) throws ExecutionException {
    			for (SubProcessEditPart subProcess : subProcesses) {
    				executeUngroupCommand(subProcess);
    			}
    			return CommandResult.newOKCommandResult();
			}};
    	
       
        
        p.getDiagramEditDomain().getDiagramCommandStack().execute(
        		new ICommandProxy(undoable));
        
    }

    /**
     * Method that ungroups shapes ina  subprocess passed
     * as an argument.
     * Should be executed in a transaction.
     * @param spEditPart
     */
    private void executeUngroupCommand(SubProcessEditPart spEditPart) {
    	SubProcessSubProcessBodyCompartmentEditPart tempComp = null;
    	for (Object ep : spEditPart.getChildren()) {
    		if (ep instanceof SubProcessSubProcessBodyCompartmentEditPart) {
    			tempComp = (SubProcessSubProcessBodyCompartmentEditPart) ep;
    			break;
    		}
    	}
    	assert(tempComp != null); // ugly and necessary.
    	SubProcessSubProcessBodyCompartmentEditPart compartmentEditPart = tempComp;

    	// get the children views.
    	List<EObject> semantic = new LinkedList<EObject>();
    	List<View> views = new LinkedList<View>();
    	
//   	 we collect the edit parts that will be removed
//   	 to refresh them explicitly
   	List removed = new LinkedList();
   	
    	for (Object part : compartmentEditPart.getChildren()) {
    		semantic.add(((IGraphicalEditPart) part).resolveSemanticElement());
    		if (((IGraphicalEditPart) part).getModel() instanceof Node) {
    			views.add((View) ((IGraphicalEditPart) part).getModel());
    			removed.addAll(((IGraphicalEditPart) part).getSourceConnections());
    			removed.addAll(((IGraphicalEditPart) part).getTargetConnections());
    		} 
    	}
    	

    	
    	Set<EObject> edges = new HashSet<EObject>();
    	for (EObject v : semantic) {
    		if (v instanceof Vertex) {
    			edges.addAll(((Vertex) v).getIncomingEdges());
    			edges.addAll(((Vertex) v).getOutgoingEdges());
    			
    		}
    	}
    	
    	semantic.addAll(edges);
    	// really not interested into empty subprocesses.
    	if (semantic.isEmpty()) {
    		return;
    	}
    	// get the location of the subprocess
    	Location location = (Location) ((Node) spEditPart.
    			getNotationView()).getLayoutConstraint();
    	// resolve the semantic element that will be the new container.
    	View targetContainer = ((IGraphicalEditPart) spEditPart.
    			getParent()).getNotationView();

    	// populate the list of children without incoming or outgoing edges
    	List<IGraphicalEditPart> incoming = new LinkedList<IGraphicalEditPart>();
    	List<IGraphicalEditPart> outgoing = new LinkedList<IGraphicalEditPart>();
    	getStartAndEndTasks(spEditPart, outgoing, incoming);

    	// before moving the activities, we remove their edges
    	// so that the diagram is valid all the time.
    	Map<Activity, List<SequenceEdge>> inEdges = new HashMap<Activity, List<SequenceEdge>>();
		Map<Activity, List<SequenceEdge>> outEdges = new HashMap<Activity, List<SequenceEdge>>();
		for (EObject obj : semantic) {
			if (obj instanceof Activity) {
				Activity act = (Activity) obj;
				inEdges.put(act, new ArrayList<SequenceEdge>(act.getIncomingEdges()));
				outEdges.put(act, new ArrayList<SequenceEdge>(act.getOutgoingEdges()));
				act.getIncomingEdges().clear();
				act.getOutgoingEdges().clear();
			}
		}
		
    	for (EObject obj : semantic) {
    		if (obj instanceof Activity) {
    			((Activity) obj).setGraph((Graph) targetContainer.getElement());
    		} else if (obj instanceof SequenceEdge) {
                ((Graph) targetContainer.getElement()).getSequenceEdges().add((SequenceEdge)obj);
            } else if (obj instanceof Artifact) {
                ((Graph) targetContainer.getElement()).getArtifacts().add((Artifact)obj);
            } else {
              //this looks very bad to me: ie unlikely to do any good.
    			targetContainer.getElement().eContents().add(obj);
    		}
    	}
    	
    	for (EObject obj : semantic) {
			if (obj instanceof Activity) {
				Activity act = (Activity) obj;
				act.getIncomingEdges().addAll(inEdges.get(act));
				act.getOutgoingEdges().addAll(outEdges.get(act));
			}
		}
    	
    	for (View v : views) {
    		spEditPart.getNotationView().removeChild(v);
    		targetContainer.insertChild(v);
    		if (v instanceof Node) {
    			Bounds constraint = (Bounds) ((Node) v).getLayoutConstraint();
    			constraint.setX(constraint.getX() + location.getX());
    			constraint.setY(constraint.getY() + location.getY());
    			((Node) v).setLayoutConstraint(constraint);
    		}
    	}

    	Command command = spEditPart.getParent().getCommand(new Request(RequestConstants.REQ_REFRESH)); 
    	command.execute();

    	// retargeting connections.


    	if (incoming.size() == 1) { 
    		// limiting the reconnection case to 1 shape.
    		for (Object edge : spEditPart.getTargetConnections()) {
    			if (!(edge instanceof SequenceEdgeEditPart)) {
    				continue;
    			}
    			SequenceEdgeEditPart sEdge = (SequenceEdgeEditPart) edge;
    			for (IGraphicalEditPart part : incoming) {
    				((SequenceEdge) sEdge.resolveSemanticElement()).
    					setTarget((Vertex) part.resolveSemanticElement());
    				((Edge) sEdge.getNotationView()).
    					setTarget(part.getNotationView());
    			}
    		}
    	}
    	if (outgoing.size() == 1) { 
    		// limiting the reconnection case to 1 shape.
    		for (Object edge : spEditPart.getSourceConnections()) {
    			if (!(edge instanceof SequenceEdgeEditPart)) {
    				continue;
    			}
    			SequenceEdgeEditPart sEdge = (SequenceEdgeEditPart) edge;
    			for (IGraphicalEditPart part : outgoing) {
    				((SequenceEdge) sEdge.resolveSemanticElement()).
    					setSource((Vertex) part.resolveSemanticElement());
    				((Edge) sEdge.getNotationView()).
    					setSource(part.getNotationView());
    			}
    		}
    	}

    	command = spEditPart.getParent().getCommand(new Request(RequestConstants.REQ_REFRESH)); 
    	command.execute();
    	
    	EditPart parent = spEditPart.getParent();
    	// finally destroy the subprocess.
    	DestroyElementRequest request = new DestroyElementRequest(spEditPart.resolveSemanticElement(), false);
    	Command destroy = spEditPart.getCommand(new EditCommandRequestWrapper(request));
    	destroy.execute();
    	
    	// refreshing the result, so that the edit parts are removed.
    	command = parent.getCommand(new Request(RequestConstants.REQ_REFRESH)); 
    	command.execute();
    	
    	for (Object r : removed) {
    		((EditPart) r).refresh();
    		 ((EditPart) r).activate();
    	}
    }

    /**
     * Searches possible end (without target sequence connections) and start
     * (without source sequence connections) elements inside the specified
     * sub-process' children. Found edit parts are added to the the specified
     * startElements and endElements lists (should be passed from the calling
     * method).
     * 
     * @param subProcessEditPart
     *            the subprocess edit part
     * @param startElements
     *            holds elements without source sequence connections
     * @param endElements
     *            holds elements without target sequence connections
     */
    private static void getStartAndEndTasks(
            SubProcessEditPart subProcessEditPart,
            List<IGraphicalEditPart> startElements,
            List<IGraphicalEditPart> endElements) {
        List children = subProcessEditPart
                .getChildBySemanticHint(
                        Integer
                                .toString(SubProcessSubProcessBodyCompartmentEditPart.VISUAL_ID))
                .getChildren();
        for (Object object : children) {
            IGraphicalEditPart editPart = (IGraphicalEditPart) object;
            List sourceConnections = editPart.getSourceConnections();
            boolean noConnections = true;
            if (!sourceConnections.isEmpty()) {
                for (Object connection : sourceConnections) {
                    if (connection instanceof SequenceEdgeEditPart) {
                        // incoming sequence connection exists so this is not
                        // start task
                        noConnections = false;
                        break;
                    }
                }
            }
            if (noConnections) {
                startElements.add(editPart);
            }
            // now test for outgoing connections
            List targetConnections = editPart.getTargetConnections();
            noConnections = true;
            if (!targetConnections.isEmpty()) {
                for (Object connection : targetConnections) {
                    if (connection instanceof SequenceEdgeEditPart) {
                        // outgoing sequence connection exists so this is not
                        // end task
                        noConnections = false;
                        break;
                    }
                }
            }
            if (noConnections) {
                endElements.add(editPart);
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.ui.IActionDelegate#selectionChanged(org.eclipse.jface.action.IAction,
     *      org.eclipse.jface.viewers.ISelection)
     */
    public void refresh() {
        IStructuredSelection structuredSelection = getStructuredSelection();
        subProcesses.clear();
        if (structuredSelection != null) {
            for (Object selElement : structuredSelection.toList()) {
                if (selElement instanceof ShapeNodeEditPart) {
                    EditPart editPart = (EditPart) selElement;
                    // select parent subprocess (or object itselt if it is
                    // subprocess)
                    SubProcessEditPart subProcess = getSubProcess(editPart);
                    if (subProcess != null
                            && !subProcesses.contains(subProcess)) {
                    	SubProcess sp = (SubProcess) subProcess.resolveSemanticElement();
                    	if (sp.getEventHandlers().isEmpty() && !sp.getVertices().isEmpty()) {
                    		subProcesses.add(subProcess);
                    	}
                    }
                }
            }
            // now remove enclosed subprocesses
            subProcesses = ToolUtilities
                    .getSelectionWithoutDependants(subProcesses);
        }
        setEnabled(!subProcesses.isEmpty());
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.ui.IObjectActionDelegate#setActivePart(org.eclipse.jface.action.IAction,
     *      org.eclipse.ui.IWorkbenchPart)
     */
    public void setActivePart(IAction action, IWorkbenchPart targetPart) {
        // TODO Auto-generated method stub
    }
}
