/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.jengine.internal.core;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.eclipse.stp.b2j.core.jengine.internal.core.datapool.SharedHashMap;
import org.eclipse.stp.b2j.core.jengine.internal.core.datapool.SharedVariable;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedBarrier;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedMutex;
import org.eclipse.stp.b2j.core.jengine.internal.core.sync.SharedSemaphore;
import org.eclipse.stp.b2j.core.jengine.internal.message.Message;

/**
 * 
 * @author amiguel
 *
 * Part of the generic engine runtime API, represents a local
 * SubController which provides access to the distributed constructs
 * of the engine.
 */
public interface SubControllerInterface {
	
	/**
	 * Get millis since the engine started (relative clock time)
	 * @return the number of milliseconds since this engine instance started the program
	 */
	public long getClock();
	public String getHost() throws Exception;
	public String getClientHost() throws Exception;
	public String[] getHosts() throws Exception;

	/**
	 * Get a hashmap to store data in which is local to this engine (instead of using static maps)
	 */
	public HashMap getEngineLocalStorageMap(String name);
	
	public String[] getVariableNames() throws Exception;

	public void setPrintHandler(PrintHandler ph) throws Exception;
	
	public void trace(Message m) throws Exception;
	public void print(String s) throws Exception;
	public void debug(String msg) throws Exception;

	public void terminate() throws Exception;
	
	//
	// Messaging
	//
	
	/**
	 * Send a message on a conversation (non-blocking)
	 * @param conversation the conversation ID to send this message on
	 * @param m the message to send on this conversation
	 */
	public void sendMessage(String conversation, Message m) throws Exception;

	/**
	 * Receive a message on a conversation (blocking)
	 * @param conversation the conversation ID to receive the message on
	 * @return the Message received on this conversation
	 * @throws Exception if an unrecoverable error occurs in the underlying implemenation
	 */
	public Message receiveMessage(String conversation) throws Exception;

	/**
	 * Receive a message on a conversation (blocking) or timeout after a given period
	 * @param conversation the conversation ID to receive the message on
	 * @param timeoutMS the number of milliseconds to timeout after
	 * @return the Message received on this conversation or null if the operation timed out
	 * @throws Exception if an unrecoverable error occurs in the underlying implemenation
	 */
	public Message receiveMessage(String conversation, long timeoutMS) throws Exception;
	
	/**
	 * Receive a message on one of a set of conversations (blocking)
	 * @param conversations the set of conversations to potentially receive a message on
	 * @return the Message received on this conversation WITH THE SUCCESSFUL CONVERSATION ID APPENDED
	 * @throws Exception if an unrecoverable error occurs in the underlying implemenation
	 */
	public Message receiveMessage(Message conversations) throws Exception;
	
	/**
	 * Receive a message on one of a set of conversations (blocking) or timeout after a given period
	 * @param conversations the set of conversations to potentially receive a message on
	 * @param timeoutMS the number of milliseconds to timeout after
	 * @return the Message received on this conversation WITH THE SUCCESSFUL CONVERSATION ID APPENDED or null if the operation timed out
	 * @throws Exception if an unrecoverable error occurs in the underlying implemenation
	 */
	public Message receiveMessage(Message conversations, long timeoutMS) throws Exception;

	/**
	 * Send a message on a given conversation ID and receive a message on a given conversation ID
	 * @param conversation the conversation ID to send this message on
	 * @param msg the message to send
	 * @param conversationReturn the conversation ID to receive the message on
	 * @return the Message received on this conversation
	 * @throws Exception if an unrecoverable error occurs in the underlying implemenation
	 */
	public Message sendAndReceiveMessage(String conversation, Message msg, String conversationReturn) throws Exception;
	
	/**
	 * Send a message on a given conversation ID and receive a message on a one of a set of given conversation IDs
	 * @param conversation the conversation ID to send this message on
	 * @param msg the message to send
	 * @param conversationReturns the set of conversation IDs to potentially receive the message on
	 * @return the Message received on this conversation WITH THE SUCCESSFUL CONVERSATION ID APPENDED
	 * @throws Exception if an unrecoverable error occurs in the underlying implemenation
	 */
	public Message sendAndReceiveMessage(String conversation, Message msg, Message conversationReturns) throws Exception;
	
	
	//
	// Threading
	//
	
	public Message launchRunner(int count, String method, int host_index) throws Exception;
	public Message launchRunner(int count, String method, int host_index, List args) throws Exception;
	public Message launchRunner(int count, String method, int host_index, Message args) throws Exception;
	public Message launchRunner(int count, String method, int host_index, String[] args) throws Exception;
	public ArrayList launchRunnerLocal(int count, String method, ArrayList args) throws Exception;

	public void notifyRunnerDeath(long id, Runner runner) throws Exception;
	public void joinRunner(Long rid) throws Exception;

	
	
	public SharedBarrier newBarrier(String name, int size) throws Exception;
	public SharedMutex newMutex(String name) throws Exception;
	public SharedSemaphore newSemaphore(String name, int initial) throws Exception;

	public SharedSemaphore getSemaphore(String name) throws Exception;
	public SharedBarrier getBarrier(String name) throws Exception;
	public SharedMutex getMutex(String name) throws Exception;
	
	public SharedVariable newVariable(String name, int type, boolean dirty) throws Exception;
	public SharedHashMap newHashMap(String name) throws Exception;

	public SharedVariable getVariable(String name) throws Exception;
	public SharedHashMap getHashMap(String name) throws Exception;
	
	public void signalSemaphore(int id, int n) throws Exception;
	public void waitSemaphore(int id, int n) throws Exception;

	public void hashmapSet(int id, String key, Object value) throws Exception;
	public String[] hashmapGetKeys(int id) throws Exception;
	public Object hashmapGet(int id, String key) throws Exception;
	public void hashmapClear(int id) throws Exception;

	public boolean isVariableDirty(int id) throws Exception;
	public void storeVariable(int id, int type, Object newval) throws Exception;
	public Object fetchVariable(int id, int type) throws Exception;
	
}