/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.saf.handler;

import java.util.Iterator;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.osgi.util.NLS;
import org.eclipse.stp.core.internal.operations.ComponentTypingDataModelProvider;
import org.eclipse.stp.core.internal.operations.IComponentTypingDataModelProperties;
import org.eclipse.stp.core.internal.saf.impl.Messages;
import org.eclipse.stp.core.saf.ISAFManager;
import org.eclipse.stp.core.saf.IUIContext;
import org.eclipse.stp.core.saf.exception.InvalidOperationException;
import org.eclipse.stp.core.saf.exception.SAFException;
import org.eclipse.stp.core.sca.Component;
import org.eclipse.stp.core.sca.ComponentType;
import org.eclipse.stp.core.sca.Interface;
import org.eclipse.stp.core.sca.SCAObject;
import org.eclipse.stp.core.sca.Service;
import org.eclipse.wst.common.frameworks.datamodel.DataModelFactory;
import org.eclipse.wst.common.frameworks.datamodel.IDataModel;
import org.eclipse.wst.common.frameworks.datamodel.IDataModelOperation;

/**
 * This class is to be extended by all component handler contributions.  
 */
public abstract class AbstractComponentHandler extends AbstractHandler
		implements IComponentHandler {

	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#canCreateComponent()
	 */
	public boolean canCreateComponent(){
		return true;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#canAddReference()
	 */
	public boolean canAddReference() {
		return true;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#canAddService()
	 */
	public boolean canAddService() {
		return true;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#canSetInterface()
	 */
	public boolean canSetInterface() {
		return true;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#canCreateComponentFor(java.lang.Object)
	 */
	public boolean canCreateComponentFor(Object implementationObj) {
		// No default support for creating components from model objects.
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#createComponentFor(java.lang.Object, org.eclipse.core.resources.IContainer, org.eclipse.stp.core.saf.IUIContext)
	 */
	public SCAObject[] createComponentFor(Object implementationObj,
			IContainer container, IUIContext callback)
			throws SAFException, InterruptedException {
		// No default support for creating components from model objects.
		throw new SAFException(Messages.stp_core_saf_operationNotSupportedForComponentKind);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#canCreateImplementation()
	 */
	public boolean canCreateImplementation() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#canSetImplementation()
	 */
	public boolean canSetImplementation() {
		return false;
	}


	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#setImplementation(org.eclipse.stp.core.sca.Component, java.lang.Object, org.eclipse.stp.core.saf.IUIContext)
	 */
	public IDataModelOperation setImplementation(Component component,
			Object implementationFile, IUIContext callback)
			throws SAFException, InterruptedException {
		// No default support for set implementation from model objects.
		throw new SAFException(Messages.stp_core_saf_operationNotSupportedForComponentKind); 
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#isInterfaceKindSupported(org.eclipse.stp.core.saf.handler.IInterfaceHandler, boolean)
	 */
	public boolean isInterfaceKindSupported(IInterfaceHandler interfaceHandler,
			boolean interfaceSet) throws SAFException {
		return true;
	}


	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#setKind(org.eclipse.stp.core.sca.Component)
	 */
	final public IDataModelOperation setKind(Component theComponent)
			throws SAFException, InvalidOperationException {

		if (theComponent == null)
			throw new SAFException(Messages.stp_core_saf_componentMustNotBeNull); 

		// Check that the type we'll be setting supports the interfaces currently set inthe component
		// (if any)
		
		ComponentType compType = theComponent.resolveComponentType();
		if(compType != null && !compType.getServices().isEmpty()){
		
			String unsupportedKinds = null;
			for (Iterator iter = compType.getServices().iterator(); iter.hasNext();) {
				Interface interfaceObj = ((Service)iter.next()).getInterface();
				IInterfaceHandler interfaceHandler = ISAFManager.INSTANCE.getHandlerFor(interfaceObj);
								
				if(interfaceHandler != null && !isInterfaceKindSupported(interfaceHandler, ISAFManager.SERVICE)){
					if(unsupportedKinds == null)
						unsupportedKinds = interfaceHandler.getKind();
					else
						unsupportedKinds = ", "+interfaceHandler.getKind(); //$NON-NLS-1$
				}
				
			}

			if(unsupportedKinds != null){
				String message = NLS.bind(Messages.stp_core_saf_unableToSetComponentKind,getKind(),unsupportedKinds);
				throw new InvalidOperationException(message);				
			}	
		
		}
		
	    IDataModel model = DataModelFactory.createDataModel(ComponentTypingDataModelProvider.class);
		model.setStringProperty(IComponentTypingDataModelProperties.TYPE, getKind());
		model.setBooleanProperty(IComponentTypingDataModelProperties.SET_TYPE, true);
		model.setProperty(IComponentTypingDataModelProperties.COMPONENT,theComponent);
			
		IStatus status = model.validate();
				
		if (status.isOK())
			return model.getDefaultOperation();	
		
		if(status.getException() instanceof InvalidOperationException)
			throw (InvalidOperationException)status.getException();
		
		throw new SAFException(status.getMessage(),status.getException());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.core.saf.handler.IComponentHandler#resetKind(org.eclipse.stp.core.sca.Component)
	 */
	final public IDataModelOperation resetKind(Component theComponent)
			throws SAFException, InvalidOperationException {
		
		if (theComponent == null)
			throw new SAFException(Messages.stp_core_saf_componentMustNotBeNull); 

		
		IDataModel model = DataModelFactory.createDataModel(ComponentTypingDataModelProvider.class);
		model.setProperty(IComponentTypingDataModelProperties.COMPONENT,theComponent);
			
		IStatus status = model.validate();
				
		if (status.isOK())
			return model.getDefaultOperation();	
		
		throw new SAFException(status.getMessage());
	}

}
