/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.sca.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.sca.Component;
import org.eclipse.stp.core.sca.ComponentType;
import org.eclipse.stp.core.sca.IConstants;
import org.eclipse.stp.core.sca.Interface;
import org.eclipse.stp.core.sca.Reference;
import org.eclipse.stp.core.sca.ReferenceValue;
import org.eclipse.stp.core.sca.WireSource;
import org.eclipse.stp.core.sca.WireTarget;

final class ComponentWireSourceHandle extends ComponentWireHandleImpl implements
      WireSource {
   private Reference reference;

   private String    referenceName;

   ComponentWireSourceHandle(Component component, Reference reference) {
      super(component);
      this.reference = reference;
   }

   ComponentWireSourceHandle(Component component, String referenceName) {
      super(component);
      this.referenceName = referenceName;
   }

   /*
    * This will be in the form "[component name]/[reference name]".
    * 
    * @see org.eclipse.stp.core.sca.SCAObject#getName()
    */
   public String getName() {
      StringBuffer buf = new StringBuffer(getOwningComponent().getName());
      buf.append(IConstants.WIRE_PATH_SEP).append(getReferenceName());
      return buf.toString();
   }

   private String getReferenceName() {
      return reference == null ? referenceName : reference.getName();
   }

   public Interface getInterface() {
      return isResolved() ? reference.getInterface() : null;
   }

   public boolean matches(WireTarget wireTarget) {
      Assert.isNotNull(wireTarget);
      if (getInterface() != null)
         return getInterface().equals(wireTarget.getInterface());
      return false;
   }

   public int getFlags() {
      return IConstants.F_POSSIBLE_WIRE_SOURCE;
   }

   public EObject getEObject() {
      /*
       * Should we use the component EObject or the reference EObject?
       */
      return getOwningComponent().getEObject();
   }

   public void setName(String name) {
      Assert.isNotSupported("ComponentWireSourceHandle#setName(String)"); //$NON-NLS-1$
   }

   /*
    * Create a ReferenceValue for this WireSource and the passed wireTarget on
    * the underlying Component.
    * 
    * @see org.eclipse.stp.core.sca.WireSource#createWireToTarget(org.eclipse.stp.core.sca.WireTarget)
    */
   public void createWireToTarget(WireTarget wireTarget) {
      if (isResolved())
         getOwningComponent().setReferenceValue(reference, wireTarget);
   }

   public List getWiredTargets() {
      List values = getOwningComponent().getReferenceValues(getReferenceName());
      if (values.isEmpty())
         return values;
      ReferenceValue wire;
      if (values.size() == 1) {
         wire = (ReferenceValue) values.get(0);
         return Collections.singletonList(wire.getTarget());
      }
      List targets = new ArrayList(values.size());
      for (int i = 0; i < values.size(); i++) {
         targets.add(((ReferenceValue) values.get(i)).getTarget());
      }
      return Collections.unmodifiableList(targets);
   }

   public String toString() {
      return getName();
   }

   void destroy() {
      this.reference = null;
      super.destroy();
   }

   public boolean deleteWireToTarget(WireTarget wireTarget) {
      if (wireTarget != null) {
         ReferenceValue refValue = getReferenceValue(wireTarget);
         if (refValue != null)
            return getOwningComponent().getReferenceValues().remove(refValue);
      }
      return false;
   }

   private ReferenceValue getReferenceValue(WireTarget wireTarget) {
      ReferenceValue refValue = null;
      String referenceName = getReferenceName();
      if (getOwningComponent() != null && referenceName != null) {
         List wires = getOwningComponent().getReferenceValues(referenceName);
         if (!wires.isEmpty()) {
            for (int i = 0; i < wires.size(); i++) {
               refValue = (ReferenceValue) wires.get(i);
               if (!refValue.targets(wireTarget))
                  refValue = null;
               else
                  break;
            }
         }
      }
      return refValue;
   }

   /*
    * Test to ensure that the reference still exists on the Component's
    * ComponentType.
    * 
    * @see org.eclipse.stp.core.sca.WireSource#isResolved()
    */
   public boolean isResolved() {
      resolveReference();
      return reference != null;
   }

   private void resolveReference() {
      ComponentType type = getOwningComponent().resolveComponentType();
      if (type != null) {
         Reference test = type.getReference(getReferenceName());
         if (test != null) {
            reference = test;
            referenceName = null;
         } else {
            if (reference != null)
               referenceName = reference.getName();
            reference = null;
         }
      }
   }
}
