/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.xef;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

public class FilterSchemaProvider implements ISchemaProvider {
    private final ISchemaProvider delegate;
    private final List<String> filters;
    private final List<String> snippetFilters;
    private List<String> allowedSchemas;
    private List<String> allowedSnippets;
    
    public static FilterSchemaProvider createSchemaFilter(ISchemaProvider delegate, String ... filters) {
        return new FilterSchemaProvider(delegate, Arrays.asList(filters), Arrays.asList(new String [] {}));
    }

    public static FilterSchemaProvider createSnippetFilter(ISchemaProvider delegate, String ... filters) {
        return new FilterSchemaProvider(delegate, Arrays.asList(new String [] {}), Arrays.asList(filters));
    }

    private FilterSchemaProvider(ISchemaProvider d, List<String> f, List<String> sf) {
        delegate = d;
        filters = f;
        snippetFilters = sf;
    }

    private synchronized void initializeSchemas() {
        if (allowedSchemas != null) {
            return;
        }
        
        allowedSchemas = new ArrayList<String>(delegate.listSchemaNamespaces(null));
        allowedSchemas.removeAll(filters);
    }
    
    private synchronized void initializeSnippets() {
        if (allowedSnippets != null) {
            return;
        }
        
        allowedSnippets = new ArrayList<String>(delegate.listSnippets(null));
        allowedSnippets.removeAll(snippetFilters);
    }

    public String getSchema(String namespace) {
        initializeSchemas();
        
        if (allowedSchemas.contains(namespace)) {
            return delegate.getSchema(namespace);
        } else {
            return null;
        }
    }

    public String getSnippet(String name) {
        initializeSnippets();
        
        if (allowedSnippets.contains(name)) {
            return delegate.getSnippet(name);
        } else {
            return null;
        }
    }

    public Collection<String> listSchemaNamespaces(String filter) {
        initializeSchemas();
        
        return allowedSchemas;
    }

    public Collection<String> listSnippets(String filter) {
        initializeSnippets();
        
        return allowedSnippets;
    }

    public void refresh() {
        delegate.refresh();
    }
}
