/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.xef;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.eclipse.stp.ui.xef.schema.SchemaAttribute;
import org.eclipse.stp.ui.xef.schema.SchemaElement;
import org.eclipse.stp.xef.util.QNameHelper;
import org.jdom.Attribute;
import org.jdom.Comment;
import org.jdom.Element;
import org.jdom.Namespace;

public class XMLInstanceElement {
    private static final String DERIVED_PREFIX = "#";
    private static final String DERIVED_POSTFIX = DERIVED_PREFIX;
    
    private final SchemaElement template;
    private final XMLInstanceElement parent;
    private Element jdomElement;
    private final List<XMLInstanceElement> children = new LinkedList<XMLInstanceElement>();
    private Map<Object, Object> data = new HashMap<Object, Object>(1);
    private Map<String, String> nsMap;
    
    public XMLInstanceElement(SchemaElement tmpl) {
        this(tmpl, null);
    }
    
    public XMLInstanceElement(SchemaElement tmpl, XMLInstanceElement par) {
        this(tmpl, par, new Element(tmpl.getName(), tmpl.getShortNameSpace(), tmpl.getNameSpace()));
    }
    
    XMLInstanceElement(SchemaElement tmpl, XMLInstanceElement par, Element jdomEl) {
        template = tmpl;
        parent = par;
        jdomElement = jdomEl;
        nsMap = getNamespaceMappings();
    }
    
    public synchronized void addMinimalSubElements() {
        // Add required subelements
        for (SchemaElement childElement : template.getMinimalSubElementList()) {
            XMLInstanceElement child = new XMLInstanceElement(childElement, this);
            child.addMinimalSubElements();
            addChildInternal(child);
        }        
    }

    public synchronized void addChild(XMLInstanceElement pi, boolean addSiblings) {
        addChildInternal(pi);
        
        List<SchemaElement> childElems = new LinkedList<SchemaElement>();
        for (XMLInstanceElement c : children) {
            childElems.add(c.getTemplate());
        }
        
        if (addSiblings) {
            List<SchemaElement> requiredSisters = pi.getTemplate().getMinimalSisterElementsList(childElems);
            for (SchemaElement sister : requiredSisters) {
                XMLInstanceElement child = new XMLInstanceElement(sister, this);
                addChildInternal(child);
                child.addMinimalSubElements();
            }
        }
    }
    
    private void addChildInternal(XMLInstanceElement pi) {
        // Adds the child in the correct position
        
        List<SchemaElement> allowed = template.getNestedElements(new LinkedList<SchemaElement>());
        int curIdx = allowed.indexOf(pi.getTemplate());
        
        int curPos = 0;
        for (int i = 0; i < children.size(); i++) {
            int childIdx = allowed.indexOf(children.get(i).getTemplate());
            if (childIdx > curIdx) {
                break;
            }
            curPos = i + 1;
        }
        
        children.add(curPos, pi);
    }
    
    public synchronized void removeChild(XMLInstanceElement pi) {
        children.remove(pi);
    }
    
    public String getAttribute(String name) {
        return jdomElement.getAttributeValue(name);
    }
    
    public void removeAttribute(String name) {
        jdomElement.removeAttribute(name);
    }        
    
    public void setAttribute(String name, String value) {
        jdomElement.setAttribute(name, value);
    }
    
    @SuppressWarnings("unchecked")
    private Map<String, String> getNamespaceMappings() {
        Map<String, String> map = new HashMap<String, String>();
        for (Namespace ns : (List<Namespace>) getAllAdditionalNamespaces()) {
            if (!map.containsKey(ns.getPrefix())) {
                map.put(ns.getPrefix(), ns.getURI());
            }
        }
        return map;
        
    }
    
    @SuppressWarnings("unchecked")
    public List getAllAdditionalNamespaces() {
        List list = new Vector();
        list.addAll(jdomElement.getAdditionalNamespaces());
        XMLInstanceElement instParElem = parent;
        while (instParElem != null) {
            Element element = instParElem.jdomElement;
            while (element != null) {
                list.add(Namespace.getNamespace(element.getNamespacePrefix(), 
                        element.getNamespaceURI()));
                list.addAll(element.getAdditionalNamespaces());
                element = element.getParentElement();
            }
            instParElem = instParElem.parent;
        }
        return list;
    }
    
    @SuppressWarnings("unchecked")
    public void addNamespaceDeclaration(String prefix, String uri) {
        if (prefix == null || prefix.length() == 0 || uri == null || uri.length() == 0) {
            return;
        }
        for (Namespace ns : (List<Namespace>)jdomElement.getAdditionalNamespaces()) {
            if (prefix.equals(ns.getPrefix())) {
                return;
            }
        }
        jdomElement.addNamespaceDeclaration(Namespace.getNamespace(prefix, uri));
    }
    
    public void removeNamespaceDeclaration(String prefix, String uri) {
        if (prefix == null || prefix.length() == 0 || uri == null || uri.length() == 0) {
            return;
        }
        jdomElement.removeNamespaceDeclaration(Namespace.getNamespace(prefix, uri));
    }
    
    public String getBasedOnSnippet() {
        return findBasedOn(jdomElement);
    }

    @SuppressWarnings("unchecked")
    public String getDependencyValue(String path) {
        String retValue = null;
        Map<String, String> tmpNsMap = new HashMap<String, String>(nsMap);
        Attribute attrib = getDependencyAttribute(path);
        if (attrib != null) {
            for (Namespace ns : (List<Namespace>) attrib.getParent().getAdditionalNamespaces()) {
                tmpNsMap.put(ns.getPrefix(), ns.getURI());
            }
            retValue = attrib.getValue();
            if (retValue.contains(":")) {
                // may be a QName
                String[] ns = retValue.split(":", 2);
                if (tmpNsMap.containsKey(ns[0])) {
                   retValue = QNameHelper.convertToFullName(tmpNsMap.get(ns[0]), ns[1]);
                }
            }
            
        }
        return retValue;
    }
    
    private Attribute getDependencyAttribute(String path) {
        Attribute retAttrib = null;
        if (path == null) {
            return retAttrib;
        }
        
        if (path.startsWith("../") && parent != null) {
            path = path.substring(3);
            retAttrib = parent.getDependencyAttribute(path);
        } else if (path.startsWith("@")) {
            retAttrib = jdomElement.getAttribute(path.substring(1));
        } else {
            String elemName = "";
            if (path.contains("/")) {
                String[] elements = path.split("/", 2);
                elemName = elements[0];
                path = elements[1];
            } else {
                elemName = path;
            }
            for (XMLInstanceElement element : children) {
                if (element.getJDOMElement().getName().equals(elemName)) {
                    retAttrib = element.getDependencyAttribute(path);
                }
            }
        }
        return retAttrib;
        
    }
    
    public String getDependencyName(String path) {
        String retName = null;
        Attribute attrib = getDependencyAttribute(path);
        if (attrib != null) {
            retName = attrib.getName();
        } else if (path.startsWith("@")) {
            retName = path.substring(1);
        }
        return retName;
    }
    
    public Object getData(Object key) {
        return data.get(key);
    }
    
    public void setData(Object key, Object value) {
        data.put(key, value);
    }

    static String findBasedOn(Element element) {
        Comment derivedComment = findBasedOnComment(element);
        if (derivedComment != null) {
            return derivedComment.getText().substring(1, derivedComment.getText().length() - 1);
        }
        return null;
    }
    
    private static Comment findBasedOnComment(Element element) {
        for (Object obj : element.getContent()) {
            if (obj instanceof Comment) {
                Comment comment = (Comment) obj;
                String s = comment.getText();
                if (s.startsWith(DERIVED_PREFIX) && s.endsWith(DERIVED_POSTFIX)) {
                    return comment;
                } else {
                    return null;
                }
            }            
        }
        return null;
    }
    
    public void setBasedOnSnippet(String origin) {
        Comment oldDerived = findBasedOnComment(jdomElement);
        if (oldDerived != null) {
            jdomElement.removeContent(oldDerived);
        }
        
        Comment newDerived = new Comment(DERIVED_PREFIX + origin + DERIVED_POSTFIX);
        jdomElement.addContent(0, newDerived);
    }
    
    public String getText() {
        return jdomElement.getText();
    }
    
    public void setText(String text) {
        jdomElement.setText(text);
    }
    
    public synchronized List<XMLInstanceElement> getChildren() {
        return Collections.unmodifiableList(children);
    }
    
    public XMLInstanceElement getParent() {
        return parent;
    }
    
    public SchemaElement getTemplate() {
        return template;
    }
    
    public String getNamespacePrefix() {
	return jdomElement.getNamespacePrefix();
    }
    
    public Element getJDOMElement() {
        // clone, because we're going to link them up here, we don't want to link up the original element  
        Element cur = (Element) jdomElement.clone(); 
        for (XMLInstanceElement child : children) {
            cur.addContent(child.getJDOMElement());
        }
        return cur;
    }
    
    public void setJDOMElement(Element el) {
	jdomElement = el;
    }
    
    public Map<String, String> getNamespaceMap() {
        return nsMap;
    }
    
    public ValidationProblem validate() {
        for (SchemaAttribute attr : getTemplate().getAttributes()) {
            if (attr.getRequired()) {
                String val = jdomElement.getAttributeValue(attr.getName());
                if (val == null || "".equals(val)) {
                    return new ValidationProblem(attr, "Required attribute '" + attr.getDisplayName() + "' not set.", this);
                }
            }
        }
        
        for (XMLInstanceElement child : children) {
            ValidationProblem vp = child.validate();
            if (vp != null) {
                return vp;
            }
        }
        return null;
    }

    @Override
    public String toString() {
	String prefix = jdomElement.getNamespacePrefix();
	String name = jdomElement.getName();
		return (prefix == null || prefix.length() == 0) ? name : prefix + ":" + name;
    }       

    public String getLabel() {
	String label = null;
	if (template != null) {
	    label = template.getDisplayName();
	}
	if (label == null) {
		label = this.toString();
        } 
        return label;
    }       
}
