/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.ui;

import java.util.List;
import java.util.Vector;

import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryLabelProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.DialogCellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerDropAdapter;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.IPackage;
import org.eclipse.stp.soas.deploy.core.Utilities;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployConfiguration;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployPackage;
import org.eclipse.stp.soas.deploy.models.deployfile.DeployServer;
import org.eclipse.stp.soas.internal.deploy.emf.AdapterFactoryContentProvider;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.Dialog;
import org.eclipse.stp.soas.internal.deploy.ui.dialogs.ServerSelectionDialogPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.eclipse.ui.part.ResourceTransfer;
import org.eclipse.ui.views.navigator.LocalSelectionTransfer;
import org.eclipse.wst.server.core.IServer;

/**
 * @author rcernich
 * 
 * Created on Jan 23, 2004
 */
public class TargetMapTableViewer {

	public static final String PROP_PACKAGE = "package"; //$NON-NLS-1$
	public static final String PROP_TARGET = "target"; //$NON-NLS-1$

	private static final String COLUMN_PROPERTIES[] = { PROP_PACKAGE,
			PROP_TARGET};

	private TableViewer mTargetMap;

	public TargetMapTableViewer(Table table) {
		super();
		createControl(table);
	}

	public TableViewer getViewer() {
		return mTargetMap;
	}

	private void createControl(Table table) {
		mTargetMap = new TableViewer(table);
		mTargetMap.getTable().setHeaderVisible(true);
		mTargetMap.getTable().setLinesVisible(true);

		mTargetMap.setContentProvider(new AdapterFactoryContentProvider(
				Utilities.getItemAdapterFactory()));
		mTargetMap.setLabelProvider(new TargetMapLabelProvider(Utilities
				.getItemAdapterFactory()));
		mTargetMap.addFilter(new ViewerFilter() {

			public boolean select(Viewer viewer, Object parentElement,
					Object element) {
				return element instanceof DeployPackage;
			}
		});
		mTargetMap.setColumnProperties(COLUMN_PROPERTIES);
		mTargetMap.setCellModifier(new TargetMapCellModifier());
		mTargetMap.setCellEditors(createCellEditors(table));
		mTargetMap.addDropSupport(DND.DROP_LINK | DND.DROP_DEFAULT,
				new Transfer[] { LocalSelectionTransfer.getInstance(),
						ResourceTransfer.getInstance()},
				new TargetMapDropListener(mTargetMap));

		TableLayout layout = new TableLayout();

		TableColumn column = new TableColumn(table, SWT.LEFT);
		column.setText(DeployCorePlugin.getDefault().getResourceString(
				"TargetMapTableViewer.column.heading.package")); //$NON-NLS-1$
		column.pack();
		//cr 387066: set package column weight at 80 units
//		layout.addColumnData(new ColumnPixelData(column.getWidth(), true));
		layout.addColumnData(new ColumnWeightData(80,true));

		
		
		column = new TableColumn(table, SWT.LEFT);
		column.setText(DeployCorePlugin.getDefault().getResourceString(
				"TargetMapTableViewer.column.heading.target")); //$NON-NLS-1$
		column.pack();
		//cr 387066: set target column weight at 20 units
//		layout.addColumnData(new ColumnPixelData(column.getWidth(), true));
		layout.addColumnData(new ColumnWeightData(20,true));
		
		table.setLayout(layout);
	}

	private CellEditor[] createCellEditors(Composite parent) {
		List editors = new Vector(COLUMN_PROPERTIES.length);
		CellEditor editor = null;
		editors.add(editor);

		//editor = new ServerSelectionCellEditor(parent);
		editors.add(editor);

		return (CellEditor[]) editors.toArray(new CellEditor[editors.size()]);
	}

	private static class TargetMapLabelProvider extends
			AdapterFactoryLabelProvider {

		private ILabelProvider mWorkbenchLabelProvider = new WorkbenchLabelProvider();

		/**
		 * @param adapterFactory
		 */
		public TargetMapLabelProvider(AdapterFactory adapterFactory) {
			super(adapterFactory);
		}

		/**
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(Object,
		 *      int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
			Image image = null;
			if (element instanceof DeployPackage) {
				DeployPackage pkg = (DeployPackage) element;
				String property = COLUMN_PROPERTIES[columnIndex];
				if (property.equals(PROP_PACKAGE)) {
					image = getImage(pkg);
				}
				else if (property.equals(PROP_TARGET)) {
					if (pkg.getTargetConfiguration().size() > 0) {
						DeployConfiguration config = (DeployConfiguration) pkg
								.getTargetConfiguration().get(0);
						image = getImage(config);
					}
				}
			}
			return image;
		}

		/**
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(Object,
		 *      int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			String text = new String();
			if (element instanceof DeployPackage) {
				String property = COLUMN_PROPERTIES[columnIndex];
				if (property.equals(PROP_PACKAGE)) {
					text = getText(element);
				}
				else if (property.equals(PROP_TARGET)) {
					DeployPackage pkg = (DeployPackage) element;
					if (pkg.getTargetConfiguration() == null
							|| pkg.getTargetConfiguration().isEmpty()) {
						text = new String();
					}
					else {
						//JFaceResources.getImageRegistry().g
						text = getText((DeployConfiguration) pkg
								.getTargetConfiguration().get(0));
					}
				}
			}
			return text;
		}

		public Image getImage(Object element) {
			Image image = null;

			// make sure we get an adapter on this guy
			adapterFactory.adapt(element, IItemLabelProvider.class);
			if (element instanceof DeployPackage) {
				IPackage ip = Utilities
						.adaptToIPackage((DeployPackage) element);
				if (ip == null) {
					image = super.getImage(element);
				}
				else {
					image = mWorkbenchLabelProvider.getImage(ip.getFile());
				}
			}
			else if (element instanceof DeployConfiguration) {
				DeployConfiguration config = (DeployConfiguration) element;
				IServer cp = Utilities
						.adaptToIServer(config);
				if (cp == null) {
					image = super.getImage(config.getTargetServer());
				}
				else {
					image = null;//ProfileImageRegistry.getInstance().getProfileImage(cp.getProvider());
				}
			}
			return image;
		}

		public String getText(Object element) {
			String text;

			// make sure we get an adapter on this guy
			adapterFactory.adapt(element, IItemLabelProvider.class);
			if (element instanceof DeployPackage) {
				IPackage ip = Utilities
						.adaptToIPackage((DeployPackage) element);
				if (ip == null) {
					text = super.getText(element);
				}
				else {
					text = ip.getFile().getFullPath().toString();
				}
			}
			else if (element instanceof DeployConfiguration) {
				DeployConfiguration config = (DeployConfiguration) element;
				IServer cp = Utilities
						.adaptToIServer(config);
				if (cp == null) {
					if (config.getTargetServer() == null
							|| config.getTargetServer().getProfileName() == null) {
						text = TargetMapTreeViewer.LABEL_UNSPECIFIED_TARGET;
					}
					else {
						text = config.getTargetServer().getProfileName();
					}
				}
				else {
					text = cp.getName();
				}
			}
			else {
				text = super.getText(element);
			}
			return text;
		}

		public void dispose() {
			mWorkbenchLabelProvider.dispose();
			super.dispose();
		}
	}

	private class TargetMapCellModifier implements ICellModifier {

		public TargetMapCellModifier() {
			super();
		}

		/**
		 * @see org.eclipse.jface.viewers.ICellModifier#canModify(Object,
		 *      String)
		 */
		public boolean canModify(Object element, String property) {
			boolean retVal = false;
			if (element instanceof DeployPackage) {
				if (property.equals(PROP_PACKAGE)) {
					retVal = false;
				}
				else if (property.equals(PROP_TARGET)) {
					//retVal = true;
				}
			}
			return retVal;
		}

		/**
		 * @see org.eclipse.jface.viewers.ICellModifier#getValue(Object, String)
		 */
		public Object getValue(Object element, String property) {
			Object retVal = null;
			if (element instanceof DeployPackage) {
				if (property.equals(PROP_PACKAGE)) {
					retVal = element;
				}
				else if (property.equals(PROP_TARGET)) {
					EList list = ((DeployPackage) element)
							.getTargetConfiguration();
					if (!list.isEmpty()) {
						retVal = list.get(0);
					}
				}
			}
			return retVal;
		}

		/**
		 * @see org.eclipse.jface.viewers.ICellModifier#modify(Object, String,
		 *      Object)
		 */
		public void modify(Object element, String property, Object value) {
			Object data = null;
			if (element instanceof Item) {
				data = ((Item) element).getData();
			}
			else {
				data = element;
			}
			if (data instanceof DeployPackage) {
				if (property.equals(PROP_PACKAGE)) {
				}
				else if (property.equals(PROP_TARGET)) {
					((DeployPackage) data).getTargetConfiguration();
					// We pass the configuration around, so nothing to modify
					// here as the cell editor should have already updated
					// the target server.
				}
				mTargetMap.refresh(data);
			}
		}
	}

	private class TargetMapDropListener extends ViewerDropAdapter {

		public TargetMapDropListener(Viewer viewer) {
			super(viewer);
		}

		public void dragOver(DropTargetEvent event) {
			if ((event.operations & DND.DROP_LINK) == DND.DROP_LINK) {
				event.detail = DND.DROP_LINK;
			}
			super.dragOver(event);
		}

		public void drop(DropTargetEvent event) {
			super.drop(event);
			event.detail = DND.DROP_LINK;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ViewerDropAdapter#performDrop(java.lang.Object)
		 */
		public boolean performDrop(Object data) {
			return true;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ViewerDropAdapter#validateDrop(java.lang.Object,
		 *      int, org.eclipse.swt.dnd.TransferData)
		 */
		public boolean validateDrop(Object target, int operation,
				TransferData transferType) {
			if (transferType != null
					&& (LocalSelectionTransfer.getInstance().isSupportedType(
							transferType) || ResourceTransfer.getInstance()
							.isSupportedType(transferType))) {
				return validateSource();
			}
			return false;
		}

		private boolean validateSource() {
			// Check to see if source is a package or server.
			return true;
		}

	}

	public class ServerSelectionCellEditor extends DialogCellEditor {

		public ServerSelectionCellEditor() {
			super();
		}

		/**
		 * @param parent
		 */
		public ServerSelectionCellEditor(Composite parent) {
			super(parent);
		}

		/**
		 * @param parent
		 * @param style
		 */
		public ServerSelectionCellEditor(Composite parent, int style) {
			super(parent, style);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.DialogCellEditor#openDialogBox(org.eclipse.swt.widgets.Control)
		 */
		protected Object openDialogBox(Control cellEditorWindow) {
			DeployConfiguration value = (DeployConfiguration) getValue();
			ServerSelectionDialogPage page = new ServerSelectionDialogPage();
			page.init(value.getSourcePackage(), value.getTargetServer());
			Dialog dialog = new Dialog(cellEditorWindow.getShell(), page);
			if (dialog.open() == Dialog.OK) {
				DeployServer server = page.getSelectedDeployServer();
				value.setTargetServer(server);
				if (server.getRoot() == null) {
					// This is a new server instance. Add it to the model.
					// We need to do this after updating the configuration
					// because it notifies the content provider, which
					// notifies the viewer, which causes the viewer to be
					// updated prior to the target being set. This causes
					// a blank to show up for the target in the viewer.
					server.setRoot(value.getSourcePackage().getRoot());
				}
			}
			return value;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.DialogCellEditor#updateContents(java.lang.Object)
		 */
		protected void updateContents(Object value) {
			Label defaultLabel = getDefaultLabel();
			if (defaultLabel == null) {
				return;
			}

			String text = null;
			if (value != null) {
				DeployServer ds = ((DeployConfiguration) value)
						.getTargetServer();
				if (ds != null) {
					text = ds.getProfileName();
				}
			}
			if (text == null) {
				text = new String();
			}
			defaultLabel.setText(text);
		}
	}
}