package org.eclipse.stp.core.infrastructure.operations;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jem.util.emf.workbench.ProjectUtilities;
import org.eclipse.stp.core.infrastructure.emf.EditModelException;
import org.eclipse.stp.core.infrastructure.emf.IEditModelFactory;
import org.eclipse.stp.core.infrastructure.emf.IEditModelScribbler;
import org.eclipse.stp.core.infrastructure.emf.IScribblerDomain;
import org.eclipse.wst.common.frameworks.datamodel.AbstractDataModelOperation;
import org.eclipse.wst.common.frameworks.datamodel.IDataModel;

public abstract class AbstractScribblerOperation extends
      AbstractDataModelOperation {
   private IProgressMonitor    progressMonitor;

   private IEditModelScribbler scribbler                      = null;

   private boolean             isArtifactEditModelSelfManaged = false;

   /**
    * Default constructor.
    */
   public AbstractScribblerOperation() {
      super();
   }

   /**
    * Constructor taking an IDataModel
    * 
    * @param model
    *           the IDataModel used to drive this operation
    */
   public AbstractScribblerOperation(IDataModel model) {
      super(model);
   }

   /**
    * Extended operations should implement doExecute to perform their work
    * 
    * @param IProgressMonitor
    *           monitor
    */
   protected abstract IStatus doExecute(IProgressMonitor monitor)
         throws ExecutionException;

   /**
    * Extended operations should implement doRedo to perform their redo work
    * 
    * @param IProgressMonitor
    *           monitor
    */
   protected abstract IStatus doRedo(IProgressMonitor monitor)
         throws ExecutionException;

   /**
    * Extended operations should implement doUndo to perform their undo work
    * 
    * @param IProgressMonitor
    *           monitor
    */
   protected abstract IStatus doUndo(IProgressMonitor monitor)
         throws ExecutionException;

   public final IStatus execute(final IProgressMonitor monitor, IAdaptable info)
         throws ExecutionException {
      setProgressMonitor(monitor);

      if (info != null)
         scribbler = (IEditModelScribbler) info
               .getAdapter(IEditModelScribbler.class);

      if (scribbler == null)
         scribbler = createEditModelScribbler();

      // All operations will save the scribbler with force flag set to false.
      // OperationUtil.runAsUnchecked(new MRunnable(){
      // public Object run() {
      try {
         // perform real work
         IStatus status = doExecute(monitor);
         if (!status.isOK())
            return status;

         if (scribbler != null)
            scribbler.save(false, monitor);
      } catch (Exception e) {
         e.printStackTrace();
      } finally {
         if (isArtifactEditModelSelfManaged && scribbler != null) {
            try {
               scribbler.close(monitor);
            } catch (EditModelException e) {
               e.printStackTrace();
            }
         }
      }

      // return null;
      // }
      // }) ;

      return Status.OK_STATUS;
   }

   public final IStatus redo(IProgressMonitor monitor, IAdaptable info)
         throws ExecutionException {
      return doRedo(monitor);
   }

   public IStatus undo(IProgressMonitor monitor, IAdaptable info)
         throws ExecutionException {
      return doUndo(monitor);
   }

   /**
    * @param newProgressMonitor
    *           org.eclipse.core.runtime.IProgressMonitor
    */
   protected void setProgressMonitor(IProgressMonitor newProgressMonitor) {
      progressMonitor = newProgressMonitor;
   }

   /**
    * @return org.eclipse.core.runtime.IProgressMonitor
    */
   protected IProgressMonitor getProgressMonitor() {
      return progressMonitor;
   }

   protected IEditModelScribbler createEditModelScribbler() {
      IEditModelScribbler aScribbler = null;

      if (model
            .isPropertySet(IAbstractScribblerDataModelProperties.EDIT_MODEL_LABEL)
            && model
                  .isPropertySet(IAbstractScribblerDataModelProperties.SCRIBBLER_DOMAINS)
            && model
                  .isPropertySet(IAbstractScribblerDataModelProperties.PROJECT_NAME)) {
         String editModelLabel = model
               .getStringProperty(IAbstractScribblerDataModelProperties.EDIT_MODEL_LABEL);
         IScribblerDomain[] domains = (IScribblerDomain[]) model
               .getProperty(IAbstractScribblerDataModelProperties.SCRIBBLER_DOMAINS);

         try {
            aScribbler = IEditModelFactory.eINSTANCE.createScribblerForWrite(
                  getProject(), editModelLabel, domains);
            isArtifactEditModelSelfManaged = true;
         } catch (EditModelException e) {
            e.printStackTrace();
         }
      }

      return aScribbler;
   }

   protected IProject getProject() {
      String pName = model
            .getStringProperty(IAbstractScribblerDataModelProperties.PROJECT_NAME);
      if (pName != null && pName.length() > 0)
         return ProjectUtilities.getProject(pName);

      return null;
   }

   protected IEditModelScribbler getEditModelScribbler() {
      if (scribbler == null)
         scribbler = createEditModelScribbler();

      return scribbler;
   }
}
