/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.internal.deploy.emf.editors;

import java.io.IOException;
import java.util.Collections;
import java.util.EventObject;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.command.BasicCommandStack;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CommandStack;
import org.eclipse.emf.common.command.CommandStackListener;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryContentProvider;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.stp.soas.internal.deploy.emf.EMFCommonPlugin;
import org.eclipse.stp.soas.internal.deploy.emf.refactoring.ResourceNameChangeAdapter;
import org.eclipse.stp.soas.internal.deploy.ui.editors.EnhancedFormEditor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IStorageEditorInput;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.SaveAsDialog;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.ui.views.properties.IPropertySourceProvider;
import org.eclipse.ui.views.properties.PropertySheetPage;

/**
 * @author rcernich
 * 
 * Created on Mar 4, 2004
 */
public abstract class EMFFormEditor extends EnhancedFormEditor implements
		IResourceEditingContainer, IResourceChangeListener {

	private AdapterFactoryEditingDomain mEditingDomain;
	private Resource mResource;
	private IPropertySheetPage mPropertySheetPage;
	// CR399298 - Add listener to update resource URIs in response to file
	// name changes in the workspace.
	private ResourceNameChangeAdapter mResourceURIUpdater;

	public EMFFormEditor() {
		super();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.IEditorPart#init(org.eclipse.ui.IEditorSite,
	 *      org.eclipse.ui.IEditorInput)
	 */
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException {
		if (!isValidContentType(input)) {
			IStatus s = new Status(IStatus.ERROR, EMFCommonPlugin
					.getDefault().getBundle().getSymbolicName(), IStatus.ERROR,
					"Wrong editor for " + input.getName(), null);
			throw new PartInitException(s);
		}

		super.init(site, input);

		try {
			initializeResource(input);
		}
		catch (CoreException e) {
			throw new PartInitException(e.getStatus());
		}

		setPartName(input.getName());

		// Load the resource through the editing domain.
		if (input instanceof IFileEditorInput) {
			((IFileEditorInput) input).getFile().getWorkspace()
					.addResourceChangeListener(this);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.ISaveablePart#isDirty()
	 */
	public boolean isDirty() {
		return super.isDirty()
				|| ((BasicCommandStack) getEditingDomain().getCommandStack())
						.isSaveNeeded();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.ISaveablePart#doSave(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void doSave(IProgressMonitor monitor) {
		super.doSave(monitor);
		// Do the work within an operation because this is a long running
		// activity that modifies the workbench.
		//
		WorkspaceModifyOperation operation = new WorkspaceModifyOperation() {

			// This is the method that gets invoked when the operation runs.
			//
			protected void execute(IProgressMonitor monitor)
					throws CoreException {
				try {
					// Save the resource to the file system.
					//
					getResource().save(getSaveOptions());
				}
				catch (Exception exception) {
					EMFCommonPlugin.getDefault().log(exception);
				}
			}
		};

		try {
			// This runs the options, and shows progress.
			// (It appears to be a bad thing to fork this onto another thread.)
			//
			new ProgressMonitorDialog(getSite().getShell()).run(false, false,
					operation);

			// Refresh the necessary state.
			//
			((BasicCommandStack) getEditingDomain().getCommandStack())
					.saveIsDone();
			firePropertyChange(IEditorPart.PROP_DIRTY);
		}
		catch (Exception exception) {
			// Something went wrong that shouldn't.
			//
			EMFCommonPlugin.getDefault().log(exception);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.ISaveablePart#doSaveAs()
	 */
	public void doSaveAs() {
		SaveAsDialog saveAsDialog = new SaveAsDialog(getSite().getShell());
		saveAsDialog.open();
		IPath path = saveAsDialog.getResult();
		if (path != null) {
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
			if (file != null) {
				mResource.setURI(URI.createPlatformResourceURI(file
						.getFullPath().toString()));
				IFileEditorInput modelFile = new FileEditorInput(file);
				setInput(modelFile);
				setPartName(file.getName());
				doSave(getEditorSite().getActionBars().getStatusLineManager()
						.getProgressMonitor());
			}
		}
	}

	public void dispose() {
		// Don't do this in close() since it nukes the resources.  If this
		// guy were dirty when close was called, and the user chose yes in
		// the save prompt, an empty IResource would be created.
		for (Iterator resIt = getEditingDomain().getResourceSet()
				.getResources().iterator(); resIt.hasNext();) {
			Resource resource = (Resource) resIt.next();
			if (resource.isLoaded()) {
				try {
					resource.unload();
				}
				catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
		getEditingDomain().getResourceSet().getResources().clear();
		mEditingDomain.getResourceSet().eAdapters().remove(mResourceURIUpdater);
		mResourceURIUpdater.dispose();
		mResourceURIUpdater = null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.ISaveablePart#isSaveAsAllowed()
	 */
	public boolean isSaveAsAllowed() {
		return true;
	}

	public EditingDomain getEditingDomain() {
		return mEditingDomain;
	}

	public Resource getResource() {
		return mResource;
	}

	public EObject getModelRoot() {
		return (EObject) mResource.getContents().get(0);
	}

	protected boolean isValidContentType(IEditorInput input) {
		return true;
	}

	public void resourceChanged(IResourceChangeEvent event) {
		IResourceDelta delta = event.getDelta();

		if (delta != null) {
			try {
				delta.accept(new IResourceDeltaVisitor() {

					public boolean visit(IResourceDelta delta) {
						IResource resource = delta.getResource();
						if ((resource != null) && (resource instanceof IFile)) {
							IFile rem = (IFile) resource;
							IEditorInput thisf = getEditorInput();
							Object f = thisf.getAdapter(IFile.class);
							if (rem.equals(f)) {
								if (delta.getKind() == IResourceDelta.REMOVED) {
									if ((IResourceDelta.MOVED_TO & delta
											.getFlags()) != 0) {
										final IPath path = delta
												.getMovedToPath();
										Display display = getSite().getShell()
												.getDisplay();
										display.asyncExec(new Runnable() {

											public void run() {
												IFile file = ResourcesPlugin
														.getWorkspace()
														.getRoot()
														.getFile(path);
												setInput(new FileEditorInput(
														file));
												setPartName(path.lastSegment());
											}
										});
									}
									else {
										close(isSaveOnCloseNeeded());
									}
								}
								return false;
							}
							return true;
						}
						return true;
					}
				});
			}
			catch (CoreException e) {
				e.printStackTrace();
			}
		}
	}

	protected BasicCommandStack createCommandStack() {
		return new BasicCommandStack();
	}

	protected abstract AdapterFactory createModelAdapterFactory();

	protected AdapterFactory getModelAdapterFactory() {
		return ((AdapterFactoryEditingDomain) getEditingDomain())
				.getAdapterFactory();
	}

	protected AdapterFactoryEditingDomain createEditingDomain() {
		return new AdapterFactoryEditingDomain(createModelAdapterFactory(),
				createCommandStack());
	}

	protected Map getLoadOptions() {
		return Collections.EMPTY_MAP;
	}

	protected Map getSaveOptions() {
		return Collections.EMPTY_MAP;
	}

	public IPropertySheetPage getPropertySheetPage() {
		if (mPropertySheetPage == null) {
			mPropertySheetPage = createPropertySheetPage();
		}

		return mPropertySheetPage;
	}

	protected IPropertySheetPage createPropertySheetPage() {
		PropertySheetPage propertySheet = new PropertySheetPage() {

			public void makeContributions(IMenuManager menuManager,
					IToolBarManager toolBarManager,
					IStatusLineManager statusLineManager) {
				super.makeContributions(menuManager, toolBarManager,
						statusLineManager);
				// Look into adding a hook back to the editor.
			}

			public void setActionBars(IActionBars actionBars) {
				super.setActionBars(actionBars);
				// Look into adding a hook back to the editor.
			}
		};
		propertySheet.setPropertySourceProvider(createPropertySourceProvider());
		return propertySheet;
	}

	protected IPropertySourceProvider createPropertySourceProvider() {
		return new AdapterFactoryContentProvider(getModelAdapterFactory());
	}

	protected void initializeResource(IEditorInput input) throws CoreException {
		mEditingDomain = createEditingDomain();
		
		// CR399298 - Add listener to update resource URIs in response to file
		// name changes in the workspace.
		mResourceURIUpdater = new ResourceNameChangeAdapter();
		mEditingDomain.getResourceSet().eAdapters().add(mResourceURIUpdater);		

		if (input instanceof IFileEditorInput) {
			IFile file = (IFile) input.getAdapter(IFile.class);
			if (file != null) {
				URI inputURI = URI.createPlatformResourceURI(file.getFullPath()
						.toString());
				mResource = mEditingDomain.createResource(inputURI.toString());
				try {
					mResource.load(getLoadOptions());
				}
				catch (IOException e) {
					IStatus s = new Status(IStatus.ERROR, EMFCommonPlugin
							.getDefault().getBundle().getSymbolicName(),
							IStatus.ERROR, "Error loading file "
									+ input.getClass(), e);
					throw new CoreException(s);
				}
			}
		}
		else if (input instanceof IStorageEditorInput) {
			URI inputURI = URI.createGenericURI("unknown", input.getClass()
					.toString(), input.getName());
			mResource = mEditingDomain.createResource(inputURI.toString());
			try {
				mResource.load(((IStorageEditorInput) input).getStorage()
						.getContents(), getLoadOptions());
			}
			catch (IOException e) {
				IStatus s = new Status(IStatus.ERROR, EMFCommonPlugin
						.getDefault().getBundle().getSymbolicName(),
						IStatus.ERROR,
						"Error loading file " + input.getClass(), e);
				throw new CoreException(s);
			}
		}
		else {
			IStatus s = new Status(IStatus.ERROR, EMFCommonPlugin
					.getDefault().getBundle().getSymbolicName(), IStatus.ERROR,
					"Unkown input type " + input.getClass(), null);
			throw new CoreException(s);
		}

		mEditingDomain.getCommandStack().addCommandStackListener(
				new CommandStackListener() {

					public void commandStackChanged(final EventObject event) {
						if (getContainer() == null) {
							handleCommandStackChanged(event);
						}
						else {
							getContainer().getDisplay().asyncExec(
									new Runnable() {

										public void run() {
											handleCommandStackChanged(event);
										}
									});
						}
					}
				});
	}
	
	protected void handleCommandStackChanged(final EventObject event) {
		firePropertyChange(IEditorPart.PROP_DIRTY);

		// Try to select the affected objects.
		//
		Command mostRecentCommand = ((CommandStack) event
				.getSource()).getMostRecentCommand();
		if (mostRecentCommand != null && getActivePageInstance() != null) {
			getActivePageInstance().selectReveal(
					mostRecentCommand
							.getAffectedObjects());
		}
	}

}