/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.saf.ui.contribution;

import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.stp.core.internal.saf.ui.Activator;
import org.eclipse.stp.core.internal.saf.ui.Messages;


/**
 * The registry for the org.eclipse.stp.core.saf.ui.propertiesContribution extension point
 */
public class SOAPropertiesRegistry {
	
	private static final String EXTENSION_POINT_PROPERTIES_CONTRIBUTION	= "propertiesContribution";			//$NON-NLS-1$
	
	private static SOAPropertiesRegistry _registry = null;
	
	private Hashtable _entries = null;
	
	/**
	 * Get the properties contribution registry for the target editor.
	 * @return the registry
	 */
	public static SOAPropertiesRegistry getRegistry() {
		if (_registry == null) {
			_registry = new SOAPropertiesRegistry();
			init(_registry);
		}
		return _registry;
	}
	
	private static void init(SOAPropertiesRegistry registry) {
		// TODO: investigate OSGi support and whether or not we need to listen for newly enabled plugins
		// which contribute, and to listen for current contributing plugins and whether they get disabled.
		IExtensionPoint extensionPoint = Platform
				.getExtensionRegistry()
				.getExtensionPoint(
						Activator.getDefault().getBundle().getSymbolicName(),
						EXTENSION_POINT_PROPERTIES_CONTRIBUTION);
		if (extensionPoint != null) {
			IConfigurationElement[] policyExtensions = extensionPoint
					.getConfigurationElements();

			for (int i = 0; i < policyExtensions.length; i++) {
				IConfigurationElement element = policyExtensions[i];
				try {
					ISOAPropertiesEntry entry = new SOAPropertiesEntry(element);
					registry.addEntry(entry);
				} catch (IllegalArgumentException exc) {
				}
			}
		}
	}
	
	/**
	 * Adds an entry to the registry
	 * @param entry
	 */
	public void addEntry(ISOAPropertiesEntry entry) {
		QName key = new QName(entry.getNamespaceURI(), entry.getTypeName());
		for (Enumeration enum1 = getEntries().elements(); enum1.hasMoreElements();) {
			ISOAPropertiesEntry contribution = (ISOAPropertiesEntry)enum1.nextElement();
			if (contribution.getId().equals(entry.getId())) {
				Activator.internalReportConfigurationError(Messages.Duplicate_id,
						new String[] {entry.getId(),
						entry.getContributorPluginID(),
						contribution.getContributorPluginID()});
				return;
			}
		}
		ISOAPropertiesEntry existingEntry = (ISOAPropertiesEntry)getEntries().get(key);
		if (existingEntry != null) {
			Activator.internalReportConfigurationError(Messages.Duplicate_name,
					new String[] {entry.getId(),
					existingEntry.getId(),
					entry.getNamespaceURI(),
					entry.getTypeName()});
			return;
		}
		getEntries().put(key, entry);
	}
	
	/**
	 * Get the entry for a type
	 * 
	 * @param namespaceURI - the namespace of the type
	 * @param typeName - the name of the type
	 * 
	 * @return the entry for the type
	 */
	public ISOAPropertiesEntry getEntry(String namespaceURI, String typeName) {
		QName key = new QName(namespaceURI, typeName);
		return (ISOAPropertiesEntry)getEntries().get(key);
	}
	
	/**
	 * get an enumeration to walk the entries in the registry
	 * 
	 * @return an enumeration of entries in the registry.  Entries are defined by class PropertiesContributionEntry
	 */
	public Enumeration getEntryEnumeration() {
		return getEntries().elements();
	}
	
	private Hashtable getEntries() {
		if (_entries == null) {
			_entries = new Hashtable();
		}
		return _entries;
	}
	
   // bug 149833 - remove dependency on org.apache.xml.utils.QName and replace it with 
   // our own implementation (below).
   
	private class QName {
		
		private String _nameSpace;
		private String _localName;
		
		private final String TOSTRING_OPEN_BRACE 		= "{"; //$NON-NLS-1$
		private final String TOSTRING_CLOSE_BRACE 	= "}"; //$NON-NLS-1$
		
		public QName(String nameSpace, String localName) {
			Assert.isNotNull(nameSpace);
			Assert.isNotNull(localName);
			
			_nameSpace = nameSpace;
			_localName = localName;
		}
		
		public String getNamespace() {
			return this._nameSpace;
		}
		
		public String getLocalName() {
			return this._localName;
		}
		
		public boolean equals(Object obj) {
			
			boolean isEquals = false;
			if (obj instanceof QName) {
				QName qname = (QName) obj;
				
				if (_localName.equals( qname.getLocalName() )) {
					if (_nameSpace.equals( qname.getNamespace() )) {
						isEquals = true;
					}
				}
			}
			
			return isEquals;
			
		}
		
		public int hashCode() {
			
			int hash = 0;
			hash += _localName.hashCode();
			hash += _nameSpace.hashCode();
			return hash;
			
		}
		
		public String toString() {
			return TOSTRING_OPEN_BRACE + _nameSpace + TOSTRING_CLOSE_BRACE + _localName;
		}
		

	}
}
