/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.create.operations;

import java.util.Set;

import org.apache.xerces.util.XMLChar;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.osgi.util.NLS;
import org.eclipse.stp.core.infrastructure.operations.AbstractScribblerDataModelProvider;
import org.eclipse.stp.core.internal.Messages;
import org.eclipse.stp.core.internal.STPCorePlugin;
import org.eclipse.stp.core.sca.Composite;
import org.eclipse.wst.common.frameworks.datamodel.IDataModelOperation;

public class ComponentCreationDataModelProvider extends
      AbstractScribblerDataModelProvider implements
      IComponentCreationDataModelProperties {
   public final static String COMPONENT_DEFAULT_NAME = "Component"; //$NON-NLS-1$

   public IDataModelOperation getDefaultOperation() {
      return new ComponentCreationOperation(model);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.eclipse.wst.common.frameworks.datamodel.IDataModelProvider#getPropertyNames()
    */
   public Set getPropertyNames() {
      Set propertyNames = super.getPropertyNames();
      propertyNames.add(NAME);
      propertyNames.add(COMPOSITE);

      return propertyNames;
   }

   public Object getDefaultProperty(String propertyName) {
      if (propertyName.equals(NAME))
         return getDefaultName();

      return super.getDefaultProperty(propertyName);
   }

   protected String getDefaultName() {
      String defaultName = COMPONENT_DEFAULT_NAME;
      Composite composite = (Composite) model.getProperty(COMPOSITE);
      int i = 0;
      if (composite != null) {
         while (composite.getComponent(defaultName) != null) {
            defaultName = COMPONENT_DEFAULT_NAME + i;
            i++;
         }
      }

      return defaultName;
   }

   public IStatus validate(String propertyName) {
      if (propertyName.equals(NAME))
         return validateName();

      if (propertyName.equals(COMPOSITE))
         return validateComposite();

      return super.validate(propertyName);
   }

   private IStatus validateComposite() {
      IStatus returnStatus = OK_STATUS;
      Composite composite = (Composite) getProperty(COMPOSITE);
      if (composite == null)
         returnStatus = STPCorePlugin.createErrorStatus(
               Messages.invalidObjectPassed_errorText, null);

      if (!returnStatus.isOK())
         return returnStatus;

      return returnStatus;
   }

   private IStatus validateName() {
      IStatus returnStatus = OK_STATUS;
      // Multiple check needs to be perfomed.

      // First Check for empty
      String name = getStringProperty(NAME);
      if (name == null || name.length() <= 0)
         returnStatus = STPCorePlugin.createErrorStatus(
               Messages.nameBlank_errorText, null);

      if (!returnStatus.isOK())
         return returnStatus;

      // Second check for valid characters
      returnStatus = validateNameForValidCharaters(name);
      if (!returnStatus.isOK())
         return returnStatus;

      // Third check if the name entered is unique.
      Composite composite = (Composite) model.getProperty(COMPOSITE);
      if (composite != null) {
         if (composite.getComponent(name) != null) {
            returnStatus = STPCorePlugin.createErrorStatus(NLS.bind(
                  Messages.nameAlreadyInUse_errorText, name), null);
         }
      }

      return returnStatus;
   }

   // The following characters are restricted characters of
   // org.eclipse.emf.common.util.URI which is
   // used while loading the extension model (.modulex) file.
   protected static final String URI_RESTRICTED_CHARACTERS = ":?#@!"; //$NON-NLS-1$

   private IStatus validateNameForValidCharaters(String name) {
      if (!XMLChar.isValidNCName(name)) {
         return STPCorePlugin.createErrorStatus(Messages.bind(
               Messages.nameNotValid_errorText, name), null);
      }
      for (int i = 0; i < URI_RESTRICTED_CHARACTERS.length(); i++) {
         if (name.indexOf(URI_RESTRICTED_CHARACTERS.charAt(i)) != -1) {
            return STPCorePlugin.createErrorStatus(Messages.bind(
                  Messages.portionOfNameNotValid_errorText, name, String
                        .valueOf(URI_RESTRICTED_CHARACTERS.charAt(i))), null);
         }
      }

      return Status.OK_STATUS;

   }
}
