/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.internal.introspection;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.ExtendedMetaData;
import org.eclipse.emf.ecore.util.FeatureMap;
import org.eclipse.stp.core.infrastructure.assertion.Assert;
import org.eclipse.stp.core.introspection.FeatureAdapter;
import org.eclipse.stp.core.introspection.IComponentTypeIntrospector;
import org.eclipse.stp.core.introspection.IModelIntrospectionManager;
import org.eclipse.stp.core.introspection.IShareableComponentTypeFactory;
import org.eclipse.stp.core.sca.AbstractImplementation;
import org.eclipse.stp.core.sca.ComponentType;
import org.eclipse.stp.core.sca.Implementation;
import org.eclipse.stp.core.sca.SCACoreRoot;
import org.eclipse.stp.core.sca.SCAFactory;
import org.eclipse.stp.core.sca.SCAObject;
import org.eclipse.stp.core.sca.SCAPackage;
import org.eclipse.stp.core.sca.UnknownImplementation;
import org.eclipse.stp.core.sca.impl.ComponentTypeImpl;

/**
 * Provides the implementaiton of
 * {@link org.eclipse.stp.core.introspection.IModelIntrospectionManager}.
 * 
 * Clients should access this implementation through
 * {@link org.eclipse.stp.core.introspection.IModelIntrospectionManager#eINSTANCE}.
 * 
 * @see org.eclipse.stp.core.introspection.IModelIntrospectionManager#eINSTANCE
 * @since 1.0
 * 
 */
public class ModelIntrospectionManager implements IModelIntrospectionManager {

   public void bind(ComponentType componentType, URI implementation) {

      synchronized (componentType) {
         if (findAdapter(componentType) == null) {
            ((ComponentTypeImpl) componentType).unbind();
            IComponentTypeIntrospector introspector = ComponentTypeIntrospectorRegistry
                  .getInstance().createAndBindIntrospector(componentType,
                        implementation);
            IStatus validComponentTypeStatus = introspector
                  .hasValidComponentType();
            if (!validComponentTypeStatus.isOK())
               componentType.addStatus(validComponentTypeStatus);
         }
      }
   }

   public void unbind(ComponentType componentType) {
      // will change to look up correct Introspector; for now we use a single
      // introspecting type
      synchronized (componentType) {
         IComponentTypeIntrospector introspector = findAdapter(componentType);
         if (introspector != null) {
            componentType.getEObject().eAdapters().remove(introspector);
            introspector.dispose();
            ((ComponentTypeImpl) componentType).unbind();
            componentType.clearStatus();
         }
      }
   }

   public ComponentType resolve(Implementation implementation) {
      Assert.isTrue(implementation != null);
      Assert.isTrue(implementation.getEObject() != null);
      /*
       * The following assert ensures that: (1) The model class is
       * AbstractImplementation OR (2) The implementation is contained by a
       * resource set =implies=> Implementation > Component > Composite/Module >
       * SCACoreRoot > Resource > ResourceSet.
       */
      Assert
            .isTrue(implementation.getEObject().eClass().equals(
                  SCAPackage.eINSTANCE.getAbstractImplementation())
                  || (implementation.getEObject().eResource() != null && implementation
                        .getEObject().eResource().getResourceSet() != null));

      synchronized (implementation) {

         EObject eObject = implementation.getEObject();
         EClass eClass = eObject.eClass();
         if (SCAPackage.eINSTANCE.getAbstractImplementation().equals(eClass))
            return ((AbstractImplementation) implementation).getComponentType();

         if (SCAPackage.eINSTANCE.getUnknownImplementation().equals(eClass)) {
            UnknownImplementation unknown = (UnknownImplementation) implementation;
            if (eObject.eResource() != null) {
               ResourceSet resourceSet = eObject.eResource().getResourceSet();
               if (resourceSet != null) {
                  Resource componentTypeResource = resourceSet.getResource(URI
                        .createURI(unknown.getUri()), true);
                  SCACoreRoot root = (SCACoreRoot) componentTypeResource
                        .getContents().get(0);
                  return root.getComponentType();
               }
            }
            return null;
         }

         IShareableComponentTypeFactory factory = ComponentTypeIntrospectorRegistry
               .getInstance().getShareableFactory(implementation);

         if (factory == null) {
            ComponentType componentType = SCAFactory.eINSTANCE
                  .createComponentType();
            IComponentTypeIntrospector introspector = ComponentTypeIntrospectorRegistry
                  .getInstance().createIntrospector(implementation);
            if (introspector != null) {
               componentType.getEObject().eAdapters().add(introspector);
               IStatus validComponentTypeStatus = introspector
                     .hasValidComponentType();
               if (!validComponentTypeStatus.isOK())
                  componentType.addStatus(validComponentTypeStatus);
            }
            return componentType;
         }

         if (eObject.eResource() != null) {
            ResourceSet resourceSet = eObject.eResource().getResourceSet();
            if (resourceSet != null) {
               EStructuralFeature feature = getStructuralFeature(implementation);
               IFile shareableFile = factory.findShareableFile(feature,
                     implementation);
               if (shareableFile != null) {
                  URI readyShareableURI = URI
                        .createURI("comptype:" + shareableFile.getFullPath().toString()); //$NON-NLS-1$
                  Resource componentTypeResource = resourceSet.getResource(
                        readyShareableURI, true);
                  if (componentTypeResource.getContents().size() > 0) {
                     SCACoreRoot root = (SCACoreRoot) componentTypeResource
                           .getContents().get(0);
                     return root.getComponentType();
                  }
               }
            }
         }
      }
      return null;
   }

   public ComponentType resolve(Implementation implementation, IProject context) {
      Assert.isTrue(implementation != null);
      Assert.isTrue(implementation.getEObject() != null);
      /*
       * The following assert ensures that: (1) The model class is
       * AbstractImplementation OR (2) The implementation is contained by a
       * resource set =implies=> Implementation > Component > Composite/Module >
       * SCACoreRoot > Resource > ResourceSet.
       */
      Assert
            .isTrue(implementation.getEObject().eClass().equals(
                  SCAPackage.eINSTANCE.getAbstractImplementation())
                  || (implementation.getEObject().eResource() != null && implementation
                        .getEObject().eResource().getResourceSet() != null));

      synchronized (implementation) {

         EObject eObject = implementation.getEObject();
         EClass eClass = eObject.eClass();
         if (SCAPackage.eINSTANCE.getAbstractImplementation().equals(eClass))
            return ((AbstractImplementation) implementation).getComponentType();

         if (SCAPackage.eINSTANCE.getUnknownImplementation().equals(eClass)) {
            UnknownImplementation unknown = (UnknownImplementation) implementation;
            if (eObject.eResource() != null) {
               ResourceSet resourceSet = eObject.eResource().getResourceSet();
               if (resourceSet != null) {
                  Resource componentTypeResource = resourceSet.getResource(URI
                        .createURI(unknown.getUri()), true);
                  SCACoreRoot root = (SCACoreRoot) componentTypeResource
                        .getContents().get(0);
                  return root.getComponentType();
               }
            }
            return null;
         }

         IShareableComponentTypeFactory factory = ComponentTypeIntrospectorRegistry
               .getInstance().getShareableFactory(implementation);

         if (factory == null) {
            ComponentType componentType = SCAFactory.eINSTANCE
                  .createComponentType();
            IComponentTypeIntrospector introspector = ComponentTypeIntrospectorRegistry
                  .getInstance().createIntrospector(implementation);
            if (introspector != null) {
               componentType.getEObject().eAdapters().add(introspector);
               IStatus validComponentTypeStatus = introspector
                     .hasValidComponentType();
               if (!validComponentTypeStatus.isOK())
                  componentType.addStatus(validComponentTypeStatus);
            }
            return componentType;
         }

         if (eObject.eResource() != null) {
            ResourceSet resourceSet = eObject.eResource().getResourceSet();
            if (resourceSet != null) {
               EStructuralFeature feature = getStructuralFeature(implementation);
               IFile shareableFile = factory.findShareableFile(feature,
                     implementation);
               if (shareableFile != null) {
                  URI readyShareableURI = URI
                        .createURI("comptype:" + shareableFile.getFullPath().toString()); //$NON-NLS-1$
                  Resource componentTypeResource = resourceSet.getResource(
                        readyShareableURI, true);
                  if (componentTypeResource.getContents().size() > 0) {
                     SCACoreRoot root = (SCACoreRoot) componentTypeResource
                           .getContents().get(0);
                     return root.getComponentType();
                  }
               }
            }
         }
      }
      return null;
   }

   /**
    * This method assumes that the Implementation is contained so that it can
    * check the FeatureMap of the containing model element. Otherwise, the
    * method checks for the FeatureAdapter on the Implementation. If neither of
    * these attempts succeeds, then this method will fail miserably (e.g. return
    * null).
    * 
    * <p>
    * This method asserts its expectations so don't get tricky.
    * </p>
    * 
    * @param implementation
    *           The Implementation which must either be contained or have a
    *           FeatureAdapter.
    * @return The structural feature if found.
    */
   public static EStructuralFeature getStructuralFeature(SCAObject anSCAObject) {
      FeatureAdapter adapter = null;
      if (anSCAObject == null)
         return null;
      Assert
            .isTrue((anSCAObject.getEObject().eContainer() != null && anSCAObject
                  .getEObject().eContainer().eContainingFeature() != null)
                  || (adapter = FeatureAdapter.findAdapter(anSCAObject)) != null);

      if (adapter != null)
         return adapter.getFeature();

      /* With the above assertion in place, feature cannot be null */
      EStructuralFeature feature = anSCAObject.getEObject()
            .eContainingFeature();
      if (((EStructuralFeature.Internal) feature).isFeatureMap()) {
         FeatureMap map = (FeatureMap) anSCAObject.getEObject().eContainer()
               .eGet(feature);
         return map.getEStructuralFeature(0);
      }
      return feature;
   }

   public static String getSubstitutionGroupName(SCAObject anSCAObject) {
      return ExtendedMetaData.INSTANCE
            .getName(getStructuralFeature(anSCAObject));
   }

   private static IComponentTypeIntrospector findAdapter(
         ComponentType componentType) {
      return (IComponentTypeIntrospector) EcoreUtil.getAdapter(componentType
            .getEObject().eAdapters(), IComponentTypeIntrospector.ADAPTER_TYPE);
   }

}
