/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.ui.xef.editor;

import java.util.Map;


import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.stp.ui.xef.PluginResources;
import org.eclipse.stp.xef.util.QNameHelper;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

public class QNameFieldEditor extends AbstractFieldEditor {
    public static final QNameFieldEditor SINGLETON = new QNameFieldEditor(); 

    protected Combo cboPrefix;
    protected Text txtNamespace;
    protected Text txtLocal;
    protected Button btnOk;
    private Map<String, String> nsMap;
    private String strQName;
    
    public QNameFieldEditor() {
        this(null);
    }
    
    public QNameFieldEditor(Shell parent) {
        super(parent);
    }
    
    /**
     * Creates and returns the contents of the upper part
     * of this dialog (above the button bar).
     *
     * @param the parent composite to contain the dialog area
     * @return the dialog area control
     */
    protected Control createDialogArea(Composite parent) {
        // Run super.
        Composite area = (Composite) super.createDialogArea(parent);
        final GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        gridLayout.makeColumnsEqualWidth = false;
        area.setLayout(gridLayout);

        // First part
        Label lblInstructions = new Label(area, SWT.NONE);
        lblInstructions.setText(PluginResources.getString("qname.info"));
        GridData gridData = new GridData();
        gridData.horizontalSpan = 2;
        lblInstructions.setLayoutData(gridData);

        // The name intake panel
        Label lblPrefix = new Label(area, SWT.LEFT);
        lblPrefix.setText(PluginResources.getString("qname.prefix"));
        
        cboPrefix = new Combo(area, SWT.BORDER);
        cboPrefix.setItems((String[])nsMap.keySet().toArray(new String[nsMap.keySet().size()]));
        cboPrefix.addModifyListener(new ModifyListener() {
           public void modifyText(ModifyEvent event) {
               Combo combo = (Combo) event.widget;
               String selItem = combo.getText();
               if (nsMap.containsKey(selItem)) {
                   txtNamespace.setEditable(false);
                   txtNamespace.setText(nsMap.get(selItem));
               } else {
                   if (!txtNamespace.getEditable()) {
                       txtNamespace.setText("");
                       txtNamespace.setEditable(true);
                   }
               }
               updateDialogState();
           }
        });
        
        Label lblNamespaceName = new Label(area, SWT.LEFT);
        lblNamespaceName.setText(PluginResources.getString("qname.namespace"));

        txtNamespace = new Text(area, SWT.BORDER);
        txtNamespace.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent event) {
                txtNamespace.setToolTipText(txtNamespace.getText());
                updateDialogState();
            }
        });
        gridData = new GridData();
        gridData.widthHint = 200;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalAlignment = GridData.FILL;
        txtNamespace.setLayoutData(gridData);

        // The name intake panel
        Label lblLocalName = new Label(area, SWT.LEFT);
        lblLocalName.setText(PluginResources.getString("qname.local"));

        txtLocal = new Text(area, SWT.BORDER);
        txtLocal.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent event) {
                updateDialogState();
            }
        });
        gridData = new GridData();
        gridData.widthHint = 200;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalAlignment = GridData.FILL;
        txtLocal.setLayoutData(gridData);
       
        return area;
    }
    
    protected void configureShell(Shell shell) {
        super.configureShell(shell);
        shell.setText(PluginResources.getString("qname.label"));
    }
    
    protected void createButtonsForButtonBar(Composite parent) {
        btnOk = createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);
        btnOk.setEnabled(false);
        createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
    }
    
    /**
     * Update the state of the dialog
     */
    private void updateDialogState() {
        if (btnOk != null) {
            btnOk.setEnabled(isDialogComplete());
        }
        return;
    }
    
    /**
     * Validate that all required input has been provided
     */
    private boolean isDialogComplete() {
        if (cboPrefix.getText().length() == 0) {
            return false;
        }
        
        if (txtNamespace.getText().length() == 0) {
            return false;
        }
        
        if (txtLocal.getText().length() == 0) {
            return false;
        }

        return true;
    }
    
    protected void okPressed() {
        if (createQName()) {
            super.okPressed();
        }
    }

    protected boolean createQName() {
        if (txtNamespace.getEditable()) {
            nsMap.put(cboPrefix.getText(), txtNamespace.getText());
        }
        strQName = QNameHelper.convertToQName(cboPrefix.getText(), txtLocal.getText());
        return true;
    }

    @SuppressWarnings("unchecked")
    public void setFieldData(Object data) {
        if (data instanceof Map) {
            nsMap = (Map) data;
        }
    }
    
    public void setFieldText(String text) {
        if (cboPrefix != null) {
            cboPrefix.setText(QNameHelper.getPrefixFromQName(text));
        }
            
        if (txtLocal != null) {
            txtLocal.setText(QNameHelper.getLocalFromQName(text));
        }
    }
    
    public String getFieldText() {
        return strQName;
    }
    

}
