/*******************************************************************************
 * Copyright (c) 2004-2006 Sybase, Inc.
 * 
 * All rights reserved. This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Sybase, Inc. - initial API and implementation
 ******************************************************************************/
package org.eclipse.stp.soas.deploy.core;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.TreeSet;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.util.Assert;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.internal.model.RuntimeCore;
import org.eclipse.stp.soas.internal.deploy.core.ConfigurablePackageExtension;
import org.eclipse.stp.soas.internal.deploy.core.LogicalPackageExtension;
import org.eclipse.stp.soas.internal.deploy.core.PackageConstructorExtension;
import org.eclipse.stp.soas.internal.deploy.core.PackageExtension;
import org.eclipse.stp.soas.internal.deploy.core.PackageExtensionException;
import org.eclipse.stp.soas.internal.deploy.core.PhysicalPackageExtension;
import org.eclipse.stp.soas.internal.deploy.core.RuntimeDefinition;
import org.eclipse.stp.soas.internal.deploy.core.ServerDefinition;
import org.eclipse.stp.soas.internal.deploy.core.TechnologyDefinition;
import org.eclipse.stp.soas.internal.deploy.core.TechnologyMap;

/**
 * @author rcernich
 * 
 * Created on Mar 18, 2004
 */
public class DeploymentExtensionManager extends DeployPackageManager {

	private static final LoggingProxy LOG = LoggingProxy
			.getlogger(DeploymentExtensionManager.class);

	public static final String EXTENSION_ID = "deploymentExtension"; //$NON-NLS-1$

	public static final String EXT_ELEM_LOGICAL_PACKAGE = "logicalPackage"; //$NON-NLS-1$
	public static final String EXT_ELEM_CONFIGURABLE_PACKAGE = "configurablePackage"; //$NON-NLS-1$
	public static final String EXT_ELEM_PHYSICAL_PACKAGE = "physicalPackage"; //$NON-NLS-1$
	public static final String EXT_ELEM_DEPLOY_DRIVER = "deployDriver"; //$NON-NLS-1$
	public static final String EXT_ELEM_SUPPORTED_SERVER = "supportedServer"; //$NON-NLS-1$
	public static final String EXT_ELEM_SUPPORTED_RUNTIME = "supportedRuntime"; //$NON-NLS-1$
	public static final String EXT_ELEM_SUPPORTED_TECHNOLOGY = "supportedTechnology"; //$NON-NLS-1$
	public static final String EXT_ELEM_TECHNOLOGY_DEFINITION = "technologyDefinition"; //$NON-NLS-1$
	public static final String EXT_ELEM_SERVER_DEFINITION = "serverDefinition"; //$NON-NLS-1$
	public static final String EXT_ELEM_TECHNOLOGY_MAP = "technologyMap"; //$NON-NLS-1$
	public static final String EXT_ELEM_PACKAGE_CONSTRUCTOR = "packageConstructor"; //$NON-NLS-1$

	public static final String EXT_POINT_WTP_MODULETYPE = "org.eclipse.wst.server.core.moduleTypes";
	public static final String EXT_POINT_WTP_SERVERTYPE = "org.eclipse.wst.server.core.serverTypes";
	public static final String EXT_POINT_WTP_RUNTIMETYPE = "org.eclipse.wst.server.core.runtimeTypes";

	public static boolean DEBUG_DEPLOYMENT_EXTENSION = true;

	private static final String OPTION_DEBUG_DEPLOYMENT_EXTENSION = "org.eclipse.stp.soas.deploy.core/deploymentextension"; //$NON-NLS-1$
	private static DeploymentExtensionManager sInstance;

	private Map mTechnologyDefsByID;
	private Map mServerDefsByID;
	private Map mRuntimeDefsByID;
	private Map mIdToConstructor;
	
	public static DeploymentExtensionManager getInstance() {
		if (sInstance == null) {
			String debug = Platform
					.getDebugOption(OPTION_DEBUG_DEPLOYMENT_EXTENSION);
			DEBUG_DEPLOYMENT_EXTENSION = debug == null ? false : (debug
					.equalsIgnoreCase("true") ? true : false); //$NON-NLS-1$
			sInstance = new DeploymentExtensionManager();
			sInstance.init();
		}
		return sInstance;
	}

	private DeploymentExtensionManager() {
		super();
	}

	public ITechnologyDefinition getTechnologyDefinition(String type) {
		ITechnologyDefinition itd;
		if (type == null || !mTechnologyDefsByID.containsKey(type)) {
			itd = null;
		} else {
			itd = (ITechnologyDefinition) mTechnologyDefsByID.get(type);
		}
		return itd;
	}

	public IServerDefinition getServerDefinition(String type) {
		IServerDefinition isd;
		if (type == null || !mServerDefsByID.containsKey(type)) {
			isd = null;
		} else {
			isd = (IServerDefinition) mServerDefsByID.get(type);
		}
		return isd;
	}

	public List<IRuntimeDefinition> getRuntimeDefinitionByType(String type) {
		List<IRuntimeDefinition> isd = new ArrayList<IRuntimeDefinition>();

		for (Iterator it = mRuntimeDefsByID.keySet().iterator(); it.hasNext();) {
			String rdId = (String) it.next();
			String rdIdWithoutVersion = RuntimeCore
					.getRuntimeIdWithoutVersion(rdId);
			if (type.equals(rdIdWithoutVersion)) {
				isd.add((IRuntimeDefinition) mRuntimeDefsByID.get(rdId));
			}
		}

		if (type != null && mRuntimeDefsByID.containsKey(type)) {
			isd.add((IRuntimeDefinition) mRuntimeDefsByID.get(type));
		}

		return isd;
	}

	public IRuntimeDefinition getRuntimeDefinition(String id) {

		IRuntimeDefinition ird;
		if (id == null || !mRuntimeDefsByID.containsKey(id)) {
			ird = null;
		} else {
			ird = (IRuntimeDefinition) mRuntimeDefsByID.get(id);
		}
		return ird;
	}

	public List getPackageConstructors() {
		return new ArrayList(mIdToConstructor.values());
	}

	public IPackageConstructorExtension getPackageConstructor(
			ILogicalPackage pkg, IServerType serverType) {
		List constructors = getPackageConstructors();

		for (Iterator it = constructors.iterator(); it.hasNext();) {
			IPackageConstructorExtension pce = (IPackageConstructorExtension) it
					.next();

			if (!pce.supportsPackage(pkg)) { 
				continue; 
			}			

			if (pce.supportsServer(serverType)) {
				return pce;
			}

		}
		return null;
	}

	public List getPackageConstructor(ILogicalPackage pkg) {

		List<IPackageConstructorExtension> lstPce = new ArrayList<IPackageConstructorExtension>();
		List constructors = getPackageConstructors();

		for (Iterator it = constructors.iterator(); it.hasNext();) {
			IPackageConstructorExtension pce = (IPackageConstructorExtension) it
					.next(); 

			if (!pce.supportsPackage(pkg)) {
				continue;
			}

			lstPce.add(pce);

		}

		return lstPce;

	}

	private void init() {
		mTechnologyDefsByID = new TreeMap();
		mServerDefsByID = new TreeMap();
		mIdToLogicalExtension = new TreeMap(new IDComparator());
		mLogicalExtensionByFileExtension = new TreeSet(
				new FileExtensionComparator());
		mIdToConfigurableExtension = new TreeMap(new IDComparator());
		mConfigurableExtensionByFileExtension = new TreeSet(
				new FileExtensionComparator());
		mIdToPhysicalExtension = new TreeMap(new IDComparator());
		mPhysicalExtensionByFileExtension = new HashMap();

		mRuntimeDefsByID = new TreeMap();
		mIdToConstructor = new TreeMap();

		processWtpExtensions();
		processExtensions();
	}

	/**
	 * we need to read module type, server type and 
	 * runtime type from wtp extension
	 */
	private void processWtpExtensions() {
		// process technolody definitions
		Collection technologyDefs = new ArrayList();
		IExtensionPoint moduleTypePoint = Platform.getExtensionRegistry()
				.getExtensionPoint(EXT_POINT_WTP_MODULETYPE);
		IExtension[] moduleExts = moduleTypePoint.getExtensions();
		for (IExtension moduleExt : moduleExts) {
			for (IConfigurationElement elem : moduleExt
					.getConfigurationElements()) {
				technologyDefs.add(elem);
			}
		}
		processTechnologyDefs(technologyDefs);

		// process server type definition
		Collection serverDefs = new ArrayList();
		IExtensionPoint serverTypePoint = Platform.getExtensionRegistry()
				.getExtensionPoint(EXT_POINT_WTP_SERVERTYPE);
		IExtension[] serverExts = serverTypePoint.getExtensions();
		for (IExtension serverExt : serverExts) {
			for (IConfigurationElement elem : serverExt
					.getConfigurationElements()) {
				serverDefs.add(elem);
			}
		}
		processServerDefs(serverDefs);

		// process runtime type definition
		Collection runtimeDefs = new ArrayList();
		IExtensionPoint runtimeTypePoint = Platform.getExtensionRegistry()
				.getExtensionPoint(EXT_POINT_WTP_RUNTIMETYPE);
		IExtension[] runtimeExts = runtimeTypePoint.getExtensions();
		for (IExtension runtimeExt : runtimeExts) {
			for (IConfigurationElement elem : runtimeExt
					.getConfigurationElements()) {
				runtimeDefs.add(elem);
			}
		}
		processRuntimeDefs(runtimeDefs);

	}

	private void processExtensions() {
		IExtensionPoint exp = Platform.getExtensionRegistry()
				.getExtensionPoint(
						DeployCorePlugin.getDefault().getBundle()
								.getSymbolicName(), EXTENSION_ID);
		IExtension[] exts = exp.getExtensions();
		Collection logicalPackages = new ArrayList();
		Collection configurablePackages = new ArrayList();
		Collection physicalPackages = new ArrayList();
		Collection deployDrivers = new ArrayList();
		Collection technologyDefs = new ArrayList();
		Collection serverDefs = new ArrayList();
		Collection technologyMaps = new ArrayList();
		Collection constructors = new ArrayList();
		for (Iterator xit = Arrays.asList(exts).iterator(); xit.hasNext();) {
			IExtension ext = (IExtension) xit.next();
			IConfigurationElement[] elems = ext.getConfigurationElements();
			for (Iterator eit = Arrays.asList(elems).iterator(); eit.hasNext();) {
				IConfigurationElement elem = (IConfigurationElement) eit.next();
				String elemName = elem.getName();
				if (EXT_ELEM_LOGICAL_PACKAGE.equals(elemName)) {
					logicalPackages.add(elem);
				} else if (EXT_ELEM_CONFIGURABLE_PACKAGE.equals(elemName)) {
					configurablePackages.add(elem);
				} else if (EXT_ELEM_PHYSICAL_PACKAGE.equals(elemName)) {
					physicalPackages.add(elem);
				} else if (EXT_ELEM_DEPLOY_DRIVER.equals(elemName)) {
					deployDrivers.add(elem);
				} else if (EXT_ELEM_TECHNOLOGY_DEFINITION.equals(elemName)) {
					//todo: remove the tech def loading, use extensions from wtp instead.
					technologyDefs.add(elem);
				} else if (EXT_ELEM_SERVER_DEFINITION.equals(elemName)) {
					//todo: remove the server def loading, use extensions from wtp instead.
					serverDefs.add(elem);
				} else if (EXT_ELEM_TECHNOLOGY_MAP.equals(elemName)) {
					technologyMaps.add(elem);
				} else if (EXT_ELEM_PACKAGE_CONSTRUCTOR.equals(elemName)) {
					constructors.add(elem);
				} else {
					if (DEBUG_DEPLOYMENT_EXTENSION) {
						System.err
								.println(DeployCorePlugin
										.getDefault()
										.getResourceString(
												"DeploymentExtensionManager.trace.error.unknownElement", //$NON-NLS-1$
												new Object[] {
														elemName,
														ext.getContributor()
																.getName() }));
						System.err.flush();
					}
				}
			}
		}
		// Process the extension elements
		// technology types first (depended upon by packages, technology maps)
		processTechnologyDefs(technologyDefs);
		// server types (depended upon by deploy drivers, physical packages)
		processServerDefs(serverDefs);
		// server technology maps
		processTechnologyMaps(technologyMaps);
		// logical packages (depended upon by physical packages)
		processLogicalPackages(logicalPackages);
		// configurable packages
		processConfigurablePackages(configurablePackages);
		// physical packages
		processPhysicalPackages(physicalPackages);

		// package constructors
		processPackageConstructors(constructors);
	}

	private void processPackageConstructors(Collection constructors) {
		for (Iterator it = constructors.iterator(); it.hasNext();) {
			processPackageConstructor((IConfigurationElement) it.next());
		}
	}

	private void processPackageConstructor(IConfigurationElement element) {

		try {
			PackageConstructorExtension de = new PackageConstructorExtension(
					element);
			Assert
					.isTrue(
							!mIdToConstructor.containsKey(de.getID()),
							DeployCorePlugin
									.getDefault()
									.getResourceString(
											"DeploymentExtensionManager.assert.duplicatePackageConstructor", //$NON-NLS-1$
											new Object[] { de.getID(),
													element.toString(),
													de.getID() }));
			mIdToConstructor.put(de.getID(), de);
		} catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.packageConstructorParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(PackageConstructorExtension.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor()
														.getName() }));
				e.printStackTrace();
				System.err.flush();
			}
		}

	}

	private void processTechnologyDefs(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processTechnologyDef((IConfigurationElement) it.next());
		}
	}

	private void processTechnologyDef(IConfigurationElement element) {
		try {
			ITechnologyDefinition itd = new TechnologyDefinition(element);
			if (mTechnologyDefsByID.containsKey(itd.getID())) {
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.duplicateTechnologyDefinitionId")); //$NON-NLS-1$
			}
			mTechnologyDefsByID.put(itd.getID(), itd);
		} catch (PackageExtensionException e) {
			LOG
					.error(DeployCorePlugin
							.getDefault()
							.getResourceString(
									"DeploymentExtensionManager.trace.error.technologyDefinitionParseError", //$NON-NLS-1$
									new Object[] {
											element
													.getAttribute(ServerDefinition.ATTR_ID),
											element.getDeclaringExtension()
													.getContributor().getName() }));
		}
	}

	private void processServerDefs(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processServerDef((IConfigurationElement) it.next());
		}
	}

	private void processServerDef(IConfigurationElement element) {
		try {
			IServerDefinition isd = new ServerDefinition(element);
			if (mServerDefsByID.containsKey(isd.getID())) {
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.duplicateServerDefinitionId")); //$NON-NLS-1$
			}
			mServerDefsByID.put(isd.getID(), isd);
		} catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.serverDefinitionParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(ServerDefinition.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor()
														.getName() }));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private void processRuntimeDefs(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processRuntimeDef((IConfigurationElement) it.next());
		}
	}

	private void processRuntimeDef(IConfigurationElement element) {
		try {
			IRuntimeDefinition isd = new RuntimeDefinition(element);
			if (mRuntimeDefsByID.containsKey(isd.getID())) {
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.duplicateRuntimeDefinitionId")); //$NON-NLS-1$
			}
			mRuntimeDefsByID.put(isd.getID(), isd);
		} catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.serverDefinitionParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(ServerDefinition.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor()
														.getName() }));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private void processTechnologyMaps(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processTechnologyMap((IConfigurationElement) it.next());
		}
	}

	private void processTechnologyMap(IConfigurationElement element) {
		try {
			new TechnologyMap(element);
		} catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.technologyMapParseError", //$NON-NLS-1$
										new Object[] { element
												.getDeclaringExtension()
												.getContributor().getName() }));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private void processLogicalPackages(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processLogicalPackage((IConfigurationElement) it.next());
		}
	}

	private void processLogicalPackage(IConfigurationElement element) {
		try {
			LogicalPackageExtension lpp = new LogicalPackageExtension(element);
			if (mIdToLogicalExtension.containsKey(lpp.getID())) {
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.duplicateLogicalPackageId")); //$NON-NLS-1$
			}

			mIdToLogicalExtension.put(lpp.getID(), lpp);
			mLogicalExtensionByFileExtension.add(lpp);
		} catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.logicalPackageParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(PackageExtension.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor()
														.getName() }));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private void processConfigurablePackages(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processConfigurablePackage((IConfigurationElement) it.next());
		}
	}

	private void processConfigurablePackage(IConfigurationElement element) {
		try {
			ConfigurablePackageExtension cpe = new ConfigurablePackageExtension(
					element);
			if (mIdToConfigurableExtension.containsKey(cpe.getID())) {
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.duplicateConfigurablePackageId")); //$NON-NLS-1$
			}
			mIdToConfigurableExtension.put(cpe.getID(), cpe);
			mConfigurableExtensionByFileExtension.add(cpe);
		} catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.configurablePackageParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(PackageExtension.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor()
														.getName() }));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private void processPhysicalPackages(Collection elements) {
		for (Iterator it = elements.iterator(); it.hasNext();) {
			processPhysicalPackage((IConfigurationElement) it.next());
		}
	}

	private void processPhysicalPackage(IConfigurationElement element) {
		try {
			PhysicalPackageExtension ppp = new PhysicalPackageExtension(element);
			if (mIdToPhysicalExtension.containsKey(ppp.getID())) {
				throw new PackageExtensionException(
						DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.exception.duplicatePhysicalPackageId")); //$NON-NLS-1$
			}
			mIdToPhysicalExtension.put(ppp.getID(), ppp);

			List<PhysicalPackageExtension> pkgExts = null;
			if (mPhysicalExtensionByFileExtension.containsKey(ppp
					.getFileExtension())) {
				pkgExts = (List) mPhysicalExtensionByFileExtension.get(ppp
						.getFileExtension());
				pkgExts.add(ppp);
			} else {
				pkgExts = new ArrayList<PhysicalPackageExtension>();
				pkgExts.add(ppp);
				mPhysicalExtensionByFileExtension.put(ppp.getFileExtension(),
						pkgExts);
			}

		} catch (PackageExtensionException e) {
			if (DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"DeploymentExtensionManager.trace.error.physicalPackageParseError", //$NON-NLS-1$
										new Object[] {
												element
														.getAttribute(PackageExtension.ATTR_ID),
												element.getDeclaringExtension()
														.getContributor()
														.getName() }));
				e.printStackTrace();
				System.err.flush();
			}
		}
	}

	private static class IDComparator implements Comparator {

		public int compare(Object o1, Object o2) {
			String s1;
			String s2;
			if (o1 instanceof String) {
				s1 = (String) o1;
			} else {
				s1 = ((IPackageExtension) o1).getID();
			}
			if (o2 instanceof String) {
				s2 = (String) o2;
			} else {
				s2 = ((IPackageExtension) o2).getID();
			}
			return s1.compareTo(s2);
		}
	}

	private static class FileExtensionComparator implements Comparator {

		public int compare(Object o1, Object o2) {
			int retVal;
			String s1;
			String s2;
			if (o1 instanceof String) {
				s1 = (String) o1;
				if (o2 instanceof String) {
					s2 = (String) o2;
				} else {
					s2 = ((IPackageExtension) o2).getFileExtension();
				}
				retVal = s1.compareTo(s2);
			} else if (o2 instanceof String) {
				s2 = (String) o2;
				if (o1 instanceof String) {
					s1 = (String) o1;
				} else {
					s1 = ((IPackageExtension) o1).getFileExtension();
				}
				retVal = s1.compareTo(s2);
			} else {
				IPackageExtension ipe1 = (IPackageExtension) o1;
				IPackageExtension ipe2 = (IPackageExtension) o2;
				retVal = ipe1.getFileExtension().compareTo(
						ipe2.getFileExtension());
				if (retVal == 0) {
					retVal = ipe1.getID().compareTo(ipe2.getID());
				}
			}
			return retVal;
		}
	}

}