package org.eclipse.stp.soas.internal.deploy.core;

import java.util.Comparator;
import java.util.Iterator;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.jface.util.Assert;
import org.eclipse.stp.soas.deploy.core.DeployCorePlugin;
import org.eclipse.stp.soas.deploy.core.DeploymentExtensionManager;
import org.eclipse.stp.soas.deploy.core.IRuntimeDefinition;
import org.eclipse.stp.soas.deploy.core.ISupportedTechnologyType;
import org.eclipse.stp.soas.deploy.core.ITechnologyDefinition;
import org.eclipse.stp.soas.deploy.core.ITechnologyMap;
import org.eclipse.stp.soas.deploy.core.IVersion;

public class RuntimeDefinition implements IRuntimeDefinition, Comparable {

	public static final String ATTR_ID = "id"; //$NON-NLS-1$

	private IConfigurationElement mElement;
	private String mID;
	private SortedMap mSupportingRuntimeVersionssByTechnologyDef;
	private SortedMap mTechnologyMapsByRuntimeVersion;

	public RuntimeDefinition(IConfigurationElement element)
			throws PackageExtensionException {
		super();
		init(element);
		mSupportingRuntimeVersionssByTechnologyDef = new TreeMap();
		mTechnologyMapsByRuntimeVersion = new TreeMap(
				new DecendingVersionOrder());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.deploy.core.IRuntimeDefinition#getID()
	 */
	public String getID() {
		return mID;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.deploy.core.IRuntimeDefinition#getTechnologyMaps()
	 */
	public ITechnologyMap[] getTechnologyMaps() {
		return (ITechnologyMap[]) mTechnologyMapsByRuntimeVersion.values()
				.toArray(new ITechnologyMap[0]);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.stp.soas.deploy.core.IRuntimeDefinition#getTechnologyMap(org.eclipse.stp.soas.deploy.core.IVersion)
	 */
	public ITechnologyMap getTechnologyMap(IVersion runtimeVersion) {
		ITechnologyMap techMap;
		Iterator it = mTechnologyMapsByRuntimeVersion.tailMap(runtimeVersion)
				.values().iterator();
		if (it.hasNext()) {
			techMap = (ITechnologyMap) it.next();
			if (!techMap.getRuntimeVersion().isCompatibleWithVersion(
					runtimeVersion, techMap.getVersionMatchRule())) {
				techMap = null;
			}
		}
		else {
			techMap = null;
		}
		return techMap;
	}

	public IVersion[] getSupportingRuntimeVersions(ITechnologyDefinition techDef) {
		return (IVersion[]) getSupportingRuntimeVersionsSet(techDef).toArray(
				new IVersion[0]);
	}

	/* package */void addTechnologyMap(ITechnologyMap itm)
			throws PackageExtensionException {
		Assert.isTrue(itm.getRuntimeDefinition().equals(this));
		if (mTechnologyMapsByRuntimeVersion.containsKey(itm.getRuntimeVersion())) {
			if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
				System.err
						.println(DeployCorePlugin
								.getDefault()
								.getResourceString(
										"RuntimeDefinition.trace.error.duplicateTechnologyMap", //$NON-NLS-1$
										new Object[] { mID,
												itm.getRuntimeVersion()}));
				System.err.flush();
			}
			throw new PackageExtensionException(
					DeployCorePlugin
							.getDefault()
							.getResourceString(
									"RuntimeDefinition.exception.duplicateTechnologyMap")); //$NON-NLS-1$
		}
		mTechnologyMapsByRuntimeVersion.put(itm.getRuntimeVersion(), itm);
		addTechnologiesToList(itm);
	}

	private void addTechnologiesToList(ITechnologyMap itm) {
		ISupportedTechnologyType[] techTypes = itm
				.getSupportedTechnologyTypes();
		for (int index = 0, count = techTypes.length; index < count; ++index) {
			ITechnologyDefinition techDef = techTypes[index]
					.getTechnologyDefinition();
			SortedSet runtimeVersions = getSupportingRuntimeVersionsSet(techDef);
			runtimeVersions.add(itm.getRuntimeVersion());
		}
	}

	private SortedSet getSupportingRuntimeVersionsSet(
			ITechnologyDefinition techDef) {
		SortedSet runtimeVersions = (SortedSet) mSupportingRuntimeVersionssByTechnologyDef
				.get(techDef);
		if (runtimeVersions == null) {
			runtimeVersions = new TreeSet(new DecendingVersionOrder());
			mSupportingRuntimeVersionssByTechnologyDef.put(techDef,
					runtimeVersions);
		}
		return runtimeVersions;
	}

	private void init(IConfigurationElement element)
			throws PackageExtensionException {
		//comment out the name check below. since we are getting server def from wtp as well
//		Assert.isTrue(DeploymentExtensionManager.EXT_ELEM_SERVER_DEFINITION
//				.equals(element.getName()));

		mElement = element;

		mID = element.getAttribute(ATTR_ID);

		if (DeploymentExtensionManager.DEBUG_DEPLOYMENT_EXTENSION) {
			System.out
					.println(DeployCorePlugin
							.getDefault()
							.getResourceString(
									"ServerDefinition.trace.error.serverDefinitionParseError", //$NON-NLS-1$
									new Object[] { getID()}));
			System.out.flush();
		}

		if (mID == null) {
			throw new PackageExtensionException(DeployCorePlugin
					.getDefault().getResourceString(
							"ServerDefinition.exception.idNotSpecified")); //$NON-NLS-1$
		}
	}

	public int compareTo(Object o) {
		return getID().compareTo(((RuntimeDefinition) o).getID());
	}

	public boolean equals(Object obj) {
		boolean retVal;
		if (this == obj) {
			retVal = true;
		}
		else if (obj instanceof RuntimeDefinition) {
			retVal = getID().equals(((RuntimeDefinition) obj).getID());
		}
		else {
			retVal = false;
		}
		return retVal;
	}

	public int hashCode() {
		return mID.hashCode();
	}

	private static class DecendingVersionOrder implements Comparator {

		public int compare(Object o1, Object o2) {
			return -(((Comparable) o1).compareTo(o2));
		}
	}
}
