/**********************************************************************
 * Copyright (c) 2006 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.xml.internal;

//import com.aem.utils.*;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;

public class FastXMLParser {
	private final static boolean DEBUG = false;
	private final static boolean ALWAYS_BUFFERED = false;

	InputStream input;
	
	private final static byte ROOT = 0;
	private final static byte TEXT_NODE = 1;
	private final static byte OPEN_TAG = 2;
	private final static byte TAG_NAME = 3;
	private final static byte TAG_ATTRIBUTE = 4;
	private final static byte CLOSE_TAG = 5;
	private final static byte ID_OR_ATTRIBUTE = 6;
	
	int pch = 0;
	int ch = 0;
	int line = 1;
	int column = 1;
	
	public FastXMLParser() {
	}

	public FastXMLParser(String in) throws IOException {
		setInput(in);
	}

	public FastXMLParser(StringBuffer in) throws IOException {
		setInput(in);
	}

	public FastXMLParser(InputStream in) throws IOException {
		setInput(in);
	}

	public void setInput(InputStream in) throws IOException {
		if (ALWAYS_BUFFERED) {
			input = new BufferedInputStream(in);
		} else {
			input = in;	
		}
		ch = input.read();
		line = 1;
		column = 1;
	}	

	public void setInput(String in) throws IOException {
		setInput(new FastByteArrayInputStream(in.getBytes()));
	}	

	public void setInput(StringBuffer in) throws IOException {
		setInput(new FastByteArrayInputStream(in.toString().getBytes()));
	}	

	XMLParserListener curlistener;
	
	public void parse(XMLParserListener listener, InputStream in) throws Exception {
		setInput(in);
		parse(listener);
	}
		
	public void parse(XMLParserListener listener) throws Exception {
		this.curlistener = listener;
		
		listener.startDocument();
		
		boolean ignored_tag = false;
		boolean start_tag = true;
		String tag_name = "ERROR";

		String key;
		String value;

		StringBuffer tmp = new StringBuffer(128);
//		HashMap attributes = new HashMap();
		HashMap attributes = null;

		byte mode = TEXT_NODE;

		int startLine = -1;
		
		while (true) {
if (DEBUG) System.out.println("ch = '"+(char)peekChar()+"'");		
			switch(mode) {
				
				//
				// Text between tags
				// <tag>[ blah ]</tag>
				//
				case TEXT_NODE:
					tmp.setLength(0);
					
					startLine = line;
					
					parseText(tmp);
					
if (DEBUG) System.out.println("TEXT NODE:'"+tmp+"'");					
					if (peekChar() == -1) {
						listener.endDocument();
						return;
					} else {
						if (tmp.length() > 0) {
							convertValue(tmp);
							listener.text(tmp.toString(),startLine,line);
						}
						mode = OPEN_TAG;
					}
					break;
				
				//
				// Tag open bracket and others
				// [<]tag ...			//start tag
				// [<!-- ... -->]		//comment
				// [<!]DOCTYPE ...		//DOCTYPE thing
				// [</]tag ...			//end tag			
				// [<?]tag ...			//xml spec thing
				//
				case OPEN_TAG:
					if (nextChar() != '<') {
						error("expected open tag < not "+prevChar());
					}
					
					//parse a comment, then return to textnode mode
					if (peekChar() == '!') {
						nextChar();
						
						if (peekChar() == '-') {
							if (nextChar() != '-') {
								error("expected comment -");
							}
							if (peekChar() != '-') {
//								error("expected comment --");
//								mode = TEXT_NODE;
							} else {
								nextChar();
								
								startLine = line;

								StringBuffer zz = new StringBuffer();
								parseComment(zz);
								listener.comment(zz.toString(),startLine,line);
//								System.out.println("PARSED COMMENT ["+zz+"]");
//								parseComment(null);

								mode = TEXT_NODE;
							}

						} else if (peekChar() == '[') {
							nextChar();
							if (nextChar() != 'C') {
								error("expected ![CDATA[ comment C");
							}
							if (nextChar() != 'D') {
								error("expected ![CDATA[ comment D");
							}
							if (nextChar() != 'A') {
								error("expected ![CDATA[ comment A");
							}
							if (nextChar() != 'T') {
								error("expected ![CDATA[ comment T");
							}
							if (nextChar() != 'A') {
								error("expected ![CDATA[ comment A");
							}
							if (nextChar() != '[') {
								error("expected ![CDATA[ comment [");
							}
							parseCDATA(null);
							mode = TEXT_NODE;
						
						} else {
							start_tag = true;
							ignored_tag = true;	
							mode = ID_OR_ATTRIBUTE;
							parseWhitespace(null);
							
						}
						
					} else {
					
						if (peekChar() == '?') {
							nextChar();
							start_tag = true; //dont allow />
							ignored_tag = true;
						} else if (peekChar() == '/') {
							nextChar();
if (DEBUG) System.out.println("OPEN END TAG:'"+tmp+"'");
							start_tag = false;
						} else {
if (DEBUG) System.out.println("OPEN START TAG:'"+tmp+"'");					
//							attributes = new HashMap();
							attributes = null;
							start_tag = true;
						}

						ignored_tag = false;	
						mode = TAG_NAME;
						parseWhitespace(null);
					
					}
					break;
				
				//
				// Tag name, <[tag] blah="blah">
				//
				case TAG_NAME:

					startLine = line;

					tmp.setLength(0);
					parseIdentifier(tmp);
					
					if (tmp.length() == 0) {
						error("expected tag name (or some other valid open tag like \"<!--\", \"<!DOCTYP\" or \"<![CDATA[\" )");	
					}
					
					tag_name = tmp.toString();
if (DEBUG) System.out.println("TAG NAME:'"+tmp+"'");					
					parseWhitespace(null);
					mode = TAG_ATTRIBUTE;
					break;

				//
				// basically an identifier or an attribute
				// <![DOCTYPE] ...
				// <!DOCTYPE [SYSTEM]
				// <!DOCTYPE SYSTEM ["C:\mydtd.dtd"]
				//				
				case ID_OR_ATTRIBUTE:
					
					if (peekChar() == '/') {
						mode = CLOSE_TAG;
					} else if (peekChar() == '?') {
						mode = CLOSE_TAG;
					} else if (peekChar() == '>') {
						mode = CLOSE_TAG;
					} else {
						
if (DEBUG) System.out.println("ID OR ATTRIBUTE:");					

						if (peekChar() == '"' || peekChar() == '\'') {

							parseQuoted(null);
							
						} else {
						
							parseIdentifier(null);
	
							parseWhitespace(null);
	
							if (peekChar() == '=') {
								nextChar();
	
								parseWhitespace(null);
	
								parseQuoted(null);
	
							}

						}

						parseWhitespace(null);
					}
					
					break;
					
				//
				// Tag attribute
				// <mytag [key=value] ...
				//
				case TAG_ATTRIBUTE:
				
					if (peekChar() == '/') {
						mode = CLOSE_TAG;
					} else if (peekChar() == '?') {
						mode = CLOSE_TAG;
					} else if (peekChar() == '>') {
						mode = CLOSE_TAG;
					} else {
						
						if (!start_tag) {
							error("attributes found in end tag");	
						}
						
						//[key]=value
						tmp.setLength(0);
						parseIdentifier(tmp);
						key = tmp.toString();
						
						if (key.length() == 0) {
							error("expected tag attribute key (blah=\"...\")");
						}
						
						parseWhitespace(null);
						
						//key[=]value
						if (nextChar() != '=') {
							error("expected attribute equals = (blah=\"...\")");
						}

						parseWhitespace(null);
						
						if (peekChar() != '"'
							&& peekChar() != '\'') {
							error("expected attribute value in quotes (blah=\"value\")");		
						}
						
						//key=[value]
						tmp.setLength(0);
						parseQuoted(tmp);
						convertValue(tmp);
						value = tmp.toString();

						if (attributes == null) attributes = new HashMap(16,0.6f);
						attributes.put(key,value);

if (DEBUG) System.out.println("TAG ATTRIBUTE:'"+key+"'='"+value+"'");					
						
						parseWhitespace(null);
					}		
					break;
				
				//
				// Any kind of close tag
				// ... [?>]		//end spec tag
				// ... [/>]		//end start tag (generate start and end tags)
				// ... [>]		//end tag (generate start or end tag)
				//
				case CLOSE_TAG: 

					if (peekChar() == '?') {
						//do nothing, its the end of a spec tag 
						nextChar();
						if (nextChar() != '>') {
							error("expected close tag \"?>\" at end of xml spec tag");	
						}
						
					} else if (peekChar() == '/') {
						if (!start_tag) {
							error("invalid close tag \"/>\" on end tag, close tag should be \">\" on end tag");	
						}
						nextChar();

						if (nextChar() != '>') {
							error("expected close tag > (e.g. \"<mytag />\")");	
						}

						if (!ignored_tag) {
							listener.startElement(tag_name,attributes,startLine);	
							listener.endElement(tag_name,line);
						}
						ignored_tag = false;

					} else {

						if (nextChar() != '>') {
							error("expected close tag >");	
						}

						if (!ignored_tag) {
							if (start_tag) {
								listener.startElement(tag_name,attributes,startLine);	
							} else {
								listener.endElement(tag_name,line);
							}
						}
						ignored_tag = false;

					}
					
if (DEBUG) System.out.println("CLOSE TAG:'"+tmp+"'");					
					mode = TEXT_NODE;
					break;				
					
			}//end switch
		}//end while
	
	}

	private int peekChar() throws IOException {
		return ch;
	}
	
	private int nextChar() throws IOException {
//		int nch = ch;
		pch = ch;
		ch = input.read();

		if (pch == '\n') {
			line++;
			column = 1;
		} else {
			column++;
		}
//		pch = fixChar(nch);

		return pch;
	}
	
	private char prevChar() throws IOException {
		return (char)pch;	
	}
/*	
	private int fixChar(int ch) throws IOException {
		if (ch == '\r') {
			if (peekChar() == '\n') {
				ch = nextChar();	
			} else {
				ch = '\n'; 
			}
		}
		
		if (ch == '\n') {
			line++;
			column = 0;
		} else {
			column++;
		}
		
		return ch;
	}
*/	
	private void error(String s) throws Exception {
		throw new Exception(s+", line "+line+", column "+column+" pch="+prevChar()+" ("+((int)prevChar())+") peek="+(char)peekChar()+" ("+peekChar()+")");
	}

	private int indexOf(StringBuffer sb, char c, int index) {
		for (int i = index; i < sb.length(); i++) {
			char x = sb.charAt(i);
			if (c == x) return i; 			
		}
		return -1;
	}

	private void convertValue(StringBuffer sb) throws IOException {
		
		char ch;
		for (int i = 0; i < sb.length(); i++) {
			ch = sb.charAt(i);
			
			if (ch == '&') {
				
				int len = sb.length() - i;
				
				boolean matched = false;
				
				if (len > 3) {
					if (sb.charAt(i+1) == '#') {
						
//						int end_parse = sb.indexOf(';',i+1);
						int end_parse = indexOf(sb,';',i+1);
						if (end_parse != -1) {
							char ccode = (char)Integer.parseInt(sb.substring(i+2,end_parse));
							sb.replace(i,end_parse+1,String.valueOf(ccode));
						}
						matched = true;
					
					} else if (sb.charAt(i+1) == 'l' 
								&& sb.charAt(i+2) == 't'
								&& sb.charAt(i+3) == ';') {
						
						sb.replace(i,i+4,"<");
						matched = true;
	
					} else if (	sb.charAt(i+1) == 'g' 
								&& sb.charAt(i+2) == 't'
								&& sb.charAt(i+3) == ';') {
	
						sb.replace(i,i+4,">");
						matched = true;
					}
				} 
				
				if (!matched && len > 4) {
					if (	sb.charAt(i+1) == 'a' 
								&& sb.charAt(i+2) == 'm'
								&& sb.charAt(i+3) == 'p'
								&& sb.charAt(i+4) == ';') {
		
						sb.replace(i,i+5,"&");
						matched = true;
					}
				}

				if (!matched && len > 5) {
					if (	sb.charAt(i+1) == 'q' 
								&& sb.charAt(i+2) == 'u'
								&& sb.charAt(i+3) == 'o'
								&& sb.charAt(i+4) == 't'
								&& sb.charAt(i+5) == ';') {
	
						sb.replace(i,i+6,"\"");
	
					} else if (	sb.charAt(i+1) == 'a' 
								&& sb.charAt(i+2) == 'p'
								&& sb.charAt(i+3) == 'o'
								&& sb.charAt(i+4) == 's'
								&& sb.charAt(i+5) == ';') {
	
						sb.replace(i,i+6,"\'");
	
					}
				}
			}
		}
	}

	private int parseCDATA(StringBuffer sb) throws IOException {
		int len = 0;
		int ch = peekChar();

		int count = 0;		

		while (	(ch > 31 && ch < 127)
				|| (ch > 127 && ch < 255)
				|| ch == '\t'
				|| ch == '\n'
				) {

			if (count == 0 && ch == ']') {
				count++;	
			} else if (count == 1 && ch == ']') {
				count++;	
			} else if (count == 2 && ch == '>') {
				nextChar();
				if (sb != null) sb.setLength(sb.length()-2);
				return len-2;	
			} else {
				count = 0;
			}
			
			if (sb != null) sb.append((char)ch);
			len++;
			nextChar();
			ch = peekChar();		
		}
		
		return len;

	}
	
	private int parseComment(StringBuffer sb) throws IOException {
		int len = 0;
		int ch = peekChar();

		int count = 0;		
		
		while (ch != -1
//				(ch > 31 && ch < 127)
//				|| (ch > 127 && ch < 255)
//				|| ch == '\t'
//				|| ch == '\n'
				) {
			
			if (count == 0 && ch == '-') {
				count++;
			} else if (count == 1 && ch == '-') {
				count++;
			} else if (count == 2 && ch == '>') {
				nextChar();
				if (sb != null) sb.setLength(sb.length()-2);
				return len-2;
			} else {
				count = 0;
			}
			
			if (sb != null) sb.append((char)ch);
			len++;
			nextChar();
			ch = peekChar();
		}
				
		return len;
	}
	
	private int parseIdentifier(StringBuffer sb) throws IOException {
		int len = 0;
		int ch = peekChar();
		
		while (	   (ch > 43 && ch < 59)  // 0-9 + :,+-
				|| (ch > 64 && ch < 91)  // A-Z
				|| (ch > 96 && ch < 123) // a-z
				|| (ch == 95)			 // _
				) {
			if (sb != null) sb.append((char)ch);
			len++;	
			nextChar();
			ch = peekChar();			
		}
		
		return len;
	}

	private int parseQuoted(StringBuffer sb) throws IOException {
		int len = 0;
		int quote = nextChar();
		int ch = peekChar();
		
		while ((ch > 31 && ch < 127)
				|| (ch > 127 && ch < 255)
				) {
			
			if (ch == quote) {
				nextChar();
				return len;
			}
			
			if (sb != null) sb.append((char)ch);
			len++;
			nextChar();
			ch = peekChar();			
		}
		
		return len;
	}

	private int parseText(StringBuffer sb) throws IOException {
		int len = 0;
		int ch = peekChar();
		
/*		while (	(ch > 31 && ch < 127)
				|| (ch > 127 && ch < 255)
				|| ch == '\t'
				|| ch == '\n'
				|| ch == '\r'
				) {*/
		while (ch != -1) {
			
			if (ch == '<') return len;
			
			if (sb != null) sb.append((char)ch);
			len++;
			nextChar();
			ch = peekChar();			
		}
		
		return len;
	}

	private int parseWhitespace(StringBuffer sb) throws IOException {
		int len = 0;
		int ch = peekChar();
		
		while (	ch == ' '
				|| ch == '\t'
				|| ch == '\n'
				|| ch == '\r'
			  ) {
			
			//skip the character
			if (sb != null) sb.append((char)ch);
			len++;
			nextChar();
			ch = peekChar();
		}
		
		return len;
	}	

}
