/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.ui.xef.schema;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.XMLConstants;

import org.apache.xerces.dom.DocumentImpl;
import org.apache.xerces.xs.XSAnnotation;
import org.eclipse.stp.xef.XefConstants;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public abstract class AbstractAnnotatedElement implements AnnotatedElement {
    String displayName;
    String docShort;
    String documentation;
    String category;
    Map<String, Boolean> qualifiers = new HashMap<String, Boolean>();
    List<List<String>> requires = new ArrayList<List<String>>();

    void handleAnnotations(XSAnnotation annotation) {
        if (annotation == null) {
            return;
        }

        documentation = getDocumentation(annotation);
                
        Document domDoc = new DocumentImpl();
        annotation.writeAnnotation(domDoc, XSAnnotation.W3C_DOM_DOCUMENT); // parse the annotation
        NodeList catNodes = domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "category");
        category = getTextValue(catNodes);
        
        NodeList displayNameNodes = domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "displayName");
        displayName = getI18nTextValue(displayNameNodes);
        
        NodeList shortDocNodes = domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "docShort");
        docShort = getI18nTextValue(shortDocNodes);
        
        NodeList qualifierNodes = domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "qualifier");
        for (int i=0; i < qualifierNodes.getLength(); i++) {
            Node n = qualifierNodes.item(i);
            String val = getTextValue(n);                        
            boolean allowMultiple = "true".equalsIgnoreCase(getAttributeValue(n, null, "multiple"));
            qualifiers.put(val, allowMultiple);
        }
        
        NodeList requiresNodes = domDoc.getElementsByTagNameNS(XefConstants.XEF_NS, "requires");
        for (int i=0; i < requiresNodes.getLength(); i++) {
            List<String> conjunction = new ArrayList<String>();
            Node n = requiresNodes.item(i);
            NodeList children = n.getChildNodes();
            for (int j=0; j < children.getLength(); j++) {
                Node c = children.item(j);
                if (XefConstants.XEF_NS.equals(c.getNamespaceURI()) &&
                    "element_qname".equals(c.getLocalName())) {
                    conjunction.add(getTextValue(c));
                }
            }
            if (conjunction.size() > 0) {
                requires.add(conjunction);
            }
        }
    }
    
    // The getDocumentation and getLanguageDocumentation methods below are duplicated in 
    // org.eclipse.stp.soa.configgen.tools.GenerateModel. It would be good if they could be shared
    // somehow, but beware that the org.eclipse.stp.ui.xef plugin shouldn't really have deep 
    // dependencies.
    static String getDocumentation(XSAnnotation annotation) {
        if (annotation == null) {
            return null;
        }
        String annotationText = annotation.getAnnotationString();
        if (annotationText == null) {
            return null;
        }

        // Try ISO 639-2 first, the 3-letter version
        String docString = getLanguageDocumentation(annotationText, Locale.getDefault().getISO3Language());
        if (docString == null) {
            // now try ISO 639-1, the 2-letter one
            docString = getLanguageDocumentation(annotationText, Locale.getDefault().getLanguage());
        } 
        if (docString == null) {        
            docString = getDefaultDocumentation(annotationText);
        }
        return docString;
    }

    static String getDefaultDocumentation(String annotationText) {
        String x = ".*<[^/][a-z]+?:documentation\\s*>(.*?)</[a-z]+?:documentation\\s*>.*";
        // Use ordinary regexps here, as we want to get the doc HTML as-is.
        return getPatternMatch(annotationText, x, 1);
    }

    static String getLanguageDocumentation(String annotationText, String langCode) {
        String x = ".*<[^/][a-z]+?:documentation\\s+xml:lang\\s*=\\s*[\"']" + langCode + "[\"']\\s*>(.*?)</[a-z]+?:documentation\\s*>.*";
        // Use ordinary regexps here, as we want to get the doc HTML as-is.
        return getPatternMatch(annotationText, x, 1);
    }

    static String getPatternMatch(String text, String pattern, int refNr) {
        Pattern p = Pattern.compile(pattern, Pattern.DOTALL);
        Matcher m = p.matcher(text);
        if (m.matches()) { 
            return m.replaceFirst("$" + refNr);
        } else {
            return null;
        }
    }    
    
    static String getAttributeValue(Node node, String ns_uri, String attr) {
        Node attrNode = node.getAttributes().getNamedItemNS(ns_uri, attr);
        if (attrNode != null) { 
            return attrNode.getTextContent();
        }
        return null;
    }

    static String getTextValue(NodeList nodes) {
        if (nodes.getLength() > 0) {
            return getTextValue(nodes.item(0));
        }
        return null;
    }
    
    static String getTextValue(Node node) {
        for (Node cur = node.getFirstChild(); cur != null; cur = cur.getNextSibling()) {
            if (cur.getNodeType() == Node.TEXT_NODE) {
                String text = cur.getTextContent().trim();
                if (!"".equals(text)) {
                    return text;
                }
            }
        }
        return null;
    }

    static String getI18nTextValue(NodeList nodes) {
        // Try the ISO 639-2 three letter version first
        Node node = getI18nNode(nodes, Locale.getDefault().getISO3Language());
        if (node == null) {
            // Try the ISO 639-1 two letter version
            node = getI18nNode(nodes, Locale.getDefault().getLanguage());
        }
        if (node == null) {
            // go for the default
            node = getI18nNode(nodes, null);
        }
        if (node == null) {
            return null;
        }
        return getTextValue(node);
    }        
    
    static Node getI18nNode(NodeList nodes, String language) {
        for (int i=0; i < nodes.getLength(); i++) {
            Node node = nodes.item(i);
            String lang = getAttributeValue(node, XMLConstants.XML_NS_URI, "lang");
            if (language == null && lang == null) {
                return node; // default language
            }
            if (language != null && language.equalsIgnoreCase(lang)) {
                return node;
            }
        }
        return null;
    }
    
    public String getCategory() {
        return category;
    }

    public String getDisplayName() {
        return displayName;
    }

    public String getDocShort() {
        return docShort;
    }
    
    public String getDocumentation() {
        return documentation;
    }       
}
