/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.xef;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;

/** This IXMLProvider implementation is a wrapper around any number of IXMLProvider instances which can expose the set
 * of providers as one provider. It can be used in cases where multiple objects need to be manipulated at the same 
 * time (e.g. adding a policy to multiple services in one go).
 */
public class MultiXMLProvider implements IXMLProvider {       
    private final List<IXMLProvider> providers;
    private final SAXBuilder builder = new SAXBuilder();
    private final Map<IXMLProvider, String> originalXMLs = new HashMap<IXMLProvider, String>();
    private String selection;
    
    public MultiXMLProvider(List<IXMLProvider> ps) {
        providers = ps;
        if (ps == null) {
            throw new NullPointerException();
        }
        if (ps.size() == 0) {
            throw new IllegalArgumentException("Must have at least one provider");
        }
    }

    public String getRootElementQName() {
        return providers.get(0).getRootElementQName();
    }

    /**
     * Obtain the XML. For this multi provider this means, look at every single IXMLProvider that is wrapped
     * and present the intersection of the XML exposed by all. 
     * @return The XML.
     */
    @SuppressWarnings("unchecked")
    public synchronized String getXML() {
        originalXMLs.clear();
        
        try {
			Element rootElement = null;
			List<String> commonElements = new LinkedList<String>();
			boolean first = true;
			for (IXMLProvider provider : providers) {
			    String orgXML = provider.getXML();
			    originalXMLs.put(provider, orgXML);
			    Document d = builder.build(new ByteArrayInputStream(orgXML.getBytes()));
			    if (first) { // Fill the list with initial stuff, some of these might later be removed
				Element root = d.getRootElement();
				rootElement = new Element(root.getName(), root.getNamespace());
				for (Element el : (List<Element>) root.getChildren()) {
					commonElements.add(XMLUtil.canonicalize(el));
				}
				first = false; 
			    } else {
				Element root = d.getRootElement();
				if (!rootElement.getName().equals(root.getName()) ||
				    !rootElement.getNamespace().getURI().equals(root.getNamespace().getURI())) {
				    // Different root elements - no intersection!
				    return "";
				}
				
				List<String> copy = new ArrayList<String>(commonElements);
				for (Element el : (List<Element>) root.getChildren()) {
					copy.remove(XMLUtil.canonicalize(el));
				}
				for (String el : copy) {
					commonElements.remove(el);
				}                    
			    }
			}
			
			StringBuilder sb = new StringBuilder();
			String canonicalRoot = XMLUtil.canonicalize(rootElement);
			int idx = canonicalRoot.indexOf("</");
			if (idx < 0) {
				return null;
			}
			sb.append(canonicalRoot.substring(0, idx));
			for (String el : commonElements) {
				sb.append(el);
			}
			sb.append(canonicalRoot.substring(idx));
			selection = sb.toString(); // store the selection locally so we can pass it back when setXML is called
			return selection;
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
    }

	/**
     * Set the modified XML in every IXMLProvider instance. It will only affect the elements that where selected
     * as part of the intersection. Other elements are left alone.<P/>
     * 
     * Note: this call assumes that getXML() has been called before.
     * @param xml The modified XML.
     */
    public void setXML(String xml) {
        for (IXMLProvider provider : providers) {
            try {
                provider.setXML(XMLUtil.mergeXMLBack(originalXMLs.get(provider), selection, xml));
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }
}
