/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.xef;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import javax.xml.namespace.QName;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.stp.ui.xef.XefPlugin;
import org.eclipse.stp.ui.xef.editor.SchemaSelectionDialog;
import org.eclipse.stp.ui.xef.schema.SchemaElement;
import org.eclipse.stp.ui.xef.schema.SchemaRegistry;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;


public class SchemaProviderFilterWrapper implements ISchemaProvider {    
    private Set<String> allowedSchemas = null;
    private Set<String> allowedSnippets = null;
    private HashMap<String, DisallowedInfo> disallowedSchemas = null;
    private HashMap<String, DisallowedInfo> disallowedSnippets = null;

    private final ISchemaProvider delegate;
    private final List<String> alreadyApplied;
    private final IShadowProvider shadowProvider;
    
    public SchemaProviderFilterWrapper(ISchemaProvider schemaProvider, List<XMLInstanceElement> filteredElements) {
        delegate = schemaProvider;
        
        alreadyApplied = new ArrayList<String>(filteredElements.size());
        for (XMLInstanceElement el : filteredElements) {
            alreadyApplied.add(el.getJDOMElement().getNamespaceURI());
        }
        
        shadowProvider = new ShadowProvider();
    }
    
    public IShadowProvider getShadowProvider() {
        return shadowProvider;
    }

    public String getSchema(String namespace) {
        initializeSchemas();

        if (!allowedSchemas.contains(namespace)) {
            return null;
        }
        
        return delegate.getSchema(namespace);
    }

    public String getSnippet(String name) {
        initializeSnippets();
        
        if (allowedSnippets.contains(name)) {
            return delegate.getSnippet(name);
        } else {
            return null;
        }
    }

    public Collection<String> listSchemaNamespaces(String filter) {
        initializeSchemas();
        
        return allowedSchemas;
    }

    public Collection<String> listSnippets(String filter) {
        initializeSnippets();     
        
        return allowedSnippets;
    }
    
    private synchronized void initializeSchemas() {
        if (allowedSchemas != null) {
            return;
        }
        
        allowedSchemas = new HashSet<String>();
        disallowedSchemas = new HashMap<String, DisallowedInfo>();
        SchemaRegistry sr = XefPlugin.getDefault().getSchemaRegistry();
        
        Map<String, List<String>> disallowedQualifiers = new HashMap<String, List<String>>();
        
        List<String> l = new ArrayList<String>(delegate.listSchemaNamespaces(null));
        for (String appl : alreadyApplied) {
            l.remove(appl);
            disallowedSchemas.put(appl, DisallowedInfo.alreadyApplied(appl));
            
            for (SchemaElement el : sr.getEntryElements(appl, true, true, delegate)) {
                for (Map.Entry<String, Boolean> entry : el.getQualifiers().entrySet()) {
                    if (!entry.getValue()) {
                        List<String> reasons = disallowedQualifiers.get(entry.getKey());
                        if (reasons == null) {
                            reasons = new ArrayList<String>();
                            disallowedQualifiers.put(entry.getKey(), reasons);
                        }
                        reasons.add(appl);
                    }
                }
            }            
        }
        
        // Look at the remaining schemas to see if any of them are not allowed because
        // of existing schemas with the same qualifier (that doesn't allow multiple instances)
        nextNamespace: 
        for (Iterator<String> it = l.iterator(); it.hasNext(); ) {            
            String s = it.next();
            for (SchemaElement el : sr.getEntryElements(s, true, true, delegate)) {
                for (String qualifier : el.getQualifiers().keySet()) {
                    if (disallowedQualifiers.containsKey(qualifier)) {
                        it.remove();
                        disallowedSchemas.put(s, DisallowedInfo.alreadyApplied(
                            disallowedQualifiers.get(qualifier)));
                        continue nextNamespace;
                    }
                }   
                
                if (el.getRequires().size() != 0) {
                    DisallowedInfo info = requirementsSatisfied(el, alreadyApplied);
                    if (info != null) {
                        it.remove();
                        disallowedSchemas.put(s, info);
                        continue nextNamespace;                    
                    }
                }
            }
        }
        allowedSchemas.addAll(l);
    }

    private static DisallowedInfo requirementsSatisfied(SchemaElement el, Collection<String> applied) {        
        List<List<String>> requirements = el.getRequires();
        if (requirements.size() == 0) {
            return null;
        }
        
        for (List<String> required : requirements) {
            if(requirementSatisfied(required, applied)) {
                return null;
            }
        }
        Collection<String> causedBy = flatten(requirements);
        causedBy.removeAll(applied);                
        return DisallowedInfo.required(causedBy);
    }

    private static boolean requirementSatisfied(List<String> required, Collection<String> applied) {
        for (String r : required) {
            QName qn = QName.valueOf(r);
            if (!applied.contains(qn.getNamespaceURI())) {
                return false;
            }
        }
        return true;
    }

    private static Collection<String> flatten(List<List<String>> requires) {
        List<String> result = new ArrayList<String>();
        for (List<String> l : requires) {
            for (String s : l) {
                QName qn = QName.valueOf(s);
                result.add(qn.getNamespaceURI());
            }
        }
        return result;
    }

    private synchronized void initializeSnippets() {
        if (allowedSnippets != null) {
            return;
        }
        initializeSchemas();
        
        allowedSnippets = new HashSet<String>();
        disallowedSnippets = new HashMap<String, DisallowedInfo>();
        SchemaRegistry sr = XefPlugin.getDefault().getSchemaRegistry();
        SAXBuilder builder = new SAXBuilder();
        
        for (String name : delegate.listSnippets(null)) {
            String snippet = delegate.getSnippet(name);
            String xml = "<root>" + snippet + "</root>";
            try {
                List<DisallowedInfo> disallowReasons = new ArrayList<DisallowedInfo>();
                Document doc = builder.build(new ByteArrayInputStream(xml.getBytes()));
                
                for (Object child : doc.getRootElement().getChildren()) {
                    if (child instanceof Element) {
                        Element el = (Element) child;                        
                        String ns = el.getNamespaceURI();
                        
                        // if disallowed contains already applied, use as is
                        // if disallowed contains requires and the snippet also - reevaluate
                        if (disallowedSchemas.containsKey(ns)) {
                            DisallowedInfo info = disallowedSchemas.get(ns);
                            if (info.reason == DisallowanceReason.ALREADY_APPLIED) {
                                disallowReasons.add(info);
                            } else {
                                List<String> allNS = getNamespaces(doc.getRootElement().getChildren());
                                allNS.remove(ns); // remove the currentNamespace
                                allNS.addAll(alreadyApplied);
                                // allNS now contains the pre-existing policies plus the ones that are part 
                                // of the snippet that we're applying (excluding the current), now re-evaluate
                                SchemaElement se = sr.getSchemaElement(ns, el.getName(), true, delegate);
                                DisallowedInfo newInfo = requirementsSatisfied(se, allNS);
                                if (newInfo != null) {
                                    disallowReasons.add(newInfo);
                                }
                            }
                        }
                    }
                }
                
                if (disallowReasons.size() == 0) {
                    allowedSnippets.add(name);
                } else {
                    disallowedSnippets.put(name, formatReasons(disallowReasons));//DisallowedInfo.alreadyApplied(disallowReasons));
                }
            } catch (Exception e) {
                e.printStackTrace();
            }           
        }
    }

    /** This method will format the reasons for disallowance if there are multiple reasons (e.g. for 
     * snippets). If both and DisallowanceReason.ALREADY_APPLIED and DisallowanceReason.REQUIRES are 
     * in there, the ALREADY_APPLIED ones will take precedence and the REQUIRES ones will be hidden until
     * all of the ALREADY_APPLIED ones have been taken care of.
     * The reasons for disallowance are then merged in the resulting object. 
     * @param disallowReasons A list of {@link DisallowedInfo} objects to merge
     * @return The resulting DisallowedInfo object.
     */
    private DisallowedInfo formatReasons(List<DisallowedInfo> disallowReasons) {
        boolean containsAlreadyApplied = false;
        for (DisallowedInfo i : disallowReasons) {
            if (i.reason == DisallowanceReason.ALREADY_APPLIED) {
                containsAlreadyApplied = true;
                break;
            }
        }
        
        if (containsAlreadyApplied) {
            for (Iterator<DisallowedInfo> it = disallowReasons.iterator(); it.hasNext(); ) {
                DisallowedInfo i = it.next();
                if (i.reason == DisallowanceReason.REQUIRES) {
                    // Remove these ones as the already applied ones take precedence
                    it.remove();
                }
            }
        }

        DisallowedInfo info = new DisallowedInfo();
        for (DisallowedInfo i : disallowReasons) {
            // All of these should have the same value in i.reason now
            info.causes.addAll(i.causes);
            info.reason = i.reason; 
        }
        
        return info;
    }

    private static List<String> getNamespaces(List children) {
        List<String> l = new ArrayList<String>();
        for (Object o : children) {
            if (o instanceof Element) {
                l.add(((Element) o).getNamespaceURI());
            }
        }
        return l;
    }

    public synchronized void refresh() {
        delegate.refresh();
        allowedSnippets = null;
    }

    private class ShadowProvider implements IShadowProvider {
        Map<String, List<Object>> shadowed = new HashMap<String, List<Object>>();
        Map<Object, String> reasons = new HashMap<Object, String>();
        boolean initialized = false;
        boolean snippetsInitialized = false;
        
        private synchronized void initialize() {
            if (initialized) { 
                return;
            }            
            initializeSchemas();
            
            SchemaRegistry sr = XefPlugin.getDefault().getSchemaRegistry();
            for (String filter : disallowedSchemas.keySet()) {
                try {
                    List<SchemaElement> elements = sr.getEntryElements(filter, true, true, delegate);
                    
                    for (SchemaElement el : elements) {
                        String cat = el.getCategory();
                        if (cat == null) {
                            cat = SchemaSelectionDialog.OTHER_CATEGORY_NAME;
                        }
                        List<Object> l = shadowed.get(cat);
                        if (l == null) {
                            l = new ArrayList<Object>();
                            shadowed.put(cat, l);
                        }
                        l.add(el);
                                                
                        DisallowedInfo info = disallowedSchemas.get(filter);
                        reasons.put(el, formatDisallowedInfo(el.getNameSpace(), info));
                    }
                } catch (Exception e) {
                    IStatus status = new Status(Status.ERROR, XefPlugin.ID, 0, "Problem handling schema", e);
                    XefPlugin.getDefault().getLog().log(status);
                }
            }
            initialized = true;
        }
        
        private synchronized void initializeSnippetCategory() {
            if (snippetsInitialized) {
                return;
            }           
            initializeSnippets();
            
            List<Object> l = shadowed.get(SchemaSelectionDialog.SNIPPET_CATEGORY_NAME);
            if (l == null) {
                l = new ArrayList<Object>();
                shadowed.put(SchemaSelectionDialog.SNIPPET_CATEGORY_NAME, l);               
            }           
            
            for (Map.Entry<String, DisallowedInfo> disallowed : disallowedSnippets.entrySet()) {
                l.add(disallowed.getKey());
                reasons.put(disallowed.getKey(), formatDisallowedInfo(disallowed.getKey(), disallowed.getValue()));
            }
            snippetsInitialized = true;
        }
        
        private String formatDisallowedInfo(String context, DisallowedInfo info) {
            if (info.causes.size() == 1 && info.reason == DisallowanceReason.ALREADY_APPLIED) {
                if (info.causes.iterator().next().equals(context)) {
                    // If the list only containt the current context element, simply return an empty string
                    return info.reason.toString();
                }
            }
            
            String sep = ", ";
            SchemaRegistry sr = XefPlugin.getDefault().getSchemaRegistry();
            StringBuilder sb = new StringBuilder();
            
            for (String uri : info.causes) {
                try {
                    for (SchemaElement se : sr.getEntryElements(uri, true, true, delegate)) {
                        sb.append(se.getDisplayName());
                        sb.append(sep);
                    }
                } catch (Exception e) {
                    IStatus status = new Status(Status.ERROR, XefPlugin.ID, 0, "Problem handling schema", e);
                    XefPlugin.getDefault().getLog().log(status);
                }
            }
            
            if (sb.length() > 2) {
                sb.setLength(sb.length() - 2); // remove the last ", "
                sb.append(' ');
            }
            
            if (info.reason == DisallowanceReason.REQUIRES) {
                // replace the last ',' with 'and/or'
                int idx = sb.lastIndexOf(sep);
                if (idx != -1) {
                    sb.replace(idx, idx + sep.length(), " and/or ");
                }
            }
            
            return sb.toString() + info.reason.toString();
        }

        public String getReason(Object shadowed) {
            initialize();
            
            return reasons.get(shadowed);
        }

        public String[] getShadowCategories() {
            initialize();
            
            Set<String> s = shadowed.keySet();
            return s.toArray(new String[s.size()]);
        }

        public Object[] getShadowed(String category) {
            if (SchemaSelectionDialog.SNIPPET_CATEGORY_NAME.equals(category)) {
                initializeSnippetCategory();
            } else {
                initialize();
            }
            
            List<Object> l = shadowed.get(category);
            if (l != null) {
                return l.toArray(new Object [l.size()]);
            }
            return null;
        }
    }
    
    enum DisallowanceReason { 
            ALREADY_APPLIED {
                @Override
                public String toString() {
                    return "already applied";
                }                
            },
            
            REQUIRES {
                @Override
                public String toString() {
                    return "required";
                }                
            } 
        };

    private static class DisallowedInfo {
        final Collection<String> causes = new TreeSet<String>();
        DisallowanceReason reason;
        
        private static DisallowedInfo alreadyApplied(String ... appl) {
            return alreadyApplied(Arrays.asList(appl));
        }
        
        private static DisallowedInfo alreadyApplied(Collection<String> appl) {
            DisallowedInfo info = new DisallowedInfo();
            info.causes.addAll(appl);
            info.reason = DisallowanceReason.ALREADY_APPLIED;
            return info;
        }

        public static DisallowedInfo required(Collection<String> req) {
            DisallowedInfo info = new DisallowedInfo();
            info.causes.addAll(req);
            info.reason = DisallowanceReason.REQUIRES;
            return info;
        }
    }
}
