/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.xef;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import javax.xml.namespace.QName;


import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.stp.ui.xef.XefPlugin;
import org.eclipse.stp.ui.xef.schema.SchemaElement;
import org.eclipse.stp.ui.xef.schema.SchemaRegistry;
import org.eclipse.stp.xef.util.InputStreamHelper;
import org.eclipse.swt.widgets.Display;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.Namespace;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

public final class XMLModelFactory {
    private static SAXBuilder builder = new SAXBuilder();
    public static boolean interactive = true; // set to false for testing
    
    private XMLModelFactory() {}

    public static List<XMLInstanceElement> getXMLInstanceElements(ISchemaProvider schemaProvider, boolean cacheSchemas, String baseElQName, IFile file) {
        String extension = file.getFileExtension();
        if (extension != null && extension.equals("policy")) {
            InputStream is = null;
            try {
                is = file.getContents(false);
                List<XMLInstanceElement> elements = new LinkedList<XMLInstanceElement>();
                createFromXML(is, baseElQName, elements, cacheSchemas, schemaProvider);
                return elements;
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                InputStreamHelper.close(is);
            }
        }   
        return null;
    }
        
    public static List<XMLInstanceElement> getXMLInstanceElements(ISchemaProvider schemaProvider, 
        boolean cacheSchemas, IXMLProvider ... xmlProviders) throws JDOMException, IOException {
        if (xmlProviders.length == 1) {
            List<XMLInstanceElement> elements = new LinkedList<XMLInstanceElement>();
            createFromXML(xmlProviders[0].getXML(), xmlProviders[0].getRootElementQName(), 
                elements, cacheSchemas, schemaProvider);
            return elements;            
        } else {
            IXMLProvider multiProvider = new MultiXMLProvider(Arrays.asList(xmlProviders));
            List<XMLInstanceElement> elements = new LinkedList<XMLInstanceElement>();
            createFromXML(multiProvider.getXML(), multiProvider.getRootElementQName(), 
                elements, cacheSchemas, schemaProvider);
            return elements;
        }
    }

    public static synchronized Element createFromXML(String xml, String baseElQName, 
        List<XMLInstanceElement> elements, boolean cacheSchemas, ISchemaProvider... providers) throws JDOMException, IOException {
        if (xml != null && xml.length() > 0) {        
            return createFromXML(new ByteArrayInputStream(xml.getBytes()), baseElQName, elements, cacheSchemas, providers);
        }
        return null;
    }
    
    @SuppressWarnings("unchecked")
    private static synchronized Element createFromXML(InputStream xmlIS, String baseElQName, 
        List<XMLInstanceElement> newPolicies, boolean cacheSchemas, ISchemaProvider... providers) 
        throws JDOMException, IOException {
        QName qname = QName.valueOf(baseElQName);        
        
        Document doc = builder.build(xmlIS);
        Element root = doc.getRootElement();
        QName rootQName = new QName(root.getNamespaceURI(), root.getName());
        Element baseEl;
        if (qname.equals(rootQName)) {
            baseEl = root;
        } else {
            baseEl = root.getChild(qname.getLocalPart(), Namespace.getNamespace(qname.getNamespaceURI()));
        }

        if (baseEl == null) {
            String errMsg = "The edited document does not conform to the expected format. " 
                + "Expected an element with qname: " + baseElQName;
            MessageDialog.openError(Display.getCurrent().getActiveShell(), "No matching document found", errMsg);
            Exception ex = new Exception(errMsg + System.getProperty("line.separator") 
                + "Actual document: " + new XMLOutputter(Format.getPrettyFormat()).outputString(doc)); 
            Status status = new Status(IStatus.ERROR, XefPlugin.ID, 0, errMsg, ex); 
            XefPlugin.getDefault().getLog().log(status);
            return null;
        }
        
        for (Element policy : (List<Element>) baseEl.getChildren()) {
            SchemaElement schema = SchemaRegistry.getDefault().getSchemaElement(policy.getNamespaceURI(), policy.getName(), cacheSchemas, providers);
            if (schema != null) {
                XMLInstanceElement pi = new XMLInstanceElement(schema, null, policy);
                newPolicies.add(pi);
                initNestedElements(policy, pi);
            } else {
                String message = "Could not find a schema for the policy " + new QName(policy.getNamespaceURI(), policy.getName()).toString() + "." +
                                    "\n\nRemoved this policy from the policy document.";
                if (interactive ) {
                    MessageDialog.openError(Display.getCurrent().getActiveShell(), "No schema found for template", message);
                } else {
                    throw new IOException(message);
                }
            }
        }
        Element newBase = new Element(baseEl.getName(), baseEl.getNamespacePrefix(), baseEl.getNamespaceURI());
        for (Namespace namespace : (List<Namespace>) baseEl.getAdditionalNamespaces()) {
            newBase.addNamespaceDeclaration(namespace);
        }
        return newBase;
    }

    @SuppressWarnings("unchecked")
    private static void initNestedElements(Element policyXML, XMLInstanceElement pi) {
        String basedOn = XMLInstanceElement.findBasedOn(policyXML);
        if (basedOn != null) {
            pi.setBasedOnSnippet(basedOn);
        }

        List<Element> detachedElements = new ArrayList<Element>(policyXML.getChildren().size());
        // work on copy because otherwise the detach wont like it
        for (Element child : new ArrayList<Element>(policyXML.getChildren())) {
            detachedElements.add((Element) child.detach());
        }
        
        
        if (pi.getTemplate().isSequenceOfAny()) {
            pi.setEmbeddedElements(detachedElements);
        } else {
            for (Element child : detachedElements) {
                SchemaElement schema = pi.getTemplate().getNestedElement(child.getName());
                if (schema != null) {
                    XMLInstanceElement piChild = new XMLInstanceElement(schema, pi, child);
                    pi.addChild(piChild, false);
                    initNestedElements(child, piChild);                    
                } else {
                    System.out.println("Subtag " + child + " not declared in schema");
                }
            }
            
//            // work on copy because otherwise the detach wont like it
//            for (Element child : new ArrayList<Element>(policyXML.getChildren())) {
//                SchemaElement schema = pi.getTemplate().getNestedElement(child.getName());
//                if (schema != null) {
//                    child.detach(); // Detach the child from the parent, because
//                                    // otherwise it will be added double again
//                                    // later.
//                    XMLInstanceElement piChild = new XMLInstanceElement(schema, pi, child);
//                    pi.addChild(piChild, false);
//                    initNestedElements(child, piChild);
//                } else {
//                    System.out.println("Subtag " + child + " not declared in schema");
//                }
//            }
        }
    }
}
