/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.runtimeprovider;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.internal.model.RuntimeCore;
import org.eclipse.stp.sc.jaxws.ScJaxWsPlugin;
import org.eclipse.stp.sc.jaxws.preferences.SCPreferenceConstants;
import org.eclipse.wst.server.core.IRuntime;
import org.eclipse.wst.server.core.ServerCore;

public class RuntimeProviderManager {
    
    private static final String EXT_POINT_SC_RUNTIME_PROVIDER = "org.eclipse.stp.sc.jaxws.runtimeProvider";
    private static final String EXT_ELT_SC_KIT_PROCESSOR = "KitProcessor";
    private static final String EXT_ATT_SC_PROCESSOR_CLASS = "kit_processor_class";
    private static final String EXT_ELT_SC_W2J_GEN = "wsdltoJavaGenerator";
    private static final String EXT_ATT_SC_W2J_GEN_CLASS = "class";
    private static final String EXT_ATT_SC_W2J_PARAM_PAGE = "parameter_page";
    private static final String EXT_ELT_SC_J2W_GEN = "javaToWsdlGenerator";
    private static final String EXT_ATT_SC_J2W_GEN_CLASS = "class";
    private static final String EXT_ELT_SC_W2J_WIZ = "wsdlToJavaWizard";
    private static final String EXT_ATT_SC_W2J_WIZ_CLASS = "class";
    private static final String EXT_ELT_SC_DIR_TEMP = "projectDirTemplate";
    private static final String EXT_ATT_SC_DIR_TEMP_CLASS = "class";
    private static final String EXT_ATT_SC_POST_PROCESSOR_CLASS = "postProjectProcessor";
    private static final String EXT_ELT_SC_JAVA_FIRST_WIZARD = "javaFirstWizard";
    private static final String EXT_ATT_SC_JAVA_FIRST_WIZARD_ADD_PAGE = "additional_page";
    private static final String EXT_ATT_SC_JAVA_FIRST_WIZARD_START_PAGE = "starting_point_page";
    private static final String EXT_ELT_SC_PROJECT_PROPERTY = "projectPropertyPage";
    private static final String EXT_ATT_SC_PROJECT_PROPERTY_CLASS = "class";
    private static final String EXT_ELT_SC_MENU_CUSTOMIZATION = "menuCustomization";
    private static final String EXT_ATT_SC_MENU_CUSTOMIZATION_ALL = "enable_default_jaxws_menu_All";
    private static final String EXT_ATT_SC_MENU_CUSTOMIZATION_PROJECTNATURE = "enable_default_jaxws_menu_ProjctNature";
    private static final String EXT_ATT_SC_MENU_CUSTOMIZATION_ANNOTATION = "enable_default_jaxws_menu_Annotation";
    private static final String EXT_ATT_SC_MENU_CUSTOMIZATION_HANDLER = "enable_default_jaxws_menu_Handler";
    private static final String EXT_ATT_SC_MENU_CUSTOMIZATION_CODE = "enable_default_jaxws_menu_CodeGenerator";
    private static final String EXT_ATT_SC_MENU_CUSTOMIZATION_WSDL = "enable_default_jaxws_menu_WSDLGenerator";
    private static final String EXT_ATT_SC_MENU_CUSTOMIZATION_ACTIVITY = "activity_group_id";
    
    private static final String EXT_ELT_SC_ENABLE_JAXWS = "jaxwsEnableHook";
    private static final String EXT_ATT_SC_ENABLE_JAXWS_CLASS = "class";
    private static final String EXT_ELT_SC_DISABLE_JAXWS = "jaxwsDisableHook";
    private static final String EXT_ATT_SC_DISABLE_JAXWS_CLASS = "class";
   
    
    
    private static final String BOOLEAN_VALUE_TRUE = "true";
    private static final String BOOLEAN_VALUE_FALSE = "false";
    
    private static final LoggingProxy LOG = LoggingProxy.getlogger(RuntimeProviderManager.class);

    private static RuntimeProviderManager instance;
    
    private static Map<String, IProjectDirTemplate> prjDirTempMap;
    
    private static Map<String, IPostProjectProcessor> prjPostProcessorMap;
    
    private RuntimeProviderManager(){
    }
    
    public static RuntimeProviderManager getInstance() {
        if (instance == null) {
            instance = new RuntimeProviderManager();
        }
        if(prjDirTempMap == null){
        	prjDirTempMap = new HashMap<String, IProjectDirTemplate>();
        }
        
        if(prjPostProcessorMap == null){
        	prjPostProcessorMap = new HashMap<String, IPostProjectProcessor>();
        }
        return instance;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeProviderManager#getWsdltoJavaGenerator(org.eclipse.core.resources.IProject)
     */
    public IWsdlToJavaGenerator getWsdltoJavaGenerator(String runtimeType){
        return loadWsdltoJavaGenerator(getRuntimeProviderUID(runtimeType));
    }
    
    public IWsdlToJavaWizard getWsdlToJavaWizard(String runtimeType) {
    	return loadWsdltoJavaWizard(getRuntimeProviderUID(runtimeType));
    }

    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeProviderManager#getJavaToWsdlGenerator(org.eclipse.core.resources.IProject)
     */
    public IJavaToWsdlGenerator getJavaToWsdlGenerator(String runtimeType){
        return loadJavaToWsdlGenerator(getRuntimeProviderUID(runtimeType));
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.stp.sc.jaxws.runtimeprovider.IRuntimeProviderManager#getRuntimeKitProcessor(org.eclipse.core.resources.IProject)
     */
    public IRuntimeKitProcessor getRuntimeKitProcessor(String runtimeType, String runtimeId){
        String providerExtUID = getRuntimeProviderUID(runtimeType);
        return loadRuntimeKitProcessor(providerExtUID);
    }

    
    
    public IProjectDirTemplate getProjectDirTemplate(String runtimeType) {
    	if(prjDirTempMap.keySet().contains(runtimeType)){
    		return prjDirTempMap.get(runtimeType);
    	}    		
    	IProjectDirTemplate template = loadProjectDirTemplate(getRuntimeProviderUID(runtimeType));
    	prjDirTempMap.put(runtimeType, template);
    	return template;
    }
    
    public IPostProjectProcessor getPostProjectProcessor(String runtimeType) {
    	if(prjPostProcessorMap.keySet().contains(runtimeType)){
    		return prjPostProcessorMap.get(runtimeType);
    	}    		
    	IPostProjectProcessor processor = loadPostProjectProcessor(getRuntimeProviderUID(runtimeType));
    	prjPostProcessorMap.put(runtimeType, processor);
    	return processor;
    }
      
    /**
     * get the runtime provider id according to runtime type
     * @param runtimeType
     * @return
     */
    public static String getRuntimeProviderUID(String runtimeType) {
    	IExtension[] providers = getAllRegistredProviders();
    	for (IExtension provider : providers) {
            for (IConfigurationElement cfgElt : provider.getConfigurationElements()) {
                if (cfgElt.getName().equals("supportedRuntime")) {
            	    //check the runtime type
                	if (cfgElt.getAttribute("type").equals(runtimeType)) {
                		return provider.getUniqueIdentifier();
                	}
            	}
            }
    	}
    	Exception e = new Exception();
    	LOG.error("no runtime provider for runtime type:" + runtimeType, e);
    	
    	return null;
    	
    }

 
    public IExtension getExtension(String ID){
        IExtensionRegistry reg = Platform.getExtensionRegistry();
        return reg.getExtension(EXT_POINT_SC_RUNTIME_PROVIDER, ID);
    }
    
    public static IExtension[] getAllRegistredProviders() {
        IExtensionRegistry reg = Platform.getExtensionRegistry();
        return reg.getExtensionPoint(EXT_POINT_SC_RUNTIME_PROVIDER).getExtensions();
    }

    private IWsdlToJavaGenerator loadWsdltoJavaGenerator(String ID){
    	if (getExtension(ID) == null) {
    		return null;
    	}
        IConfigurationElement[] extElts = getExtension(ID).getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_W2J_GEN)) {
                try {
                    Object kitProcessorObj = cfgElt.createExecutableExtension(EXT_ATT_SC_W2J_GEN_CLASS);
                    if (kitProcessorObj instanceof IWsdlToJavaGenerator) {
                        return (IWsdlToJavaGenerator)kitProcessorObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(ID,
                                                         EXT_ATT_SC_PROCESSOR_CLASS,
                                                         IWsdlToJavaGenerator.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(ID, EXT_ATT_SC_PROCESSOR_CLASS), e);
                }
                return null;
            }
        }
        return null;
    }
    
    public IProjectDirTemplate loadProjectDirTemplate(String providerExtUID) {
    	if (getExtension(providerExtUID) == null) {
    		return null;
    	}
        IConfigurationElement[] extElts = getExtension(providerExtUID).getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_DIR_TEMP)) {
                try {
                    Object kitProcessorObj = cfgElt.createExecutableExtension(EXT_ATT_SC_DIR_TEMP_CLASS);
                    if (kitProcessorObj instanceof IProjectDirTemplate) {
                        return (IProjectDirTemplate)kitProcessorObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(providerExtUID,
                    		EXT_ATT_SC_DIR_TEMP_CLASS,
                            IProjectDirTemplate.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(providerExtUID, EXT_ATT_SC_DIR_TEMP_CLASS), e);
                }
                return null;
            }
        }
        return null;
    }

    public IPostProjectProcessor loadPostProjectProcessor(String providerExtUID) {
        IConfigurationElement[] extElts = getExtension(providerExtUID).getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_DIR_TEMP)) {
                try {
                	Object obj = cfgElt
							.getAttribute(EXT_ATT_SC_POST_PROCESSOR_CLASS);
					if (obj == null) {
						return null;
					}
					    
                	
                    Object prjProcessor = cfgElt.createExecutableExtension(EXT_ATT_SC_POST_PROCESSOR_CLASS);
                    if (prjProcessor instanceof IPostProjectProcessor) {
                        return (IPostProjectProcessor)prjProcessor;
                    }
                    LOG.error(getLogErrMsgWrongClassType(providerExtUID,
                    		EXT_ATT_SC_POST_PROCESSOR_CLASS,
                    		IPostProjectProcessor.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(providerExtUID, EXT_ATT_SC_POST_PROCESSOR_CLASS), e);
                }
                return null;
            }
        }
        return null;
    }
    
    
    private IWsdlToJavaWizard loadWsdltoJavaWizard(String ID){
    	if (getExtension(ID) == null) {
    		return null;
    	}
        IConfigurationElement[] extElts = getExtension(ID).getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_W2J_WIZ)) {
                try {
                    Object kitProcessorObj = cfgElt.createExecutableExtension(EXT_ATT_SC_W2J_WIZ_CLASS);
                    if (kitProcessorObj instanceof IWsdlToJavaWizard) {
                        return (IWsdlToJavaWizard)kitProcessorObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(ID,
                                                         EXT_ATT_SC_PROCESSOR_CLASS,
                                                         IWsdlToJavaWizard.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(ID, EXT_ATT_SC_PROCESSOR_CLASS), e);
                }
                return null;
            }
        }
        return null;
    }

    private IJavaToWsdlGenerator loadJavaToWsdlGenerator(String ID){
    	if (getExtension(ID) == null) {
    		return null;
    	}
        IConfigurationElement[] extElts = getExtension(ID).getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_J2W_GEN)) {
                try {
                    Object kitProcessorObj = cfgElt.createExecutableExtension(EXT_ATT_SC_J2W_GEN_CLASS);
                    if (kitProcessorObj instanceof IJavaToWsdlGenerator) {
                        return (IJavaToWsdlGenerator)kitProcessorObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(ID,
                                                         EXT_ATT_SC_PROCESSOR_CLASS,
                                                         IJavaToWsdlGenerator.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(ID, EXT_ATT_SC_PROCESSOR_CLASS), e);
                }
                return null;
            }
        }
        return null;
    }
    
    public IRuntimeKitProcessor loadRuntimeKitProcessor(String uniqueIdentifier){
    	if (getExtension(uniqueIdentifier) == null) {
    		return null;
    	}
        IConfigurationElement[] extElts = getExtension(uniqueIdentifier).getConfigurationElements();
        
        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_KIT_PROCESSOR)) {
                try {
                    Object kitProcessorObj = cfgElt.createExecutableExtension(EXT_ATT_SC_PROCESSOR_CLASS);
                    if (kitProcessorObj instanceof IRuntimeKitProcessor) {
                        return (IRuntimeKitProcessor)kitProcessorObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(uniqueIdentifier,
                                                         EXT_ATT_SC_PROCESSOR_CLASS,
                                                         IRuntimeKitProcessor.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(uniqueIdentifier, EXT_ATT_SC_PROCESSOR_CLASS), e);
                }
                return null;
            }
        }
        return null;
    }
    
    private String getLogErrMsgNotCLass(String extId, String attributeName) {
        // this msg is intended for the logs, no point to use resources bundles for this
        return "the 'IRuntimeKitProcessor' could not be instanciated from the extension "
                      + extId 
                      + ". check value of the attribute '"
                      + attributeName 
                      + "'";
    }

    @SuppressWarnings("unchecked")
    private String getLogErrMsgWrongClassType(String extId,
                                                     String attributeName,
                                                     Class superClass) {
        // this msg is intended for the logs, no point to use resources bundles for this
        return "the class specified in the attribute '"
                  + attributeName 
                  + "' of the extension '"
                  + extId
                  + "' does not extend the interface '"
                  + superClass.getCanonicalName()
                  + "'";
    }

    /**
     * get parameter page according to uid
     * @param id
     * @return
     */
    private IParameterPage loadParameterPage(String id) {
        IConfigurationElement[] extElts = getExtension(id).getConfigurationElements();

        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_W2J_GEN)) {
                try {
                    Object kitParamPageObj = cfgElt.createExecutableExtension(EXT_ATT_SC_W2J_PARAM_PAGE);
                    if (kitParamPageObj instanceof IParameterPage) {
                        return (IParameterPage)kitParamPageObj;
                    }
                    LOG.error(getLogErrMsgWrongClassType(id,
                                                         EXT_ATT_SC_W2J_PARAM_PAGE,
                                                         IJavaToWsdlGenerator.class));
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(id, EXT_ATT_SC_W2J_PARAM_PAGE), e);
                }
                return null;
            }
        }
        return null;
    }

	public IParameterPage getParameterPage(String runtimeType) {
		String providerExtUID = getRuntimeProviderUID(runtimeType);
        return loadParameterPage(providerExtUID);
	}
	
	public void setUseDefaultRuntime(boolean isUseDefault){
		IPreferenceStore mStore = ScJaxWsPlugin.getDefault().getPreferenceStore();
		mStore.setValue(SCPreferenceConstants.KEY_USE_DEFAULT_RUNTIME, isUseDefault);
	}
	
	public boolean getUseDefaultRuntime(){
		IPreferenceStore mStore = ScJaxWsPlugin.getDefault().getPreferenceStore();
		return mStore.getBoolean(SCPreferenceConstants.KEY_USE_DEFAULT_RUNTIME);
	}
	
	public void setDefaultRuntime(String type, String id){
		String[] ids = getRuntimeIDs(type);
		if(ids != null && ids.length > 0){
			for(String newID:ids){
				if (newID.equals(id)){
					IPreferenceStore mStore = ScJaxWsPlugin.getDefault().getPreferenceStore();
					mStore.setValue(SCPreferenceConstants.KEY_DEFAULT_RUNTIME_TYPE, type);
					mStore.setValue(SCPreferenceConstants.KEY_DEFAULT_RUNTIME_ID, id);
					return;
				}
			}
		}
		LOG.error("can not find the runtime id: " + id +" corresponding to the type: " + type);
	}
	
	public String getDefaultRuntimeType(){
		IPreferenceStore mStore = ScJaxWsPlugin.getDefault().getPreferenceStore();
		return mStore.getString(SCPreferenceConstants.KEY_DEFAULT_RUNTIME_TYPE);
	}
	
	public String getDefaultRuntimeID(){
		IPreferenceStore mStore = ScJaxWsPlugin.getDefault().getPreferenceStore();
		return mStore.getString(SCPreferenceConstants.KEY_DEFAULT_RUNTIME_ID);
	}
	
	public IAdditionalPageContents getAdditionalPage(String runtimeType) {
		String providerExtUID = getRuntimeProviderUID(runtimeType);
        return (IAdditionalPageContents)loadPage(providerExtUID, EXT_ELT_SC_JAVA_FIRST_WIZARD, EXT_ATT_SC_JAVA_FIRST_WIZARD_ADD_PAGE, IAdditionalPageContents.class);
	}
	
	public IStartingPointPageContents getStartingPointPage(String runtimeType) {
		String providerExtUID = getRuntimeProviderUID(runtimeType);
        return (IStartingPointPageContents)loadPage(providerExtUID, EXT_ELT_SC_JAVA_FIRST_WIZARD, EXT_ATT_SC_JAVA_FIRST_WIZARD_START_PAGE, IAdditionalPageContents.class);
	}
	
	public IPropertyPageContents getPropertyPage(String runtimeType) {
		String providerExtUID = getRuntimeProviderUID(runtimeType);
        return (IPropertyPageContents)loadPage(providerExtUID, EXT_ELT_SC_PROJECT_PROPERTY, EXT_ATT_SC_PROJECT_PROPERTY_CLASS, IPropertyPageContents.class);
	}
	
	public String[] getRuntimeIDs(String runtimeType){
		ArrayList<String> runtimeIDs = new ArrayList<String>();
		IRuntime[] runtimes = ServerCore.getRuntimes();
		if(runtimes != null && runtimes.length > 0){
			for(IRuntime runtime:runtimes){
				if(runtime.getRuntimeType().getId().endsWith(runtimeType)){
					runtimeIDs.add(runtime.getId());
				}
			}
		}
		return runtimeIDs.toArray(new String[]{});
	}
	
	//public IWizardBasePage loadPage(String runtimeType)
	
	//public IWizardPageBase getWizardPage()

    /**
     * get starting point page according to uid
     * @param id
     * @return
     */
    private IExtPageContents loadPage(String id, String element, String attribute, Class superClass) {
        IConfigurationElement[] extElts = getExtension(id).getConfigurationElements();

        for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(element)) {
                try {
                	if(cfgElt.getAttribute(attribute) != null){
                		Object pageObj = cfgElt.createExecutableExtension(attribute);
                        if (pageObj instanceof IExtPageContents) {
                            return (IExtPageContents)pageObj;
                        }
                        LOG.error(getLogErrMsgWrongClassType(id,
                        		attribute,
                        		superClass));
                	}else{
                		LOG.info(getLogErrMsgWrongClassType(id,
                        		attribute,
                        		superClass));
                	}
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(id, attribute), e);
                }
                return null;
            }
        }
        return null;
    }
    
    public IEnableJaxwsHook getEnableJaxwsHook(String runtimeType){
    	String providerExtUID = getRuntimeProviderUID(runtimeType);
    	IConfigurationElement[] extElts = getExtension(providerExtUID).getConfigurationElements();
    	for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_ENABLE_JAXWS)) {
                try {
                	if(cfgElt.getAttribute(EXT_ATT_SC_ENABLE_JAXWS_CLASS) != null){
                		Object pageObj = cfgElt.createExecutableExtension(EXT_ATT_SC_ENABLE_JAXWS_CLASS);
                        if (pageObj instanceof IEnableJaxwsHook) {
                            return (IEnableJaxwsHook)pageObj;
                        }
                        LOG.error(getLogErrMsgWrongClassType(EXT_ELT_SC_ENABLE_JAXWS,
                        		EXT_ATT_SC_ENABLE_JAXWS_CLASS,
                        		IEnableJaxwsHook.class));
                	}else{
                		LOG.info(getLogErrMsgWrongClassType(EXT_ELT_SC_ENABLE_JAXWS,
                				EXT_ATT_SC_ENABLE_JAXWS_CLASS,
                				IEnableJaxwsHook.class));
                	}
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(EXT_ELT_SC_ENABLE_JAXWS, EXT_ATT_SC_ENABLE_JAXWS_CLASS), e);
                }
                return null;
            }
        }
        return null;
    	
    }
    
    public IDisableJaxwsHook getDisableJaxwsHook(String runtimeType){
    	String providerExtUID = getRuntimeProviderUID(runtimeType);
    	IConfigurationElement[] extElts = getExtension(providerExtUID).getConfigurationElements();
    	for (int extIndex = 0; extIndex < extElts.length; extIndex++) {
            IConfigurationElement cfgElt = extElts[extIndex];
            if (cfgElt.getName().equals(EXT_ELT_SC_DISABLE_JAXWS)) {
                try {
                	if(cfgElt.getAttribute(EXT_ATT_SC_DISABLE_JAXWS_CLASS) != null){
                		Object pageObj = cfgElt.createExecutableExtension(EXT_ATT_SC_DISABLE_JAXWS_CLASS);
                        if (pageObj instanceof IDisableJaxwsHook) {
                            return (IDisableJaxwsHook)pageObj;
                        }
                        LOG.error(getLogErrMsgWrongClassType(EXT_ELT_SC_DISABLE_JAXWS,
                        		EXT_ATT_SC_DISABLE_JAXWS_CLASS,
                        		IDisableJaxwsHook.class));
                	}else{
                		LOG.info(getLogErrMsgWrongClassType(EXT_ELT_SC_DISABLE_JAXWS,
                				EXT_ATT_SC_DISABLE_JAXWS_CLASS,
                				IDisableJaxwsHook.class));
                	}
                } catch (CoreException e) {
                    LOG.error(getLogErrMsgNotCLass(EXT_ELT_SC_DISABLE_JAXWS, EXT_ATT_SC_DISABLE_JAXWS_CLASS), e);
                }
                return null;
            }
        }
        return null;
    	
    }
    
    public String getRuntimeType(IProject project){
    	return RuntimeCore.getRuntimeType(project);
    }
      
}
