/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.jaxws.wizards.annotations;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.dom.Annotation;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.MemberValuePair;
import org.eclipse.jdt.core.dom.rewrite.ASTRewrite;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.Document;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.annotations.ScAnnotationSupportUtils;
import org.eclipse.stp.sc.common.utils.JDTUtils;
import org.eclipse.stp.sc.common.wizards.AbstractScWizard;
import org.eclipse.stp.sc.jaxws.ScJaxWsResources;
import org.eclipse.stp.sc.jaxws.preferences.PreferencesAccessor;
import org.eclipse.stp.sc.jaxws.utils.ScJDTUtils;
import org.eclipse.stp.sc.jaxws.wizards.annotations.webservice.CreateWebServiceWizard;
import org.eclipse.text.edits.TextEdit;

public abstract class CreateAnnotationWizard extends AbstractScWizard {
    // key to wizard data
    public static final String DATA_KEY_COMP_UNIT = "DATA_KEY_COMP_UNIT";
    public static final String DATA_KEY_AST_ROOT = "DATA_KEY_AST_ROOT";
    public static final String DATA_KEY_MEMBER = "DATA_KEY_MEMBER";
    public static final String DATA_KEY_ORIGINAL_TARGET = "DATA_KEY_ORIGINAL_TARGET";
    public static final String DATA_KEY_ANNOTATION_VALUES = "DATA_KEY_ANNOTATION_VALUES";
    
    private IMember targetMember;
    
    @SuppressWarnings("unchecked")
    protected Map<Class<? extends java.lang.annotation.Annotation>, Boolean> ANNOTATION_REQUIRED =
        new TreeMap<Class<? extends java.lang.annotation.Annotation>, Boolean>(new Comparator(){
            public int compare(Object arg0, Object arg1) {
                String arg0Str = ((Class)arg0).getSimpleName();
                String arg1Str = ((Class)arg1).getSimpleName();
                return arg0Str.compareToIgnoreCase(arg1Str);
            }
        });

    private static final LoggingProxy LOG = LoggingProxy.getlogger(CreateAnnotationWizard.class);
    

    public CreateAnnotationWizard(Object target) {
        super();
        
        this.
        setOriginalWizardTargetToData(target);
        
        if (target instanceof IMember) {
            initializeWizardData((IMember)target);
        }
    }

    public void initializeWizardData(IMember aTargetMember) {
        
        targetMember = aTargetMember;

        setCuToData(targetMember.getCompilationUnit());
        setCuAstNodeToData(JDTUtils.getDomRootCompilationUnit(targetMember));
        setJavaMemberToData(targetMember);
    }

    @Override
    public boolean performFinish() {
        ICompilationUnit icu = getCuFromData();
        CompilationUnit astRoot = getCuAstNodeFromData();

        try {
            // Take a snapshot of the editor content as it is
            // right now so that we can apply edits to it in an
            // incremental manner later.
            Document document = new Document(icu.getBuffer().getContents());

            // Make the updates using a rewriter
            ASTRewrite rewrite = ASTRewrite.create(astRoot.getAST());

            if (updateAST(astRoot, rewrite)) {
                // Compute the text edits from the previous content 
                // and apply them.
                TextEdit edits = rewrite.rewriteAST(document, icu.getJavaProject().getOptions(true));
                edits.apply(document);

                // Update the contents of the editor
                icu.getBuffer().setContents(document.get());

//                ScNature.addToProject(astRoot.getJavaElement().getJavaProject(), null);
            }
        } catch (Exception e) {
            LOG.error(e);
        }

        return true;
    }

    
    protected boolean verifyUpdateAnnotation(IMember member, Annotation annotation) {

        boolean update = true;
        
        // Check if an annotation of this type already exists
        if (JDTUtils.hasAnnotation(member, annotation)) {
            String strTitle = ScJaxWsResources.getString("ConfirmAnnotationReplace.Title");
            String strDesc = ScJaxWsResources.getFromMsgFormat("ConfirmAnnotationReplace.Exists.Msg",
                                                               member.getElementName()
                                                                   + ","
                                                                   + annotation.getTypeName(),
                                                               false);
            // query if replace or cancel
            update = MessageDialog.openConfirm(null, strTitle, strDesc);
        }
        return update;
    }

    protected abstract boolean updateAST(CompilationUnit astRoot, ASTRewrite astRewrite);
    
    protected boolean addAnnotationToMethodMember(CompilationUnit astRoot,
                                                  ASTRewrite astRewrite,
                                                  IMethod methodMember,
                                                  Class<? extends java.lang.annotation.Annotation> annClass,
                                                  boolean supportDefault) {
        
        String wizDataKey = getAttributesDataKey(annClass);
        Annotation annotationToAdd;
        if (supportDefault && !PreferencesAccessor.areAnnWizardsDisplaied()) {
            annotationToAdd = ScAnnotationSupportUtils.getDefaultedAnnotationNode(annClass,
                                                                              astRoot,
                                                                              methodMember,
                                                                              null);
        } else if (getAnnAttributesFromData(wizDataKey) != null) {
            annotationToAdd = JDTUtils.newNormalAnnotation(astRoot,
                                                           annClass.getSimpleName(),
                                                           getAnnAttributesFromData(wizDataKey));
        } else {
            annotationToAdd = null;
        }
        
        if (annotationToAdd != null && verifyUpdateAnnotation(methodMember, annotationToAdd)) {
            ScJDTUtils.addAnnotationOnMethod(astRoot, annotationToAdd, methodMember, astRewrite, true);
            return true;
        }
        return false;
    }

    protected boolean addAnnotationToTypeMember(CompilationUnit astRoot,
                                                ASTRewrite astRewrite,
                                                IType typeMember,
                                                Class<? extends java.lang.annotation.Annotation> annClass,
                                                boolean supportDefault) {
        
        String wizDataKey = getAttributesDataKey(annClass);
        Annotation annotationToAdd;
        if (supportDefault && !PreferencesAccessor.areAnnWizardsDisplaied()) {
            annotationToAdd = ScAnnotationSupportUtils.getDefaultedAnnotationNode(annClass,
                                                                              astRoot,
                                                                              typeMember,
                                                                              null);
        } else if (getAnnAttributesFromData(wizDataKey) != null) {
            annotationToAdd = JDTUtils.newNormalAnnotation(astRoot,
                                                           annClass.getSimpleName(),
                                                           getAnnAttributesFromData(wizDataKey));
        } else {
            annotationToAdd = null;
        }
        
        if (annotationToAdd != null && verifyUpdateAnnotation(typeMember, annotationToAdd)) {
            ScJDTUtils.addAnnotationOnType(astRoot, annotationToAdd, typeMember, astRewrite, true);
            return true;
        }
        return false;
    }

    /**
     * conveniant accessor method to the data model of the wizard
     */
    public final IMember getJavaMemberFromData() {
        return (IMember)wizardData.get(CreateWebServiceWizard.DATA_KEY_MEMBER);
    }
    
    /**
     * conveniant accessor method to the data model of the wizard
     */
    public final Object getOriginalWizardTargetFromData() {
        return wizardData.get(CreateWebServiceWizard.DATA_KEY_ORIGINAL_TARGET);
    }
    
    
    /**
     * conveniant accessor method to the data model of the wizard
     */
    public final CompilationUnit getCuAstNodeFromData() {
        return (CompilationUnit)wizardData.get(CreateWebServiceWizard.DATA_KEY_AST_ROOT);
    }

    /**
     * conveniant accessor method to the data model of the wizard
     */
    public final ICompilationUnit getCuFromData() {
        return (ICompilationUnit)wizardData.get(CreateWebServiceWizard.DATA_KEY_COMP_UNIT);
    }

    /**
     * conveniant accessor method to the data model of the wizard
     * @param wizDataKey TODO
     */
    @SuppressWarnings("unchecked")
    public final List<MemberValuePair> getAnnAttributesFromData(String wizDataKey) {
        return (List<MemberValuePair>)wizardData.get(wizDataKey);
    }
    
    /**
     * conveniant accessor method to the data model of the wizard
     */
    public final void setJavaMemberToData(IMember aJavaMember) {
        wizardData.put(CreateWebServiceWizard.DATA_KEY_MEMBER, aJavaMember);
    }
    
    /**
     * conveniant accessor method to the data model of the wizard
     */
    public final void setOriginalWizardTargetToData(Object aTarget) {
        wizardData.put(CreateWebServiceWizard.DATA_KEY_ORIGINAL_TARGET, aTarget);
    }

    /**
     * conveniant accessor method to the data model of the wizard
     */
    public final void setCuAstNodeToData(CompilationUnit aCompilationUnitAstNode) {
        wizardData.put(CreateWebServiceWizard.DATA_KEY_AST_ROOT, aCompilationUnitAstNode);
    }

    /**
     * conveniant accessor method to the data model of the wizard
     */
    public final void setCuToData(ICompilationUnit aCompilationUnit) {
        wizardData.put(CreateWebServiceWizard.DATA_KEY_COMP_UNIT, aCompilationUnit);
    }

    /**
     * conveniant accessor method to the data model of the wizard
     * @param wizDataKey TODO
     */
    public final void setAnnAttributesToData(String wizDataKey, List<MemberValuePair> someAnnAttributes) {
        wizardData.put(wizDataKey, someAnnAttributes);
    }

    public static String getAttributesDataKey(Class<? extends java.lang.annotation.Annotation> annotationClass) {
        return DATA_KEY_ANNOTATION_VALUES + annotationClass.getSimpleName();
    }
}
