/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.ui.internal.preferences;

import java.net.URL;

import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.stp.b2j.core.publicapi.importresolver.StandardWsdlImportResolver;
import org.eclipse.stp.b2j.ui.UiPlugin;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

public class ImportsPrefPage extends PreferencePage implements IWorkbenchPreferencePage, SelectionListener {

	Composite main_panel;

	Group proxy_panel;
	Button proxy_use;
	Label proxy_host_label;
	Text proxy_host;
	Label proxy_port_label;
	Text proxy_port;
		
	Group cache_panel;
	Button cache_use;
	List cache_list;
	Button cache_add;
	Button cache_delete;
	Button cache_clear;
	Text cache_content;

	String previous_url = "";
	
		protected Control createContents(Composite parent) {
			
			main_panel = new Composite(parent,SWT.NONE);
			main_panel.setLayout(new GridLayout());
			
			//
			// Proxy panel
			//
			proxy_panel = new Group(main_panel,SWT.NONE);
			proxy_panel.setLayout(new GridLayout(2,false));
			proxy_panel.setLayoutData(new GridData(GridData.FILL,GridData.FILL,true,false,1,1));
			
			proxy_use = new Button(proxy_panel,SWT.CHECK);
			proxy_use.setText(UiPlugin.getString("USE_PROXY"));
			proxy_use.setLayoutData(new GridData(GridData.BEGINNING,GridData.BEGINNING,true,false,2,1));
			
			proxy_host_label = new Label(proxy_panel,SWT.NONE);
			proxy_host_label.setText(UiPlugin.getString("PROXY_HOST"));
			proxy_host_label.setLayoutData(new GridData(GridData.FILL,GridData.BEGINNING,false,false,1,1));

			proxy_host = new Text(proxy_panel,SWT.BORDER);
			proxy_host.setLayoutData(new GridData(GridData.FILL,GridData.BEGINNING,true,false,1,1));
			
			proxy_port_label = new Label(proxy_panel,SWT.NONE);
			proxy_port_label.setText(UiPlugin.getString("PROXY_PORT"));
			proxy_port_label.setLayoutData(new GridData(GridData.FILL,GridData.BEGINNING,false,false,1,1));

			proxy_port = new Text(proxy_panel,SWT.BORDER);
			proxy_port.setLayoutData(new GridData(GridData.FILL,GridData.BEGINNING,true,false,1,1));
			
			//
			// Cache panel
			//
			cache_panel = new Group(main_panel,SWT.NONE);
			cache_panel.setLayout(new GridLayout(4,false));
			cache_panel.setLayoutData(new GridData(GridData.FILL,GridData.FILL,true,true,1,1));
			
			cache_use = new Button(cache_panel,SWT.CHECK);
			cache_use.setText(UiPlugin.getString("USE_CACHE"));
			cache_use.setLayoutData(new GridData(GridData.BEGINNING,GridData.BEGINNING,true,false,4,1));
			
			cache_list = new List(cache_panel,SWT.BORDER|SWT.V_SCROLL|SWT.H_SCROLL);
			cache_list.setLayoutData(new GridData(GridData.FILL,GridData.FILL,true,true,4,1));
			
			cache_add = new Button(cache_panel,SWT.PUSH);
			cache_add.setText(UiPlugin.getString("ADD"));
			cache_add.setLayoutData(new GridData(GridData.BEGINNING,GridData.BEGINNING,false,false,1,1));

			cache_delete = new Button(cache_panel,SWT.PUSH);
			cache_delete.setText(UiPlugin.getString("REMOVE"));
			cache_delete.setLayoutData(new GridData(GridData.BEGINNING,GridData.BEGINNING,false,false,1,1));

			cache_clear = new Button(cache_panel,SWT.PUSH);
			cache_clear.setText(UiPlugin.getString("CLEAR"));
			cache_clear.setLayoutData(new GridData(GridData.BEGINNING,GridData.BEGINNING,false,false,1,1));

			cache_content = new Text(cache_panel,SWT.MULTI|SWT.BORDER|SWT.H_SCROLL|SWT.V_SCROLL);
			cache_content.setLayoutData(new GridData(GridData.FILL,GridData.FILL,true,true,4,1));
			
			//
			// Event handling
			//
			proxy_use.addSelectionListener(this);
			cache_use.addSelectionListener(this);
			cache_list.addSelectionListener(this);
			cache_add.addSelectionListener(this);
			cache_delete.addSelectionListener(this);
			cache_clear.addSelectionListener(this);
			
			//
			// Initial state
			//
			performInit();
			
			return main_panel;
		}

		public void init(IWorkbench workbench) {
			setPreferenceStore(UiPlugin.getDefault().getPreferenceStore());
		}
		
		private void updateEnabled() {
			if (proxy_use.getSelection()) {
				proxy_host.setEnabled(true);
				proxy_port.setEnabled(true);
			} else {
				proxy_host.setEnabled(false);
				proxy_port.setEnabled(false);
			}
			
			if (cache_use.getSelection()) {
				cache_list.setEnabled(true);
				cache_add.setEnabled(true);
				cache_delete.setEnabled(true);
				cache_clear.setEnabled(true);
				cache_content.setEnabled(true);
			} else {
				cache_list.setEnabled(false);
				cache_add.setEnabled(false);
				cache_delete.setEnabled(false);
				cache_clear.setEnabled(false);
				cache_content.setEnabled(false);
			}
		}
		
		protected IPreferenceStore doGetPreferenceStore() {
			return UiPlugin.getDefault().getPreferenceStore();
		}
		
		private void performInit() {
			//
			// Load from preference store
			//
			IPreferenceStore store = getPreferenceStore();
			
			proxy_use.setSelection(store.getBoolean(PreferenceConstants.USE_PROXY));
			proxy_host.setText(store.getString(PreferenceConstants.PROXY_HOST));
			proxy_port.setText(""+store.getInt(PreferenceConstants.PROXY_PORT));
			
			cache_use.setSelection(store.getBoolean(PreferenceConstants.USE_CACHE));
			
			int cache_size = store.getInt(PreferenceConstants.CACHE_SIZE);
			String[] locations = new String[cache_size];
			for (int i = 0; i < cache_size; i++) {
				locations[i] = store.getString(PreferenceConstants.CACHE_LOCATION+i);
			}
			cache_list.setItems(locations);
			
			updateEnabled();
		}
		
		protected void performDefaults() {
			//
			// Load from preference store (defaults)
			//
			IPreferenceStore store = getPreferenceStore();
			
			proxy_use.setSelection(store.getDefaultBoolean(PreferenceConstants.USE_PROXY));
			proxy_host.setText(store.getDefaultString(PreferenceConstants.PROXY_HOST));
			proxy_port.setText(""+store.getDefaultInt(PreferenceConstants.PROXY_PORT));
			
			cache_use.setSelection(store.getDefaultBoolean(PreferenceConstants.USE_CACHE));
			
			int cache_size = store.getInt(PreferenceConstants.CACHE_SIZE);
			String[] locations = new String[cache_size];
			for (int i = 0; i < cache_size; i++) {
				locations[i] = store.getString(PreferenceConstants.CACHE_LOCATION+i);
			}
			cache_list.setItems(locations);

			updateEnabled();
			
			super.performDefaults();
		}			
		
		public boolean performOk() {
			//
			// Store to preference store
			//
			IPreferenceStore store = getPreferenceStore();
			
			store.setValue(PreferenceConstants.USE_PROXY,proxy_use.getSelection());
			store.setValue(PreferenceConstants.PROXY_HOST,proxy_host.getText());
			try {
				store.setValue(PreferenceConstants.PROXY_PORT,Integer.parseInt(proxy_port.getText()));
			} catch (Exception e) {
				store.setValue(PreferenceConstants.PROXY_PORT,store.getDefaultInt(PreferenceConstants.PROXY_PORT));
			}
			
			store.setValue(PreferenceConstants.USE_CACHE,cache_use.getSelection());
			
			String[] locations = cache_list.getItems();
			store.setValue(PreferenceConstants.CACHE_SIZE,locations.length);
			for (int i = 0; i < locations.length; i++) {
				store.setValue(PreferenceConstants.CACHE_LOCATION+i,locations[i]);
			}
			
			updateEnabled();
			return super.performOk();
		}
		
		private void showCacheContent(int n) {
			IPreferenceStore store = getPreferenceStore();
			String tmp = store.getString(PreferenceConstants.CACHE_CONTENT+n);
			if (tmp != null) {
				cache_content.setText(tmp);
			}
		}
		
		private void deleteCacheEntry(int index) {
			IPreferenceStore store = getPreferenceStore();

			int size = store.getInt(PreferenceConstants.CACHE_SIZE);
			store.setValue(PreferenceConstants.CACHE_SIZE,Math.max(0,size-1));

			for (int i = index; i < size; i++) {
				store.setValue(PreferenceConstants.CACHE_LOCATION+i,store.getString(PreferenceConstants.CACHE_LOCATION+(i+1)));
			}
			store.setValue(PreferenceConstants.CACHE_LOCATION+(size-1),"");
			
			try {
				cache_list.remove(index);
			} catch (Exception e) {
			}
		}
		
		public void widgetSelected(Object o) {
			if (o == proxy_use) {
				updateEnabled();
			} else if (o == cache_use) {
				updateEnabled();
			} else if (o == cache_list) {
				int index = cache_list.getSelectionIndex();
				if (index != -1) {
					showCacheContent(index);
				}
				
			} else if (o == cache_clear) {
				IPreferenceStore store = getPreferenceStore();
				int size = store.getInt(PreferenceConstants.CACHE_SIZE);
				
				for (int i = 0; i < size; i++) {
					deleteCacheEntry(0);
				}
				
			} else if (o == cache_delete) {
				IPreferenceStore store = getPreferenceStore();

				int index = cache_list.getSelectionIndex();
				if (index != -1) {

					deleteCacheEntry(index);
				}
				
			} else if (o == cache_add) {
				//
				// show a dialog and have the user input a URL
				//
				InputDialog idialog = new InputDialog(getShell(),UiPlugin.getString("URL_DIALOG_TITLE"),UiPlugin.getString("URL_DIALOG_MSG"),previous_url,null);
				if (idialog.open() == InputDialog.OK) {
					previous_url = idialog.getValue();

					boolean duplicate = false;
					
					String[] existing = cache_list.getItems();
					for (int i = 0; i < existing.length; i++) {
						if (previous_url.equals(existing[i])) {
							duplicate = true;
						}
					}
					
					if (!duplicate) {
						//
						// Fetch from previous_url and store as a new import
						//
						try {
							URL url = new URL(previous_url);
	
//							//set the proxy if we need to
//							if (proxy_use.getSelection()) {
//								System.setProperty("http.proxyHost",proxy_host.getText());
//								System.setProperty("http.proxyPort",proxy_port.getText());
//							}
//							
//							InputStream in = url.openStream();
//							String s = StreamUtils.readAllAsString(in);
							
							String s = new StandardWsdlImportResolver().fetchUrl(url,false,false);
							
							String[] items = cache_list.getItems();
							String[] newitems = new String[items.length+1];
							System.arraycopy(items,0,newitems,0,items.length);
							newitems[items.length] = previous_url;
							
							//store this to the store immediately
							IPreferenceStore store = getPreferenceStore();
							
							store.setValue(PreferenceConstants.CACHE_SIZE,newitems.length);
							store.setValue(PreferenceConstants.CACHE_LOCATION+(newitems.length-1),previous_url);
							store.setValue(PreferenceConstants.CACHE_CONTENT+(newitems.length-1),s);
							
							showCacheContent(newitems.length-1);
	
							cache_list.setItems(newitems);
							
						} catch (Exception e) {
							UiPlugin.DBG.logVisibleError(e,UiPlugin.getString("ERROR_CACHE_FETCH"),true);
						}
					} else {
						MessageBox mbox = new MessageBox(getShell(),SWT.OK|SWT.ICON_ERROR);
						mbox.setText(UiPlugin.getString("URL_DIALOG_ERROR"));
						mbox.setMessage(UiPlugin.getString("ERROR_CACHE_DUP"));
						mbox.open();
					}
					
				}
			}
		}
		
		//
		// SelectionListener
		//
		public void widgetSelected(SelectionEvent e) {
			widgetSelected(e.getSource());
		}
		public void widgetDefaultSelected(SelectionEvent e) {
			widgetSelected(e.getSource());
		}

}