/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.stp.b2j.core.publicapi.example;

import java.io.File;

import org.eclipse.stp.b2j.core.publicapi.engine.BPELEngineListener;
import org.eclipse.stp.b2j.core.publicapi.engine.IndependantBPELEngine;
import org.eclipse.stp.b2j.core.publicapi.program.BPELProgram;
import org.eclipse.stp.b2j.core.publicapi.program.IndependantBPELProgram;
import org.eclipse.stp.b2j.core.publicapi.transport.session.SessionAddress;

public class IndependantAPIExample {

	/**
	 * This method demonstrates how to run a BPEL process on a local engine
	 * The process will run within this JVM and no external daemon process is required
	 * @param bpel_src_file the BPEL source file
	 * @throws Exception if an error occurs while setting up or running this BPEL process
	 */
	private static void runLocalBpelProgram(File bpel_src_file) throws Exception {
		
		//Create a new BPEL program
		BPELProgram program = new IndependantBPELProgram(bpel_src_file);		
		
		//Create a new BPEL engine
		IndependantBPELEngine engine = new IndependantBPELEngine(program);
		
		//Run the program
//		engine.runProgram();
		engine.runProgram(new PrintoutEngineListener(), false, true, true, true);
		engine.waitForProgramCompletion();
	}
	
	/**
	 * This method demonstrates how to run a BPEL process distributed over a number of hosts
	 * The process will run spread out over the specified hosts so a b2j engine
	 * daemon must be running on all of the given hosts
	 * @param bpel_src_file the BPEL source file
	 * @param hosts an array of hostnames or IPs to distribute the BPEL process over
	 * @throws Exception if an error occurs while setting up or running this BPEL process
	 */
	private static void runDistributedBpelProgram(File bpel_src_file, String[] hosts) throws Exception {
		
		//Create a new BPEL program
		BPELProgram program = new IndependantBPELProgram(bpel_src_file);		
		
		runDistributedBpelProgram(program,hosts);
	}

	private static void runDistributedBpelProgram(BPELProgram program, String[] hosts) throws Exception {
		
		//Create an address to represent the workbench -> coordinator daemon link
		SessionAddress coord_daemon_address = new SessionAddress(
				"localhost",
				SessionAddress.TRANSPORT_PORT_ANY,
				SessionAddress.TRANSPORT_PORT_ANY,
				hosts[0],
				11000,
				11000
				);
		//HTTP, not HTTPS
		coord_daemon_address.setRequiresEncryption(false);
		//no password
		coord_daemon_address.setRequiresPassword(false);

		//Create an address to represent the workbench -> coordinator host link
		SessionAddress coord_address = new SessionAddress(
				"localhost",
				SessionAddress.TRANSPORT_PORT_ANY,
				SessionAddress.TRANSPORT_PORT_ANY,
				hosts[0],
				SessionAddress.TRANSPORT_PORT_ANY,
				SessionAddress.TRANSPORT_PORT_ANY
				);
		
		//Set the coordinator host
		program.setCoordinatorHost(coord_daemon_address,coord_address);
		
		//Add worker hosts
		for (int i = 0; i < hosts.length; i++) {

			//Create an address to represent the coordinator host -> work host [i] daemon link
			SessionAddress worker_daemon_address = new SessionAddress(
					"localhost",
					SessionAddress.TRANSPORT_PORT_ANY,
					SessionAddress.TRANSPORT_PORT_ANY,
					hosts[i],
					11000,
					11000
					);
			//HTTP, not HTTPS
			worker_daemon_address.setRequiresEncryption(false);
			//no password
			worker_daemon_address.setRequiresPassword(false);

			//Create an address to represent the coordinator host -> worker host [i] link
			SessionAddress worker_address = new SessionAddress(
					"localhost",
					SessionAddress.TRANSPORT_PORT_ANY,
					SessionAddress.TRANSPORT_PORT_ANY,
					hosts[i],
					SessionAddress.TRANSPORT_PORT_ANY,
					SessionAddress.TRANSPORT_PORT_ANY
					);
			
			//Add the worker host
			program.addWorkerHost(worker_daemon_address,worker_address);
		}
		
		//Create a new BPEL engine
		IndependantBPELEngine engine = new IndependantBPELEngine(program);
		
		//Run the program
//		engine.runProgram();
		engine.runProgram(new PrintoutEngineListener(), false, true, false, true);
		engine.waitForProgramCompletion();
	}	
	
	public static void main(String[] args) throws Exception {
		BPELProgram program = new IndependantBPELProgram(new File(args[0]));		
		runDistributedBpelProgram(program,new String[]{"localhost"});
		for (int i = 0; i < 1000; i++) {
			IndependantBPELEngine engine = new IndependantBPELEngine(program);
			engine.markProgramCompiled();
			engine.runProgram(new PrintoutEngineListener(), false, true, false, true);
			engine.waitForProgramCompletion();
		}
//		runLocalBpelProgram(new File(args[0]));
	}
	
	private static class PrintoutEngineListener implements BPELEngineListener {

		public void printInfo(String s) {
			System.out.println("INFO: "+s);
		}

		public void printDebug(String s) {
			System.out.println("DEBUG: "+s);
		}

		public void printEngineInfo(String s) {
			System.out.println("ENGINE_INFO: "+s);
		}

		public void printEngineDebug(String s) {
			System.out.println("ENGINE_DEBUG: "+s);
		}
		
	}
	
}