/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.ui.xef.editor;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;


import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.stp.ui.xef.schema.IContextProvider;
import org.eclipse.stp.xef.ISchemaProvider;
import org.eclipse.stp.xef.IXMLProvider;
import org.eclipse.stp.xef.util.InputStreamHelper;
import org.eclipse.ui.IPersistableElement;
import org.eclipse.ui.part.FileEditorInput;

/** An instance of this class can be provided to 
 * {@link org.eclipse.ui.ide.IDE#openEditor(org.eclipse.ui.IWorkbenchPage, org.eclipse.ui.IEditorInput, String)}
 * to get Eclipse to open the XEF editor.
 */
public class XMLProviderEditorInput extends FileEditorInput {
	private static volatile int counter = 0; 
    private final String title;
    private final IXMLProvider provider;
    private final ISchemaProvider schemaProvider;
    private final IContextProvider contextProvider;
    
    /**
     * Create an EditorInput for a XEF editor. This EditorInput can then be used as an argument
     * to {@link org.eclipse.ui.ide.IDE#openEditor(org.eclipse.ui.IWorkbenchPage, org.eclipse.ui.IEditorInput, String)}.
     * @param t The title of the editor.
     * @param p The XML Provider used to read/write the XML from.
     * @param sp The Schema Provider used to present a schema catalog from.
     * @param cp A contextprovider that can be used to obtain context information.
     */
    public XMLProviderEditorInput(String t, IProject prj, IXMLProvider p, ISchemaProvider sp, IContextProvider cp) throws IOException, CoreException {
        super(createTempFile(prj, p));
        title = t;
        provider = p;
        schemaProvider = sp;
        contextProvider = cp;
    }
    
    private static IFile createTempFile(IProject project, IXMLProvider provider) throws IOException, CoreException {
      IFile tempfile = project.getFile(".xml.tmp" + System.currentTimeMillis() + "_" + counter++);
      tempfile.getLocation().toFile().deleteOnExit(); 
      tempfile.create(new ByteArrayInputStream(provider.getXML().getBytes()), true, new NullProgressMonitor());
      project.refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor()); // TODO restrict the depth
      return tempfile;
    }
    
    /**
     * Return the context provider. The context provider is used to obtain context information
     * that is external to the schema on which the editor is based, such as possible values for
     * a particular field, if these values cannot be specified in the schema.
     * @return The contextprovider.
     */
    public IContextProvider getContextProvider() {
        return contextProvider;
    }
    
    /**
     * Returns the schema provider. The schema provider can provide a catalog of XML-Schemas that new sub-elements
     * can be created for through the editor. 
     * @return The schema provider, or <tt>null</tt> if no schema provider is present.
     */
    public ISchemaProvider getSchemaProvider() {
        return schemaProvider;
    }
    
    /**
     * Returns the IXMLProvider associated with this editor input.
     * @return The IXMLProvider.
     */
    public IXMLProvider getXMLProvider() {
        return provider;
    }
    
    /**
     * We don't want this editor to persist across sessions.
     */
    @Override
    public IPersistableElement getPersistable() {
        return null;
    }

    /** 
     * Returns the title of the editor.
     * @return The title.
     */
    public String getTitle() {
        return title;
    }

    public void propagate() throws IOException, CoreException {
        InputStream is = getFile().getContents();
        try {
            provider.setXML(new String(InputStreamHelper.drain(is)));
            getFile().refreshLocal(IResource.DEPTH_ZERO, new NullProgressMonitor());
        } finally {
            is.close();
        }
    }
    
    @Override
    public boolean equals(Object other) {
        // A correct implementation of equals ensure that IDE.openEditor does 
        // not open multiple editors for equivalent IEditorInput instances.
        //
        if (this == other) {
            return true;
        } 
        
        if (other instanceof XMLProviderEditorInput) {
		XMLProviderEditorInput otherInput = (XMLProviderEditorInput)other;
		
		return getXMLProvider().equals(otherInput.getXMLProvider());	
        }

        return false;
    }

}
