/*******************************************************************************
 * Copyright (c) 2006-2007 IONA Technologies.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IONA Technologies - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.ui.xef.editor;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.window.Window;
import org.eclipse.stp.ui.xef.PluginResources;
import org.eclipse.stp.ui.xef.XefPlugin;
import org.eclipse.stp.ui.xef.help.XefHelpView;
import org.eclipse.stp.ui.xef.schema.AnnotatedElement;
import org.eclipse.stp.ui.xef.schema.AnyElement;
import org.eclipse.stp.ui.xef.schema.SchemaAttribute;
import org.eclipse.stp.ui.xef.schema.SchemaElement;
import org.eclipse.stp.ui.xef.schema.WidgetType;
import org.eclipse.stp.xef.XMLInstanceElement;
import org.eclipse.stp.xef.XefConstants;
import org.eclipse.stp.xef.util.QNameHelper;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.swt.widgets.TypedListener;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.forms.IDetailsPage;
import org.eclipse.ui.forms.IFormPart;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.forms.widgets.TableWrapData;
import org.eclipse.ui.forms.widgets.TableWrapLayout;
import org.jdom.Attribute;
import org.jdom.input.SAXBuilder;

public class XefDetailsPage implements IDetailsPage, IXefDetailsPage {
    private static final String DEFAULT_CAT_NAME = "Main";
    private static final Pattern WHITESPACE_PATTERN = Pattern.compile("\\s+");
    private static final Pattern NCNAME_PATTERN = Pattern.compile("([a-zA-Z_])([a-zA-Z0-9-_:\\.])*");

    static final String DEFAULT_TOOLITEM_BUTTON = "Default Button";
    static final String ACCESSOR = "Control Accessor";
    static final String CONTROLS = "The actual controls";
    static final String FIXED = "Fixed";
    static final String LABEL = "Label";
    static final String PREV_VALUE = "Previous value";
    static final String UNITS = "Units";
    static final String NS_MAPPINGS = "Namespace Mappings";
    static boolean interactive = true; // used for testing    
    
    final XMLInstanceElement element;
    final Map<Control, Object> values = new HashMap<Control, Object>();
    boolean dirty = false;

    private final XefEditor editor;    
    private IManagedForm managedForm;
    private Listener modifyListener;

    public XefDetailsPage(XefEditor ed, XMLInstanceElement el) {
        editor = ed;
        element = el;
    }

    public void initialize(IManagedForm form) {
        managedForm = form;
    }

    public void createContents(Composite parent) {
        modifyListener = new Listener() {
            public void handleEvent(Event event) {
                if (event.widget instanceof Control) {
                    markDirty((Control) event.widget);
                }
            }            
        };
        
        TableWrapLayout layout = new TableWrapLayout();
        layout.topMargin = 5;
        layout.leftMargin = 5;
        layout.rightMargin = 2;
        layout.bottomMargin = 2;
        parent.setLayout(layout);
        
        FormToolkit toolkit = managedForm.getToolkit();
        Section s1 = toolkit.createSection(parent, Section.DESCRIPTION | Section.TITLE_BAR);
        s1.marginWidth = 10;
        s1.setText("Policy Values");
        s1.setDescription("Element: " + XefEditMasterDetailsBlock.getLabel(element));
        s1.setLayout(new GridLayout(1, false));
        TableWrapData td = new TableWrapData(TableWrapData.FILL, TableWrapData.TOP);
        td.grabHorizontal = true;
        s1.setLayoutData(td);

        List<String> categoryNames = getFurtherSections();
        Map<String, Composite> categories = new LinkedHashMap<String, Composite>();

        Composite holder = toolkit.createComposite(s1);
        GridLayout glayout = new GridLayout();
        glayout.marginWidth = 0;
        glayout.marginHeight = 0;
        glayout.numColumns = 1;
        holder.setLayout(glayout);
        
        Composite mainContents = createMainAttributeSection(toolkit, holder);
        categories.put(DEFAULT_CAT_NAME, mainContents);
        
        for (String categoryName : categoryNames) {
            Section ec = toolkit.createSection(holder, Section.TWISTIE | Section.CLIENT_INDENT);
            ec.setText(categoryName);
            
            Composite contents = toolkit.createComposite(ec);
            GridLayout gl = new GridLayout();
            gl.marginWidth = 0;
            gl.marginHeight = 0;
            gl.numColumns = 4;
            contents.setLayout(gl);
          
            ec.setClient(contents);
            categories.put(categoryName, contents);
        }
        s1.setClient(holder);

        createAttributeControls(toolkit, categories);        
        createAnyContentControls(toolkit, mainContents);
        createTextAreaControls(toolkit, mainContents);
//        for (Composite c : categories.values()) {
//		toolkit.paintBordersFor(c);
//        }
    }

    private Composite createMainAttributeSection(FormToolkit toolkit, Composite composite) {
        Composite contents = toolkit.createComposite(composite);
        GridLayout glayout = new GridLayout();
        glayout.marginWidth = 0;
        glayout.marginHeight = 0;
        glayout.numColumns = 4;
        contents.setLayout(glayout);

        // this guy is needed to make sure all the lines at the top get drawn
        Composite sep = new Composite(contents, SWT.NONE);
        sep.setBackground(contents.getBackground());
        sep.setLayout(new RowLayout());
        GridData gd = new GridData();
        gd.horizontalSpan = 4;
        sep.setLayoutData(gd);
        return contents;
    }

    private void createTextAreaControls(FormToolkit toolkit, Composite parent) {
        if (!element.getTemplate().hasText()) {
            return;            
        }
        
        if (element.getTemplate().isSequenceOfAny()) {
            return;
        }
        
        
        // if it's an any...
        // mark it nicely with a display name
        
        GridData gd = new GridData();
        gd.horizontalSpan = 4;
        Label l = toolkit.createLabel(parent, "Value:");
        l.setLayoutData(gd);
        
        gd = new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL | GridData.GRAB_VERTICAL);
        gd.horizontalSpan = 3;
        gd.heightHint = 70;
        gd.horizontalIndent = 5;
        gd.verticalIndent = 5;
        
        Text t = toolkit.createText(parent, "", SWT.MULTI | SWT.WRAP);
        t.setLayoutData(gd);
        t.setData(ACCESSOR, new TextControlAccessor(t));
        t.setData(CONTROLS, new Control[] {t});
        t.setData(LABEL, l);
        t.addListener(SWT.Modify, modifyListener);
        values.put(t, element.getTemplate());

        toolkit.createLabel(parent, ""); // forces the correct drawing of a line
        toolkit.createLabel(parent, ""); // forces the correct drawing of a line
    }
    
    private void createAnyContentControls(FormToolkit toolkit, Composite parent) {
        if (!element.getTemplate().isSequenceOfAny()) {
            return;
        }
        
        // if it's an any...
        // mark it nicely with a display name
        AnyElement anyElement = element.getTemplate().getAnySequenceElement();
        
        GridData gd = new GridData();
        gd.horizontalSpan = 4;
        Label l = toolkit.createLabel(parent, anyElement.getDisplayName() + ":");
        String toolTipText = normalizeTooltipText(anyElement);
        l.setToolTipText(toolTipText);
        l.setLayoutData(gd);
        
        gd = new GridData(GridData.FILL_BOTH | GridData.GRAB_HORIZONTAL | GridData.GRAB_VERTICAL);
        gd.horizontalSpan = 3;
        gd.widthHint = 250;
        gd.heightHint = 250;
        gd.horizontalIndent = 5;
        gd.verticalIndent = 5;
        
        final Text t = toolkit.createText(parent, "", SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.READ_ONLY);
        t.setLayoutData(gd);
        t.setToolTipText(toolTipText);
        t.setEnabled(false);
        TextControlAccessor tca = new TextControlAccessor(t);
        t.setData(ACCESSOR, tca);
        t.setData(CONTROLS, new Control[] {t});
        t.setData(LABEL, l);
        values.put(t, element.getTemplate());
        
        tca.setValue(element.getEmbeddedXML());
        t.addListener(SWT.Modify, modifyListener);
        t.addFocusListener(new HelpViewSynchingFocusListener(anyElement));
        
        Button editBtn = toolkit.createButton(parent, "...", SWT.PUSH);
        gd = new GridData(GridData.VERTICAL_ALIGN_BEGINNING);
        gd.verticalIndent = 5;
        editBtn.setLayoutData(gd);
        editBtn.setToolTipText("Click to edit the XML content");
        editBtn.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent arg0) {
                XMLEditDialog dlg = new XMLEditDialog(editor.getSite().getShell(), t.getText(), element.getTemplate().getName());
                if (dlg.open() == Window.OK) {
                    t.setText(dlg.getXML());
                }
            }            
        });
    }
    
    private List<String> getFurtherSections() {
        Set<String> s = new LinkedHashSet<String>();
        
        for (SchemaAttribute attr : element.getTemplate().getAttributes()) {
            String category = attr.getCategory();
            if (category != null) {
                s.add(category);
            }
        }
        return new ArrayList<String>(s);
    }

    private void createAttributeControls(FormToolkit toolkit, Map<String, Composite> categories) {

        for (SchemaAttribute attr : element.getTemplate().getAttributes()) {
            try {
				String category = attr.getCategory();
				Composite parent = categories.get(category != null ? category : DEFAULT_CAT_NAME);
				if (parent == null) {
				    // should not happen
				    continue;
				}
				
				Label label = toolkit.createLabel(parent, attr.getDisplayName() + ":");
				label.setToolTipText(normalizeTooltipText(attr));
				label.setLayoutData(new GridData(GridData.GRAB_HORIZONTAL));
				
				Control control = createControl(attr, parent, toolkit);
				control.setData(LABEL, label);
				
				values.put(control, attr);
				
				if (attr.getWidget() == WidgetType.PASSWORD) {
				    addSecondaryPasswordControl(attr, control, parent, toolkit);
				}
			} catch (CoreException e) {
				e.printStackTrace();
				XefPlugin.getDefault().getLog().log(e.getStatus());
			}
        }
    }

    private void addSecondaryPasswordControl(SchemaAttribute attr, Control pwdCtrl1,
        Composite parent, FormToolkit toolkit) throws CoreException {
        Text pwText1 = null;
        if (pwdCtrl1 instanceof Text) {
            pwText1 = (Text) pwdCtrl1;
        }
        
        PasswordControlAccessor pca = null;         
        if (pwText1.getData(ACCESSOR) instanceof PasswordControlAccessor) {
            pca = (PasswordControlAccessor) pwText1.getData(ACCESSOR); 
        }
        
        toolkit.createLabel(parent, "Confirm:");
        Control pwdCtrl2 = createControlWidgets(attr, parent, toolkit);        
        toolkit.createLabel(parent, "");
        toolkit.createLabel(parent, "");  
        
        Text pwText2 = null;
        if (pwdCtrl2 instanceof Text) {
            pwText2 = (Text) pwdCtrl2;
        }
        
        if (pwText1 == null || pwText2 == null || pca == null) {
            MessageDialog.openError(editor.getSite().getShell(),  
                "Could not create password control", 
                "An unexpected error occurred when creating a password control");
            return;
        }
        
        pca.setSecondaryText(pwText2);
        pwText2.setText(pwText1.getText());
        Control [] controls = (Control[]) pwdCtrl1.getData(CONTROLS);
        Control [] newControls = new Control[controls.length + 1];
        System.arraycopy(controls, 0, newControls, 0, controls.length);
        newControls[controls.length] = pwdCtrl2;
        pwdCtrl1.setData(CONTROLS, newControls);
        
        final Text pwt1 = pwText1;
        final Text pwt2 = pwText2;

        pwt1.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent e) {
                pwt2.setFocus();
                editor.getSite().getShell().getDisplay().asyncExec(new Runnable() {
                    public void run() {
                        pwt2.setFocus();
                    }                        
                });                
            }            
        });
        
//        pwt2.addFocusListener(new FocusAdapter() {
//            @Override
//            public void focusLost(FocusEvent e) {
//                validatePasswordField(pwt1);
//            }            
//        });
    }

    private Control createControl(final SchemaAttribute attr, Composite container, FormToolkit toolkit) throws CoreException {        
        String curValue = element.getAttribute(attr.getName());
        Control logicalControl = createControlWidgets(attr, container, toolkit);        
        
        if (curValue != null) {
            ControlAccessor accessor = (ControlAccessor) logicalControl.getData(ACCESSOR);
            accessor.setValue(curValue);
            logicalControl.setData(PREV_VALUE, curValue);
        }
        
        if (attr.getFixed() != null) {
            setFixedValue(logicalControl, attr.getFixed());
        }
        
        String units = attr.getUnits() == null ? "" : "(" + attr.getUnits() + ")";
        Label unitLabel = toolkit.createLabel(container, units);
        logicalControl.setData(UNITS, unitLabel);
        
        if (!attr.getRequired() && attr.getFixed() == null) {
            addDefaultButton(toolkit, container, logicalControl, curValue, attr.getDefault());
        } else {
            addEmptyButton(toolkit, container);
        }
        if (attr.getRequired()) {
            if (curValue == null && attr.getExample() != null) {
                ((ControlAccessor) logicalControl.getData(ACCESSOR)).setValue(attr.getExample());
                markDirty(logicalControl);
            }
        } 
        
        Control [] controls = (Control []) logicalControl.getData(CONTROLS);
        for (Control control : controls) {
            if (control instanceof Button) {
                control.addListener(SWT.Selection, modifyListener);
            } else {
                control.addListener(SWT.Modify, modifyListener);
            }
            
            if (attr.getPattern() != null) {
                Listener verifyListener = new TypedListener(
                        new TextFieldVerifyListener(attr.getPattern()));
                control.addListener(SWT.Verify, verifyListener);
            }
             
             if (attr.getContextXPath() != null) {
                 String xpath = attr.getContextXPath();
                 addContextListener(control, xpath);
             }
            
            if (control.getToolTipText() == null) {
                control.setToolTipText(normalizeTooltipText(attr));
            }
            
            control.addFocusListener(new HelpViewSynchingFocusListener(attr));
        }

        return logicalControl;
    }
    
    private void addDefaultButton(final FormToolkit toolkit, final Composite container, 
            final Control logicalControl, String value, String def) {
        final String defaultValue = def == null ? "" : def;
        
        final Control [] controls = (Control []) logicalControl.getData(CONTROLS);
        final ControlAccessor accessor = (ControlAccessor) logicalControl.getData(ACCESSOR);
        final ToolItem defaultToolitem = createDefaultButton(toolkit, container);
        
        final Color activeColor = controls[0].getForeground();
        final Color inActiveColor = controls[0].getDisplay().getSystemColor(SWT.COLOR_GRAY);
        logicalControl.setData(DEFAULT_TOOLITEM_BUTTON, defaultToolitem);
        
        if (value == null || value.length() == 0) {
            for (Control control : controls) {
                control.setForeground(inActiveColor);
            }
            accessor.setValue(defaultValue);
            defaultToolitem.setSelection(true);
            defaultToolitem.setImage(XefPlugin.getDefault().getImageRegistry().get(XefPlugin.IMG_DEFAULT));
        } else {
            defaultToolitem.setSelection(false);
            defaultToolitem.setImage(XefPlugin.getDefault().getImageRegistry().get(XefPlugin.IMG_DEFAULT_OFF));
        }
        
        defaultToolitem.addSelectionListener(new SelectionListener() {
            public void widgetDefaultSelected(SelectionEvent e) {
                widgetSelected(e);
            }
            
            public void widgetSelected(SelectionEvent e) {
                for (Control control : controls) {
                    control.setForeground(defaultToolitem.getSelection() ? inActiveColor : activeColor);
                }
                
                if (defaultToolitem.getSelection()) {
                    String curVal = accessor.getValue();
                    accessor.setValue(defaultValue);    
                    logicalControl.setData(PREV_VALUE, curVal);
                    defaultToolitem.setImage(XefPlugin.getDefault().getImageRegistry().get(
                            XefPlugin.IMG_DEFAULT));
                    editor.policyEditPage.block.getViewer().getTree().setFocus(); // move the focus away
                } else {
                    Object prev = logicalControl.getData(PREV_VALUE);
                    if (prev != null) {
                        accessor.setValue(prev.toString());
                    } else {
                        accessor.setValue("");
                    }
                    defaultToolitem.setImage(XefPlugin.getDefault().getImageRegistry().get(
                            XefPlugin.IMG_DEFAULT_OFF));
                }
            }                
        });
        
        for (Control control : controls) {
            control.addMouseListener(new MouseAdapter() {
                public void mouseDown(MouseEvent e) {
                    if (defaultToolitem.getSelection()) {
                        defaultToolitem.setSelection(false);
                        defaultToolitem.notifyListeners(SWT.Selection, new Event());
                    }
                }
            });
            
            control.addFocusListener(new FocusAdapter() {
                public void focusGained(FocusEvent e) {
                    if (defaultToolitem.getSelection()) {
                        defaultToolitem.setSelection(false);
                        defaultToolitem.notifyListeners(SWT.Selection, new Event());
                    }
                }
            });
        }
    }

    private ToolItem createDefaultButton(final FormToolkit toolkit, final Composite container) {
        ToolBar tb = new ToolBar(container, SWT.FLAT);
        tb.setBackground(container.getBackground());
        tb.setForeground(container.getForeground());
        
        ToolItem ti = new ToolItem(tb, SWT.CHECK);
//        ti.setImage(XefPlugin.getDefault().getImageRegistry().get(XefPlugin.IMG_DEFAULT));
        ti.setToolTipText("Toggle default value");
        
        return ti;
    }

    private void addEmptyButton(FormToolkit toolkit, Composite container) {
        ToolItem btn = createDefaultButton(toolkit, container);
        btn.getParent().setVisible(false);
    }
    
    private void setFixedValue(Control logicalControl, String value) {
        final ControlAccessor accessor = (ControlAccessor) logicalControl.getData(ACCESSOR);
        accessor.setValue(value);
        logicalControl.setData(FIXED, FIXED);
    }

    private Control createControlWidgets(SchemaAttribute attr, Composite parent, FormToolkit toolkit) throws CoreException {
        Control logicalControl = createControlWidgets2(attr, parent, toolkit);     
        
        GridData gd = new GridData();
        if (logicalControl instanceof Spinner) {
		gd.widthHint = 40;
		gd.horizontalAlignment = SWT.END;
		gd.grabExcessHorizontalSpace = false;
        } else if (logicalControl instanceof Text && (((Text) logicalControl).getStyle() & SWT.MULTI) > 0) {
            // The multiline text is rendered differently, and needs more space.
            // It also needs to be stretched out so that even if there is no text
            // in it, it will still be usefully visible.
            gd.horizontalSpan = 2;
            gd.heightHint = 150;
            gd.widthHint = 400;
            gd.horizontalAlignment = SWT.FILL;
            gd.grabExcessVerticalSpace = false;
        } else {
		    gd.minimumWidth = 150;
		gd.horizontalAlignment = SWT.FILL;
		gd.grabExcessHorizontalSpace = true;
        }
	    logicalControl.setLayoutData(gd);    
        
	    return logicalControl;
    }
    
    private Control createControlWidgets2(SchemaAttribute attr, Composite parent, FormToolkit toolkit) throws CoreException {
        boolean enabled = (attr.getFixed() == null);
        switch (attr.getValue().getType()) {
        case ENUMERATION:
            String[] allowed = attr.getValue().getAllowedValues().toArray(new String [] {});
            return createCombo(attr, allowed, enabled, parent, toolkit);
        case BOOLEAN:
            if (attr.getWidget() == WidgetType.CHECK) {
                Button booleanCheckButton = toolkit.createButton(parent, "", SWT.CHECK);
                booleanCheckButton.setEnabled(enabled);
                booleanCheckButton.setData(ACCESSOR, new BooleanCheckControlAccessor(booleanCheckButton));
                booleanCheckButton.setData(CONTROLS, new Control [] {booleanCheckButton});
                return booleanCheckButton;
            } else {
                Composite container = toolkit.createComposite(parent);
                container.setLayout(new RowLayout(SWT.HORIZONTAL));
                Button trueButton = toolkit.createButton(container, "True", SWT.RADIO);
                Button falseButton = toolkit.createButton(container, "False", SWT.RADIO);            
                trueButton.setEnabled(enabled);
                falseButton.setEnabled(enabled);
                container.setEnabled(enabled);
                toolkit.paintBordersFor(container);
    
                container.setData(Boolean.TRUE.toString(), trueButton);
                container.setData(Boolean.FALSE.toString(), falseButton);
                container.setData(ACCESSOR, new BooleanControlAcccessor(container));
                container.setData(CONTROLS, new Control[] {trueButton, falseButton});
                return container;                
            }
        case INTEGER:
        case SHORT:
        case BYTE:
        case UINT:
        case USHORT:
        case UBYTE:
//        case NEGATIVE_INT: // not yet supported by spinner
        case POSITIVE_INT:
        case NON_NEGATIVE_INT:
//        case NON_POSITIVE_INT: // not yet supported by spinner
		return createSpinnerControl(attr, enabled, parent, toolkit);
        case QNAME:
            return createQNameControl(attr, enabled, parent, toolkit);
        case NCNAME:
            attr.setPattern(NCNAME_PATTERN.toString());
        default:
		if (attr.getWidget() == WidgetType.PASSWORD) {
			return createPasswordControl(attr, parent, toolkit);
		}
        
            if (editor.getEditorInput() instanceof XMLProviderEditorInput) {
                XMLProviderEditorInput editorInput = (XMLProviderEditorInput)editor.getEditorInput();
                if (editorInput.getContextProvider() != null) {
                    String[] values = attr.getContextValues(editorInput.getContextProvider(),
                                            element.getDependencyValue(attr.getContextXPath()));

                    if (values != null) {
                       return createCombo(attr, values, enabled, parent, toolkit);
                    }
                }
            }
            
            boolean multiline = (attr.getWidget() == WidgetType.MULTILINE);

            return createPlainTextControl(attr, enabled, multiline, parent, toolkit);
        }
    }

	public void dispose() {}
    
    void markDirty(Control control) {
        setDirty(true);
    }

    public boolean isDirty() {
        return dirty;
    }
    
    public void setDirty(boolean value) {
        dirty = value;
        managedForm.dirtyStateChanged();
        if (dirty) {
            editor.fireSaveNeeded();
        }
    }

    public void commit(boolean onSave) {
        for (Control control : values.keySet()) {
            storeControlInJDomElement(control);
        }
    }

	@SuppressWarnings("unchecked")
	private void storeControlInJDomElement(Control control) {
        Object schemaType = values.get(control);
        if (schemaType instanceof SchemaAttribute) {
            String name = ((SchemaAttribute) schemaType).getName();
            ToolItem defBtn = (ToolItem) control.getData(DEFAULT_TOOLITEM_BUTTON);
            
            if (defBtn != null && defBtn.getSelection() || control.getData(FIXED) != null) {
                // remove the attribute if the default is to be used or if it has a fixed value
                element.removeAttribute(name);
            } else {
                element.setAttribute(name, getValue(control));
            }
        } else if (schemaType instanceof SchemaElement) { // This one is used for Text elements
        	if (((SchemaElement) schemaType).isSequenceOfAny()) {
        		element.setEmbeddedXML(getValue(control));
        	} else {
        		element.setText(getValue(control));
        	}
        }
    }

    private String getValue(Control control) {
        ControlAccessor accessor = (ControlAccessor) control.getData(ACCESSOR);
        return accessor.getValue();
    }
    
    public Map<Control, Object> getValues() {
	return values;
    }

    private void updateValues(Control control, String filterValue) {
        if (!(values.get(control) instanceof SchemaAttribute)
                || !(control instanceof Combo)) {
            return;
        }
        
        boolean needsUpdating = false;
        Combo combo = (Combo) control;
        String[] oldValues = combo.getItems();
        String oldValue = ((ControlAccessor) control.getData(ACCESSOR)).getValue();
        SchemaAttribute attr = (SchemaAttribute) values.get(control);
        if (editor.getEditorInput() instanceof XMLProviderEditorInput) {
            XMLProviderEditorInput editorInput = (XMLProviderEditorInput)editor.getEditorInput();
            if (editorInput.getContextProvider() != null) {
                String[] newValues = attr.getContextValues(editorInput.getContextProvider(),
                                                           filterValue);

                if (newValues != null) {
                    if (newValues.length != oldValues.length) {
                        needsUpdating = true;
                    } else {
                        if (combo.getData(ACCESSOR) instanceof QNameControlAccessor) {
                            newValues = ((QNameControlAccessor) combo.getData(ACCESSOR)).getQNameValues(newValues);
                        }
                        for (int i = 0; i < newValues.length; i++) {
                            if (!(newValues[i].equals(oldValues[i]))) {
                                needsUpdating = true;
                                break;
                            }
                        }
                    }

                    if (needsUpdating) {
                        combo.setItems(newValues);
                        combo.select(combo.indexOf(oldValue));
                        combo.notifyListeners(SWT.Selection, new Event());
                    }
                }
            }
        }                    
    }
    
    public boolean setFormInput(Object input) {
        // TODO Auto-generated method stub
        return false;
    }

    public void setFocus() {
        // TODO Auto-generated method stub
        
    }

    public boolean isStale() {
        // TODO Auto-generated method stub
        return false;
    }

    public void refresh() {
        for (Control control : values.keySet()) {
            Object object = values.get(control);
            if (object instanceof SchemaAttribute) {
                SchemaAttribute attr = (SchemaAttribute) object;
                if (attr.getContextXPath() != null) {
                    String filter = element.getDependencyValue(attr.getContextXPath());
                    updateValues(control, filter);
                }
            }
        }
//        String n = selection == null ? "" : selection.toString();
//        formText.setText("Policy Name: " + n, false, false);
//        
////        Composite parent = values.getParent();
////        values.dispose();
////        values = new Composite(parent, SWT.NONE);
//        new Label(values, SWT.NONE).setText("Added this label " + System.currentTimeMillis());
//        values.getParent().changed(new Control[] {values});
//        values.redraw();
//        values.getParent().redraw();
//        values.redraw();
    } 

    public void selectionChanged(IFormPart part, ISelection sel) {
        refresh();
//        IStructuredSelection ssel = (IStructuredSelection) sel;
//        if (ssel.size() == 1) {
//            selection = (PolicyTemplate) ssel.getFirstElement();
//        } else { 
//            selection  = null;
//        }
//        refresh();
    }
    
    private Combo createCombo(SchemaAttribute attr, String[] values, boolean enabled,
                               Composite parent, FormToolkit toolkit) {
        int style = SWT.DROP_DOWN | SWT.FLAT;
        if (attr.getWidget() != WidgetType.READ_WRITE) {
            style = style | SWT.READ_ONLY;
        }
        
        Combo combo = new Combo(parent, style);
        toolkit.adapt(combo, true, true);
        combo.setEnabled(enabled);
        combo.setData(ACCESSOR, new ComboControlAccessor(combo));
        combo.setData(CONTROLS, new Control[] {combo});

        if (values != null && values.length > 0) {
            combo.setItems(values);
            combo.setData(PREV_VALUE, values[0]);
        }
//        combo.select(allowed.indexOf(value));
        
        toolkit.paintBordersFor(parent);
        return combo;
        
    }

	private Control createSpinnerControl(SchemaAttribute attr, boolean enabled, Composite parent, FormToolkit toolkit) {
		Spinner spinner = new Spinner(parent, SWT.WRAP);

		switch (attr.getValue().getType()) {
		case INTEGER:
			spinner.setMinimum(Integer.MIN_VALUE);
			spinner.setMaximum(Integer.MAX_VALUE);
			break;
		case SHORT:
			spinner.setMinimum(-32768);
			spinner.setMaximum(32767);
			break;
		case BYTE:
			spinner.setMinimum(-128);
			spinner.setMaximum(127);
			break;
		case UINT:
			spinner.setMinimum(0);
			spinner.setMaximum(Integer.MAX_VALUE);
			break;			
		case USHORT:
			spinner.setMinimum(0);
			spinner.setMaximum(65535);
			break;
		case UBYTE:
			spinner.setMinimum(0);
			spinner.setMaximum(255);
			break;
		case NEGATIVE_INT:
			spinner.setMinimum(Integer.MIN_VALUE);
			spinner.setMaximum(-1);
			break;
		case POSITIVE_INT:			
			spinner.setMinimum(1);
			spinner.setMaximum(Integer.MAX_VALUE);
			break;
		case NON_NEGATIVE_INT:
			spinner.setMinimum(0);
			spinner.setMaximum(Integer.MAX_VALUE);
			break;
		case NON_POSITIVE_INT:
			spinner.setMinimum(Integer.MIN_VALUE);
			spinner.setMaximum(0);
			break;
		}
		
		toolkit.adapt(spinner, true, true);
		spinner.setEnabled(enabled);
		spinner.setData(ACCESSOR, new SpinnerControlAccessor(spinner));
		spinner.setData(CONTROLS, new Control[] {spinner});
//		toolkit.paintBordersFor(spinner); TODO doesn't work
		return spinner;
	}	

    private Text createPasswordControl(SchemaAttribute attr, Composite parent, FormToolkit toolkit) throws CoreException {
		Text text = toolkit.createText(parent, "", SWT.SINGLE | SWT.PASSWORD);

		TextFilter filter = attr.getFilterId() != null ? FilterRegistry.getFilter(attr.getFilterId()) : SimpleHashFilter.SINGLETON;
		text.setData(ACCESSOR, new PasswordControlAccessor(text, filter));
		text.setData(CONTROLS, new Control[] {text});
		
		return text;
	}	

    private Combo createQNameControl(SchemaAttribute attr, boolean enabled,
                                     Composite parent, FormToolkit toolkit) throws CoreException {
        Composite qname = new Composite(parent, SWT.NONE);
        GridLayout gl = new GridLayout();
        gl.marginWidth = 0;
        gl.marginHeight = 0;
        gl.numColumns = 2;
        gl.horizontalSpacing = 0;
        qname.setLayout(gl);
        GridData gd = new GridData(GridData.FILL_BOTH | GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL);
        qname.setLayoutData(gd);

        int style = SWT.DROP_DOWN | SWT.FLAT | SWT.READ_ONLY;
        final Combo combo = new Combo(qname, style);
        toolkit.adapt(combo, true, true);
        combo.setData(FormToolkit.KEY_DRAW_BORDER, FormToolkit.TEXT_BORDER);
        combo.setEnabled(enabled);

        String[] values = new String[] {};
        if (editor.getEditorInput() instanceof XMLProviderEditorInput) {
            XMLProviderEditorInput editorInput = (XMLProviderEditorInput)editor.getEditorInput();
            if (editorInput.getContextProvider() != null) {
                values = attr.getContextValues(editorInput.getContextProvider(),
                                               element.getDependencyValue(attr.getContextXPath()));
            }
        }

        Map<String, String> namespaceMap = 
            QNameControlAccessor.createNamespaceMap(element.getNamespaceMap(), values);
        QNameControlAccessor accessor = new QNameControlAccessor(combo, namespaceMap);
        combo.setData(NS_MAPPINGS, namespaceMap);
        combo.setData(ACCESSOR, accessor);
        combo.setData(CONTROLS, new Control[] {combo});

        if (values != null && values.length > 0) {
            String[] displayValues = accessor.getQNameValues(values);
            combo.setItems(displayValues);
            combo.setData(PREV_VALUE, displayValues[0]);
        }

        combo.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent event) {
                Combo combo = (Combo) event.widget;
                QNameControlAccessor accessor = (QNameControlAccessor) combo.getData(ACCESSOR);
                updateQNameNamespaceDecl(combo);
                
                String selItem = combo.getText();
                combo.setData(PREV_VALUE, selItem);
                combo.setToolTipText(accessor.convertToFullName(selItem));
            }

        });

        // if READ_WRITE widget, add field editor button to control
        if (attr.getWidget() != WidgetType.READ_ONLY) {
            gl.horizontalSpacing = 5;
            qname.setLayout(gl);
            final AbstractFieldEditor fieldEditor =
                attr.getFieldEditorId() != null 
                        ? FieldEditorRegistry.getFieldEditor(attr.getFieldEditorId())
                        : QNameFieldEditor.SINGLETON;
            fieldEditor.setFieldData(namespaceMap);
            Button button = toolkit.createButton(qname, 
                    PluginResources.getString("fieldEditor.lbl"),
                    SWT.PUSH);
            button.addSelectionListener(new SelectionAdapter() {
                public void widgetSelected(SelectionEvent event) {
                    ControlAccessor accessor = (ControlAccessor) combo.getData(ACCESSOR);
                    String newQName = fieldEditor.getNewValue(accessor.getValue());
                    if (newQName != null) {
                        accessor.setValue(newQName);
                        combo.notifyListeners(SWT.Modify, new Event());
                    }
                }
            });
        }

//        combo.select(allowed.indexOf(value));
        
        toolkit.paintBordersFor(parent);
        return combo;
    }
    
    private Text createPlainTextControl(SchemaAttribute attr, boolean enabled, 
            boolean multiline, Composite parent, FormToolkit toolkit) throws CoreException {
        Composite textComp;
        if (attr.getFieldEditorId() != null) {
            textComp = new Composite(parent, SWT.NONE);
            GridLayout gl = new GridLayout();
            gl.marginWidth = 0;
            gl.marginHeight = 0;
            gl.numColumns = 2;
            textComp.setLayout(gl);
            GridData gd = new GridData(GridData.FILL_BOTH | GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_FILL);
            textComp.setLayoutData(gd);
        } else {
            textComp = parent;
        }
        
        int style;
        if (multiline) {
            style = SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER;
            // Add these empty controls to move the big text control onto the
            // next row of the grid layout so it has lots of space to itself.
            new Label(textComp, SWT.NONE);
            new Label(textComp, SWT.NONE);
            new Label(textComp, SWT.NONE);
        } else {
            style = SWT.SINGLE;
        }
		
        final Text text = toolkit.createText(textComp, "", style);
        
        if (multiline) {
            
        }
        
		text.setEnabled(enabled);
        
		text.setData(ACCESSOR, new TextControlAccessor(text));
		text.setData(CONTROLS, new Control[] {text});
        
        if (attr.getFieldEditorId() != null) {
            // create edit button now...
            final AbstractFieldEditor fieldEditor =
                        FieldEditorRegistry.getFieldEditor(attr.getFieldEditorId());
            if (editor.getEditorInput() instanceof XMLProviderEditorInput) {
                XMLProviderEditorInput editorInput = (XMLProviderEditorInput)editor.getEditorInput();
                if (editorInput.getContextProvider() != null) {
                    Object data = attr.getContextData(editorInput.getContextProvider());
                    fieldEditor.setFieldData(data);
                }
            }
            Button button = toolkit.createButton(textComp, 
                    PluginResources.getString("fieldEditor.lbl"),
                    SWT.PUSH);
            button.addSelectionListener(new SelectionAdapter() {
                public void widgetSelected(SelectionEvent event) {
                    ControlAccessor accessor = (ControlAccessor) text.getData(ACCESSOR);
                    String newExpr = fieldEditor.getNewValue(accessor.getValue());
                    if (newExpr != null) {
                        accessor.setValue(newExpr);
                        text.notifyListeners(SWT.Modify, new Event());
                    }
                }
            });            
        }
        toolkit.paintBordersFor(textComp);
		return text;
	}

    public void setFocus(Object target) {
        // Reverse lookup, could possibly be optimized
        for (Map.Entry<Control, Object> entry : values.entrySet()) {
            if (entry.getValue().equals(target)) {
                final Control control = entry.getKey();
                
                Runnable setFocus = new Runnable() {
                    public void run() {
                        control.setFocus();
                    }                        
                };
                if (interactive) {
                    editor.getSite().getShell().getDisplay().asyncExec(setFocus);
                } else {
                    setFocus.run();
                }
                return;
            }
        }
    }
    
    private String normalizeTooltipText(AnnotatedElement attr) {        
        if (attr == null) {
            return null;
        }
        
        String doc = attr.getDocShort();
        if (doc == null) {
            doc = attr.getDocumentation();
        }
        
        if (doc == null) {
            return null;
        }

        Matcher m = WHITESPACE_PATTERN.matcher(doc);
        return m.replaceAll(" ");
    }
    
    public boolean validate() {
        for (Map.Entry<Control, Object> entry : values.entrySet()) {
            ControlAccessor acc = (ControlAccessor) entry.getKey().getData(ACCESSOR);
            if (acc instanceof PasswordControlAccessor) {
                if (!validatePasswordField(entry.getKey())) {
                    return false;
                }
            }
            
            if (element.getTemplate().isSequenceOfAny()) {
                if (!validateSequenceOfAny(entry, acc)) {
                    return false;
                }
            }            
        }        
        
        return true;
    }

    private boolean validateSequenceOfAny(Map.Entry<Control, Object> entry, ControlAccessor acc) {
        if (entry.getValue() == element.getTemplate()) {
            String s = acc.getValue();
            if (s != null && s.length() > 0) {
                // Wrapping it in another tag is needed to support more than one root element
                String elName = element.getTemplate().getName();
                s = "<" + elName + ">" + s + "</" + elName + ">";
                
                SAXBuilder builder = new SAXBuilder();                    
                try {
                    builder.build(new ByteArrayInputStream(s.getBytes()));
                } catch (Exception e) {
                    if (interactive) {
                        MessageDialog.openError(editor.getSite().getShell(), 
                            "XML content incorrect", 
                            "The XML content of the field is not well-formed.\n\n\tMessage: " + 
                            e.getMessage());
                    }
                    return false;
                }                        
            }
        }
        return true;
    }
    
    private boolean validatePasswordField(Control ctrl) {
        PasswordControlAccessor acc = null;
        if (ctrl.getData(ACCESSOR) instanceof PasswordControlAccessor) {
            acc = (PasswordControlAccessor) ctrl.getData(ACCESSOR);
        }
        
        if (acc == null) {
            return false;
        }
        
        final Text pwt1 = acc.getPrimaryText();
        final Text pwt2 = acc.getSecondaryText();
        
        if (pwt1.getText().equals(pwt2.getText())) {
            return true;
        }

        if (interactive) {
            MessageDialog.openError(editor.getSite().getShell(),
                "Passwords not the same",
                "The two password values are not identical, please re-enter the passwords.");
        }                    
        pwt2.setText("");
        pwt1.setFocus();
        pwt1.setText("");
        editor.getSite().getShell().getDisplay().asyncExec(new Runnable() {
            public void run() {
                pwt1.setFocus();
            }                        
        });
        return false;
    }
    
    private void addContextListener(Control control, String xpath) {
        String attribName = element.getDependencyName(xpath);
        for (Control masterControl : values.keySet()) {
            Object object = values.get(masterControl);
            if (object instanceof SchemaAttribute) {
                if (((SchemaAttribute) object).getName().equals(attribName)) {
                    final Control dependentControl = control;
                    masterControl.addListener(SWT.Modify, new Listener() {
                        public void handleEvent(Event event) {
                            ControlAccessor accessor = (ControlAccessor) event.widget.getData(ACCESSOR);
                            String filter = accessor.getValue();
                            if (accessor instanceof QNameControlAccessor) {
                                filter = ((QNameControlAccessor) accessor).convertToFullName(filter);
                            }
                            updateValues(dependentControl, filter);
                        }            
                    });
                }
            }
        }        
    }
    
    @SuppressWarnings("unchecked")
    private void updateQNameNamespaceDecl(Control control) {
        if (!(control instanceof Combo) ||
            !(control.getData(ACCESSOR) instanceof QNameControlAccessor)) {
            return;
        }
        Combo combo = (Combo) control;
        
        // remove the namespace decl associated with prev item
        String prevItem = (String) control.getData(PREV_VALUE);
        if (prevItem != null && prevItem.length() != 0) {
            boolean inUse = false;
            String prefix = QNameHelper.getPrefixFromQName(prevItem);
            // go thru all attributes looking for values starting with this prefix
            for (Attribute attribute : (List<Attribute>) element.getJDOMElement().getAttributes()) { 
                if (QNameHelper.getPrefixFromQName(attribute.getValue()).equals(prefix)) {
                    inUse = true;
                }
            }
            
            if (!inUse) {
                element.removeNamespaceDeclaration(
                        QNameHelper.getPrefixFromQName(prevItem),
                        QNameHelper.getNamespaceFromQName(((Map) control.getData(NS_MAPPINGS)), prevItem));
            }
        }
        
        // add the namespace decl associated with current item
        String selItem = combo.getText();
        element.addNamespaceDeclaration(
                    QNameHelper.getPrefixFromQName(selItem),
                    QNameHelper.getNamespaceFromQName(((Map) control.getData(NS_MAPPINGS)), selItem));
        
    }
    
    private class HelpViewSynchingFocusListener extends FocusAdapter implements FocusListener {
        private final AnnotatedElement item;
        
        private HelpViewSynchingFocusListener(AnnotatedElement ae) {
            item = ae;
        }
        
        @Override
        public void focusGained(FocusEvent e) {
            IWorkbenchPage page = editor.getSite().getWorkbenchWindow().getActivePage();
            IViewPart v = page.findView(XefConstants.POLICY_HELP_VIEW);
            if (v instanceof XefHelpView) {
                XefHelpView view = (XefHelpView) v;
                
                if (item.getDocumentation() != null && 
                    !item.getDocumentation().equals("")) {
                    view.setHTMLText(
                        "<font size=\"+1\"><b>" +
                        item.getDisplayName() + "</b></font> <font size=\"-1\">(" + 
                        XefEditMasterDetailsBlock.getLabel(element) + 
                        ")</font><p/>", 
                        item.getDocumentation());
                } else {
                    editor.policyEditPage.block.setHelpText(element);
                }
            }
        }        
    }    
}
