/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.stp.core.sca;

import org.eclipse.stp.core.internal.sca.SCAInternalObject;

/**
 * A convenience class for identifying attributes and characteristics of
 * elements in the SCA Model hierarchy. Often these flags are a quick indication
 * of what an object can be casted down to.
 * 
 * <p>
 * This class should be used in conjunction with
 * {@link org.eclipse.stp.core.sca.SCAObject#getFlags()}.
 * </p>
 * 
 * @since 1.0
 * 
 */
public final class Flags {

   /**
    * 
    * @param flags
    *           The flags from a given {@link SCAInternalObject}
    * @return True if the flags indicate a potential {@link WireSource}.
    * 
    * @see WireSource
    */
   public static boolean isPossibleWireSource(int flags) {
      return (flags & IConstants.F_POSSIBLE_WIRE_SOURCE) != 0;
   }

   /**
    * Returns whether the given integer includes an indication of being a
    * possible WireTarget.
    * 
    * @param flags
    *           The flags from a given {@link SCAInternalObject}
    * @return boolean
    * 
    * @see WireSource
    */
   public static boolean isPossibleWireTarget(int flags) {
      return (flags & IConstants.F_POSSIBLE_WIRE_TARGET) != 0;
   }

   /**
    * 
    * @param flags
    *           The flags from a given {@link SCAInternalObject}
    * @return True if the given flags indicate a boundary object. A 'boundary'
    *         object is either an {@link EntryPoint} or an
    *         {@link ExternalService} .
    * 
    * @see org.eclipse.stp.core.sca.ExternalService
    * @see org.eclipse.stp.core.sca.EntryPoint
    */
   public static boolean isBoundaryElement(int flags) {
      return (flags & IConstants.F_BOUNDARY_ELEMENT) != 0;
   }

   /**
    * A {@link Service} that is 'bindable' may be cast down to a
    * {@link ModuleService} and similarly, a 'bindable' {@link Reference} may be
    * cast down to a {@link ModuleReference}.
    * 
    * @param flags
    *           The flags from a given {@link SCAInternalObject}
    * @return Ttrue if the element is bindable. A 'bindable' element is either a
    *         {@link ModuleService} or a {@link ModuleReference}, false
    *         otherwise.
    * 
    * @see ModuleService
    * @see ModuleReference
    */
   public static boolean isBindableElement(int flags) {
      return (flags & IConstants.F_BINDABLE_ELEMENT) != 0;
   }
}
