/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.stp.core.sca.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;

public class ModelFileIndexer implements IResourceChangeListener {

   private static final Object     lock                  = new Object();

   public static final String      COMPOSITE_EXT         = "composite";    //$NON-NLS-1$

   public static final String      COMPONENT_TYPE_EXT    = "componentType"; //$NON-NLS-1$

   public static final String      SUBSYSTEM_EXT         = "subsystem";    //$NON-NLS-1$

   public static final String      MODULE_EXT            = "module";       //$NON-NLS-1$

   private static final Set        interestingExtensions = new HashSet();

   static {
      interestingExtensions.add(COMPOSITE_EXT);
      interestingExtensions.add(COMPONENT_TYPE_EXT);
      interestingExtensions.add(SUBSYSTEM_EXT);
      interestingExtensions.add(MODULE_EXT);
   }

   private static ModelFileIndexer instance;

   public static ModelFileIndexer getInstance() {
      if (instance != null)
         return instance;
      synchronized (lock) {
         if (instance == null)
            instance = new ModelFileIndexer();
      }
      return instance;
   }

   private ModelFileIndexer() {
      ResourcesPlugin.getWorkspace().addResourceChangeListener(this,
            IResourceChangeEvent.POST_CHANGE);
   }

   private final Map indices = new HashMap();

   public void resourceChanged(IResourceChangeEvent event) {

      if (IResourceChangeEvent.POST_CHANGE == event.getType()) {
         IResourceDelta[] addedDeltas = event.getDelta().getAffectedChildren(
               IResourceDelta.ADDED);
         IResource source = null;
         for (int i = 0; i < addedDeltas.length; i++) {
            source = addedDeltas[i].getResource();
            if (IResource.FILE == source.getType()
                  && interestingExtensions.contains(source.getFileExtension())) {
               getModelIndex(source.getProject()).addModelFile((IFile) source);
            }
         }
      }
   }

   public ModelFileIndex getModelIndex(IProject project) {
      ModelFileIndex index = (ModelFileIndex) indices.get(project);
      if (index != null)
         return index;
      synchronized (this) {
         index = (ModelFileIndex) indices.get(project);
         if (index == null) {
            indices.put(project, (index = new ModelFileIndex(project)));
         }
      }
      return index;
   }

}
