/*******************************************************************************
* Copyright (c) 2006 IONA Technologies PLC
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
* 
* Contributors:
*     IONA Technologies PLC - initial API and implementation
*******************************************************************************/
package org.eclipse.stp.sc.cxf.generators;

import java.io.File;
import java.util.ArrayList;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.stp.common.logging.LoggingProxy;
import org.eclipse.stp.sc.common.utils.JDTUtils;
import org.eclipse.stp.sc.common.utils.LaunchUtil;
import org.eclipse.stp.sc.cxf.launchers.JavaLaunchConfig;
import org.eclipse.stp.sc.jaxws.ScJaxWsPlugin;
import org.eclipse.stp.sc.jaxws.preferences.SCPreferenceConstants;
import org.eclipse.stp.sc.jaxws.properties.ScJaxWsPropertyConstants;
import org.eclipse.stp.sc.jaxws.wizards.wsdltojava.WsdlToJavaGenerateAction;
import org.eclipse.stp.sc.jaxws.workspace.JaxWsWorkspaceManager;
import org.eclipse.stp.sc.jaxws.workspace.ScNature;

public class CxfJava2wsGenerator {
	
	private static final String TEMP_SRC_FOLDER_NAME = "tempFolder";
	
	private static final LoggingProxy LOG = LoggingProxy.getlogger(CxfJava2wsGenerator.class);
	
	private static  CxfJava2wsGenerator instance = null;
	
	private CxfJava2wsGenerator(){
	}
	
	public static synchronized CxfJava2wsGenerator getInstance(){
		if(instance == null){
			instance = new CxfJava2wsGenerator();
		}
		return instance;
	}
	
	public void run(IFile file, IProject project) throws CoreException {
        try {
        	
        	IPreferenceStore preferenceStore = ScJaxWsPlugin.getDefault()
				.getPreferenceStore();
        	boolean mergeFlg = preferenceStore
				.getBoolean(SCPreferenceConstants.KEY_MERGE);
        	boolean isGenClient = false;
        	boolean isGenServer = false;
        	
        	String tempFolderName = TEMP_SRC_FOLDER_NAME;

    		//make sure the temp folder does not exist
    		while(project.getFolder(tempFolderName).exists()){
    			tempFolderName = tempFolderName + "1"; 
    		}
    		
    		//all the generated source code will be put into the temp folder firstly
    		IFolder tempFolder = project.getFolder(tempFolderName);
    		tempFolder.create(true, true, null);
        	
        	IJavaProject javaProject = JDTUtils.findJavaProject(project.getName());

        	//program arguments
        	ArrayList<String> args = new ArrayList<String>();

        	//data binding
        	args.add(ToolConstants.JAVA2WS_PARA_DATA_BINDING);
            args.add(project.getPersistentProperty(ToolConstants.PROPERTY_KEY_JAVA2WS_PARA_DATA_BINDING));
            
            //front end
            args.add(ToolConstants.JAVA2WS_PARA_FRONT_END);
            if(project.hasNature(ScNature.NATURE_ID)){
            	args.add(ToolConstants.PROPERTY_VALUE_JAXWS_FRONTEND);
			}else{
	            args.add(project.getPersistentProperty(ToolConstants.PROPERTY_KEY_JAVA2WS_PARA_FRONT_END));
			}
            
            //wsdl
            String genWsdl = project.getPersistentProperty(ToolConstants.PROPERTY_KEY_JAVA2WS_PARA_WSDL);
            if(ToolConstants.PROPERTY_VALUE_TRUE.equalsIgnoreCase(genWsdl)){
            	IContainer wsdlLocation = JaxWsWorkspaceManager.getWSDLGenFolder(null, project);
                String wsdlUrl = wsdlLocation.getLocation().toOSString()
                                    + File.separator 
                                    + project.getName()
                                    + ".wsdl";
            	args.add(ToolConstants.JAVA2WS_PARA_WSDL);
            	args.add(ToolConstants.JAVA2WS_PARA_OUTPUT_FILE);
            	args.add(wsdlUrl);
            }
            
            //output dir
            args.add(ToolConstants.JAVA2WS_PARA_OUTPUT_DIR);
            args.add(tempFolder.getLocation().toOSString());
            
            //wrapper bean
            //no need to generate wrapper bean TODO
            //args.add(ToolConstants.JAVA2WS_PARA_WRAPPER_BEAN);
            
            //source dir
            args.add(ToolConstants.JAVA2WS_PARA_SOURCE_DIR);
            args.add(tempFolder.getLocation().toOSString());      
            
            //class dir
            args.add(ToolConstants.JAVA2WS_PARA_CLASS_DIR);
            args.add(JaxWsWorkspaceManager.getClassesFolder(project).getLocation().toOSString());      
            
            //client
            String genClient = project.getPersistentProperty(ToolConstants.PROPERTY_KEY_JAVA2WS_PARA_CLIENT);
            if(ToolConstants.PROPERTY_VALUE_TRUE.equalsIgnoreCase(genClient)){
            	isGenClient = true;
            	args.add(ToolConstants.JAVA2WS_PARA_CLIENT);
            }
            
            //Server
            String genServer = project.getPersistentProperty(ToolConstants.PROPERTY_KEY_JAVA2WS_PARA_SERVER);
            if(ToolConstants.PROPERTY_VALUE_TRUE.equalsIgnoreCase(genServer)){
            	args.add(ToolConstants.JAVA2WS_PARA_SERVER);
            	isGenServer = true;
            }
            
            //class path (from which the starting point class can be found) --- not jvm class path
            args.add(ToolConstants.JAVA2WS_PARA_CLASSPATH);
            args.add(JaxWsWorkspaceManager.getClassesFolder(project).getLocation().toOSString());

            
            //SOAP
            String soap12 =  project.getPersistentProperty(ToolConstants.PROPERTY_KEY_JAVA2WS_PARA_SOAP12);
            if(ToolConstants.PROPERTY_VALUE_SOAP12.equalsIgnoreCase(soap12)){
            	args.add(ToolConstants.JAVA2WS_PARA_SOAP12);
            }
            
            //XSD imports
            String xsdImports =  project.getPersistentProperty(ToolConstants.PROPERTY_KEY_JAVA2WS_PARA_XSD_IMPORT);
            if(ToolConstants.PROPERTY_VALUE_TRUE.equalsIgnoreCase(xsdImports)){
            	args.add(ToolConstants.JAVA2WS_PARA_XSD_IMPORT);
            } 
            
            //verbose
            String verbose =  project.getPersistentProperty(ToolConstants.PROPERTY_KEY_JAVA2WS_PARA_VERBOSE);
            if(ToolConstants.PROPERTY_VALUE_TRUE.equalsIgnoreCase(verbose)){
            	args.add(ToolConstants.JAVA2WS_PARA_VERBOSE);
            }   
            
            //input class
            args.add(JDTUtils.getJavaClassNameFromFile(file));
            
            //launch the generator
            LaunchUtil.launchJavaProgram(javaProject, ToolConstants.JAVA2WS_GENERATOR_CLASS, null, args.toArray(new String[0]), null);
            
            //generate launch configuration if needed
            if(isGenServer || isGenClient){
            	generateLaunchConfiguration(isGenServer, isGenClient, tempFolder, project);
            }

            if(mergeFlg){
            	new WsdlToJavaGenerateAction().mergeDir(JaxWsWorkspaceManager.getSrcFolder(project).getLocation().toOSString(), 
            			tempFolder.getLocation().toOSString(), 
            			true);
            }else{
            	new WsdlToJavaGenerateAction().mergeDir(JaxWsWorkspaceManager.getSrcFolder(project).getLocation().toOSString(), 
            			tempFolder.getLocation().toOSString(), 
            			false);
            }
            
            //remove the temporary folder 
            if(tempFolder.exists()){
            	tempFolder.delete(true, null);
            }
            
            project.refreshLocal(IProject.DEPTH_INFINITE, null);
        } catch (CoreException e){
        	throw e;
        } catch (Exception e) {
            LOG.error("java to web service generation error with the start point: " + file, e);
        }
	}
	
	public void run(IProject project) throws CoreException{
		String fileName = project.getPersistentProperty(ScJaxWsPropertyConstants.PROPERTY_KEY_STARTING_POINT);
		IFile file = project.getFile(new Path(fileName));
		if(file != null && file.exists()){
			run(file, project);
		}
	}
	
    /**
     * This method creates a server launch configuration to launch the produced client
     * code, if any, from within the eclipse environment  
     */
    private void createServerLaunchConfiguration(String configBaseName, String serverFullClassName,String progArgs, IProject project) {
    	String projectName = project.getName();
        //create the java launch configuration
        JavaLaunchConfig launchConfig = new JavaLaunchConfig();
        launchConfig.setProgramArgs(progArgs);
        launchConfig.setConfigBaseName(configBaseName);
        launchConfig.setProjectName(projectName);
        launchConfig.setServerMainClassName(serverFullClassName);
        launchConfig.createServerLaunchConfig();
    }    
    
    /**
	 * This method creates a client launch configuration to launch the produced
	 * client code, if any, from within the eclipse environment
	 */
    private void createClientLaunchConfiguration(String configBaseName, String clientFullClassName, String progArgs, IProject project) {
    	String projectName = project.getName();
        //create the java launch configuration
        JavaLaunchConfig launchConfig = new JavaLaunchConfig();
        launchConfig.setProgramArgs(progArgs);
        launchConfig.setConfigBaseName(configBaseName);
        launchConfig.setProjectName(projectName);
        launchConfig.setClientMainClassName(clientFullClassName);
        launchConfig.createClientLaunchConfig();
        
    }
    
    private void generateLaunchConfiguration(boolean isGenServer, boolean isGenClient, IFolder tempFolder, IProject project){
    	try{
            String clientName = null;
            String serverName = null;
            String packageName = "";
            File tempFile = new File(tempFolder.getLocation().toOSString());
            File[] files = tempFile.listFiles();
            while(files != null && files.length > 0){
            	tempFile = files[0];
            	if(tempFile.isFile()){
            		break;
            	}
            	if(packageName.length() > 0 ){
            		packageName = packageName + "." + tempFile.getName();
            	}else{
            		packageName = tempFile.getName();
            	}
            	files = files[0].listFiles();
            }
            
            for(int i = 0; i < files.length; i++){
            	if(files[i].getName().endsWith("Client.java")){
            		clientName = files[i].getName();
            	}
            	if(files[i].getName().endsWith("Server.java")){
            		serverName = files[i].getName();
            	}
            	if(clientName != null && serverName != null){
            		break;
            	}
            }
            
            
            if(packageName.length() > 0){
            	if(clientName != null)
            		clientName = packageName + "." + clientName.substring(0, clientName.length() - 5);
            	if(serverName != null)
            		serverName = packageName + "." + serverName.substring(0, serverName.length() - 5);
            }
            
            //generate launch configuration
            if(isGenServer){
            	createServerLaunchConfiguration(project.getName(), serverName, null, project);
            }
            if(isGenClient){
            	createClientLaunchConfiguration(project.getName(), clientName, null, project);
            }
    	}catch(Exception ex){
    		LOG.error("error while create launch configuration", ex);
    	}
    }
}
